package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridLayout;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.function.Supplier;
import javax.swing.BorderFactory;
import javax.swing.JComboBox;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import org.lsst.ccs.command.DictionaryArgument;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Graphics component that displays information on a command and accepts arguments input.
 *
 * @author emarin
 */
public final class ArgInputPanel extends JPanel implements ListSelectionListener {
    
// -- Fields : -----------------------------------------------------------------

    private DictionaryCommand command;
    private JLabel[] argName;
    private Supplier<String>[] argGetters;
    private final JPanel formPane;
    private final JEditorPane commandDesc;
    
    private final Map<String,String[]> history = new LinkedHashMap<String,String[]>(4, 0.8f, true) {
        @Override
        protected boolean removeEldestEntry(Map.Entry<String,String[]> eldest) {
            return size() > 50;
        }
    };

// -- Life cycle : -------------------------------------------------------------
    
    public ArgInputPanel() {
        setLayout(new BorderLayout());
        
        // Command description panel:
        
        commandDesc = new JEditorPane();
        add(commandDesc, BorderLayout.CENTER);
        commandDesc.setBackground(new Color(0xed, 0xec, 0xeb)); //TODO : color must match system colors
        commandDesc.setEditable(false);
        commandDesc.setContentType("text/html");
        commandDesc.setText("      ");
        
        // Argument input panel:
        
        formPane = new JPanel();
        formPane.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEtchedBorder(), BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));
    }
    
    
// -- Updates : ----------------------------------------------------------------
    
    @Override
    public void valueChanged(ListSelectionEvent lse) {
        command = ((JList<DictionaryCommand>)lse.getSource()).getSelectedValue();
        clear();
        if (command != null) {
            fill();
        }
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns a pretty string for displaying the values to the user.
     * @return Human-readable representation of argument values.
     */
    public String getFormattedArgumentValues() {
        StringBuilder res = new StringBuilder();
        for (Supplier<String> argGetter : argGetters) {
            res.append(argGetter.get()).append(" ");
        }
        return res.toString();
    }
    
    public String[] getFormattedValues() {
        String[] res = new String[argGetters.length];
        for (int i = 0; i < argGetters.length; i++) {
            res[i] = argGetters[i].get();
        }
        if (command != null && res.length != 0) {
            history.put(command.getCommandName(), (String[])res.clone());
        }
        return res;
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void clear() {
        formPane.removeAll();
        commandDesc.setText("       ");
    }
    
    private void fill() {
        clear();
        
        // Argument input panel:
        
        final int nArgs = command.getArguments().length;
        formPane.setLayout(new GridLayout(nArgs, 2));
        argName = new JLabel[nArgs];
        argGetters = new Supplier[nArgs];
        String[] hist = history.get(command.getCommandName());
        if (hist != null && hist.length != nArgs) {
            hist = null;
        }
        for (int i = 0; i < nArgs; i++) {
            DictionaryArgument da = command.getArguments()[i];
            argName[i] = new JLabel(da.getName());
            formPane.add(argName[i]);
            if(da.getAllowedValues().isEmpty()) {
                HintTextField htf = new HintTextField(command.getArguments()[i].getSimpleType());
                argGetters[i] = () -> htf.getText();
                formPane.add(htf);
                if (hist != null && !hist[i].isEmpty()) htf.setText(hist[i]);
            } else {
                JComboBox<String> allowedValuesBox = new JComboBox<>(da.getAllowedValues().toArray(new String[0]));
                if (hist != null) allowedValuesBox.setSelectedItem(hist[i]);
                argGetters[i] = () -> allowedValuesBox.getSelectedItem().toString();
                formPane.add(allowedValuesBox);
            }
        }
        
        // Command description panel:
        
        StringBuilder sb = new StringBuilder("<html>");
        sb.append(command.getType()).append(". Level: ").append(command.getLevel());
        sb.append("<h3>").append(command.getDescription()).append("</h3>");
        for (DictionaryArgument arg : command.getArguments()) {
            sb.append(" <b>").append(arg.getSimpleType()).append("</b> ").append(arg.getName()).append(" : ").append(arg.getDescription()).append("<br>");
        }
        commandDesc.setText(sb.toString());
        
        // Refresh display:
        
        setEnabled(isEnabled());
        repaint();
        validate();
    }

    @Override
    public void setEnabled(boolean enabled) {
        super.setEnabled(enabled);
        if (enabled) {
            add(formPane, BorderLayout.SOUTH);
        } else {
            this.remove(formPane);
        }
    }
    
    
    
    
    
    
}
