package org.lsst.ccs.gconsole.services.aggregator;

import java.util.List;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;

/**
 * Encapsulates static properties and current status of a data channel.
 * Each channel has a central value (an {@code Object}) and optional attributes identified by string keys.
 * A channel is identified by a unique path (slash-separated string, with the subsystem name as the first element). 
 * <p>
 * Implementations should ensure that the path is used to establish channel equality and hash code.
 *
 * @author onoprien
 */
public interface AgentChannel {
    
    /**
     * Enumeration of standard attributes.
     */
    public interface Key {
        
        final String VALUE = "value";
        
        final String DESCR = Attribute.DESCRIPTION.getName();
        final String UNITS = Attribute.UNITS.getName();
        final String TYPE = Attribute.TYPE.getName();
        final String FORMAT = Attribute.FORMAT.getName();
        final String PAGE = Attribute.PAGE.getName();
        final String SECTION = Attribute.SECTION.getName();
        final String ALERT_HIGH = Attribute.ALARMHI.getName();
        final String ALERT_LOW = Attribute.ALARMLO.getName();
        final String STATE = Attribute.STATE.getName();
        final String DATA_GROUP = Attribute.DATA_GROUP.getName();
        final String DATA_TYPE = Attribute.DATA_TYPE.getName();
        final String RTD_TYPE = Attribute.RTD_TYPE.getName();
        
        final String LOW_ALARM = "limitLo";
        final String LOW_WARN = "dbandLo";
        final String HIGH_ALARM = "limitHi";
        final String HIGH_WARN = "dbandHi";
        
        final String TRENDING = "_key_";
        
    }
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Return the unique path of this channel.
     * @return Channel path in (agent name)/(local path) form.
     */
    public String getPath();
    
    /**
     * Returns the name of the {@code Agent} this channel belongs to.
     * @return Agent name.
     */
    public String getAgentName();
    
    /**
     * Returns the descriptor of the {@code Agent} this channel belongs to. 
     * @return Agent descriptor.
     */
    public AgentInfo getAgent();
    
    /**
     * Returns the path of this channel with the agent name and the following slash stripped.
     * Local paths are unique inside an agent.
     * @return Local path.
     */
    public String getLocalPath();
    
    /**
     * Returns the value of the channel attribute identified by the {@code key}.
     * 
     * @param <T> Type of the attribute value.
     * @param key The attribute is identified by the string returned by {@code toString()} method of the key object.
     *            If {@code key} is {@code null} or evaluates to {@link Key#VALUE}, the central value is returned.
     * @return Attribute value, or {@code null} if this channel does not have the specified attribute.
     */
    public <T> T get(Object key);
    
    /**
     * Returns the current central value of this channel.
     * @param <T> Type of the central value.
     * @return Current value.
     */
    public <T> T get();
    
    /**
     * Returns the list of keys of attributes this channel has.
     * @return List of existing attribute keys.
     */
    public List<String> getAttributes();
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the value of the specified attribute.
     * The attribute value does not change if the new value is equal to the old one.
     * 
     * @param key The attribute is identified by the string returned by {@code toString()} method of the key object.
     *            If {@code key} is {@code null} or evaluates to {@link AgentChannel.Key#VALUE}, the central value is set.
     * @param value The new value.
     * @return {@code true} if the attribute value has changed as a result of this call.
     */
    public boolean set(Object key, Object value);
    
    /**
     * Sets the value of this channel.
     * The value will not change if the new value is equal to the old one.
     * 
     * @param value New value.
     * @return {@code true} if the value has changed as a result of this call.
     */
    public boolean set(Object value);
}
