package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.SwingUtilities;
import org.freehep.jas.plugin.console.ConsoleService;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.tracer.filters.StandardMessageFilter;
import org.lsst.ccs.messaging.AgentPresenceListener;

/**
 * Graphical console plugin that creates and manages Tracers.
 */
@Plugin(name="LSST Tracer Plugin",
        id="tracer",
        description="LSST CCS alert notification service.")
public class LsstTracerPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------

    private final String defaultTracerName = "Messages";
    
    private final FilterRegistry filterRegistry;
    
    private final Action newFilterAction;
    private final Action defaultFilterAction;
    private final Action loadFilterAction;
    
    private final AgentPresenceListener agentConnectionListener;


// -- Life cycle : -------------------------------------------------------------
    
    public LsstTracerPlugin() {
        
        filterRegistry = new FilterRegistry(this);
        
        defaultFilterAction = new AbstractAction("Default") {
            @Override
            public void actionPerformed(ActionEvent e) {
                createTracer("Messages", new StandardMessageFilter.Default());
            }
        };
        newFilterAction = new AbstractAction("New filter…") {
            @Override
            public void actionPerformed(ActionEvent e) {
                FilterEditor.showDialog(null, null, LsstTracerPlugin.this, createTracer(defaultTracerName));
            }
        };
        loadFilterAction = new AbstractAction("Load filter...") {
            @Override
            public void actionPerformed(ActionEvent e) {

                
//                String filterName = FilterChooser.showDialog(getConsole().getWindow(), filterRegistry);
//                if (filterName != null) {
//                    int i = filterName.lastIndexOf("/");
//                    
//                    createTracer(filterName.substring(i+1), filterRegistry.getFilter(filterName));
//                }
                
                MessageFilter filter = filterRegistry.selectFilter(null);
                if (filter != null) createTracer(filter.getName(), filter);
                
            }
        };
        
        agentConnectionListener = new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    AgentInfo.AgentType type = agent.getType();
                    if (!(AgentInfo.AgentType.CONSOLE.equals(type) || AgentInfo.AgentType.LISTENER.equals(type))) {
                        SwingUtilities.invokeLater(() -> {
                            Action action = new AbstractAction(agent.getName()) {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    createTracer(agent.getName(), new StandardMessageFilter.Default(agent.getName()));
                                }

                            };
                            getServices().addMenu(action, "400: CCS Tools :-1:2", "Message Viewer:100:100", "Subsystems:1");
                            action = new AbstractAction("Message Viewer") {
                                @Override
                                public void actionPerformed(ActionEvent e) {
                                    createTracer(agent.getName(), new StandardMessageFilter.Default(agent.getName()));
                                }

                            };
                            getServices().addMenu(action, "CCS Subsystems", agent.getName() + ":-10:2");
                        });
                    }
                }
            }
            @Override
            public void disconnected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        Console.getConsole().removeMenu(" CCS Tools ", "Message Viewer", "Subsystems", agent.getName());
                        Console.getConsole().removeMenu("CCS Subsystems", agent.getName(), "Message Viewer");
                    }
                });
            }
        };
    }
    
    @Override
    public void initialize() {
        getServices().addMenu(defaultFilterAction, "400: CCS Tools :-1:2", "Message Viewer:1");
        getServices().addMenu(newFilterAction, "400: CCS Tools :-1:2", "Message Viewer:2");
        getServices().addMenu(loadFilterAction, "400: CCS Tools :-1:2", "Message Viewer:3");
    }

    @Override
    public void start() {
//        JConsoleService cs = (JConsoleService) getConsole().getConsoleLookup().lookup(JConsoleService.class);
//        if (cs == null) throw new IllegalStateException("JConsole service is not available");
//        cs.addConsoleProvider(JLineConsole.class, new JLineConsoleProvider());
        filterRegistry.init();
        getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(agentConnectionListener);
    }

    @Override
    public void stop() {
        getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(agentConnectionListener);
    }


// -- Getters : ----------------------------------------------------------------
    
    public FilterRegistry getFilterRegistry() {
        return filterRegistry;
    }
    
    
// -- Handling tracers : -------------------------------------------------------

    public Tracer createTracer(String name) {
        return createTracer(name, (MessageFilter)null);
    }

    public Tracer createTracer(String name, String filterName) {
        return createTracer(name, filterRegistry.getFilter(filterName));
    }

    public Tracer createTracer(String name, MessageFilter filter) {
        ConsoleService cs = (ConsoleService) getConsole().getConsoleLookup().lookup(ConsoleService.class);
        Tracer tracer = null;
        if (cs != null) {
            tracer = new Tracer(this, filter);
            org.freehep.jas.plugin.console.Console console = cs.createConsole(name, null, tracer);
//            SimpleAttributeSet style = new SimpleAttributeSet();
//            ConsoleOutputStream out = console.getOutputStream(style);
//            PrintWriter pw = new PrintWriter(out, true);
            TracerWriter pw = new TracerWriter(console);
            cs.getPageContextForConsole(console).addPageListener(tracer);
            tracer.start(pw);
        }
        return tracer;
    }

//    /** Creates JLineConsole based tracer. Not used at the moment. */
//    private JLineConsole createJTracerConsole(String name) {
//        FreeHEPLookup lookup = getApplication().getConsoleLookup();
//        JConsoleService cs = (JConsoleService) lookup.lookup(JConsoleService.class);
//        if (cs != null) {
//            Tracer tracer = new Tracer(getMessagingAccess());
//            JLineConsole console = (JLineConsole) cs.createConsole(JLineConsole.class, name, null, tracer);
//            PrintWriter pw = new PrintWriter(console.getConsoleReader().getOutput(), true);
//            cs.getPageContextForConsole(console).addPageListener(tracer);
//            tracer.start(pw); // This returns immediately
//            return console;
//        }
//        return null;
//    }
    
}
