package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.math.BigInteger;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Formatter for monitoring data.
 * <p>
 * This class is an entry point for all client code that needs to create formatted values to be
 * displayed by monitoring views (some views may choose to call methods of {@link MonitorField}
 * directly for efficiency or other reasons, but not all {@link MonitorField} constants support such use).
 * This class may be used as is or subclassed to modify formatting logic.
 * <p>
 * As implemented by this class, all convenience methods that take {@link MonitorCell} or lists
 * of data channels (as {@link AgentChannel} or {@link DisplayChannel}) eventually forward the
 * call to {@link #format(MonitorField, List) format(MonitorField, List[AgentChannel])}.
 * That method uses the following logic:
 * <ul>
 * <li>call {@code MonitorField.format(channels)};
 * <li>if the field returns {@code null}, indicating that it does not support formatting multiple
 *     channels, call {@code MonitorField.format(channel)} for each channel and merge the results
 *     using {@link #merge(MonitorField, List) merge(monitorField, formattedValues)};
 * <li>if the resulting formatted value is incomplete, finish formatting by calling
 *     {@link #format(FormattedValue) format(FormattedValue)}.
 * <li>If the formatted value is still incomplete, return {@code FormattedValue.NA},
 *     indication that the data is currently unavailable
 * </ul>
 * Documentation for these methods describes their logic as implemented by this class.
 * Asking the {@code MonitorField} to format a list of channels can be bypassed by calling
 * {@link #format(MonitorField field, AgentChannel channel)}
 *
 * @author onoprien
 */
public class MonitorFormat {
    
// -- Static constants : -------------------------------------------------------
    
    static public final DefaultMonitorFormat DEFAULT = new DefaultMonitorFormat();
    
    static public final Color COLOR_FG = Color.BLACK; // default foreground
    static public final Color COLOR_BG = Color.WHITE; // default background
    
    static public final Color COLOR_CHANGED = Color.BLUE;
    
    static public final Color COLOR_GOOD = new Color(160, 255, 160);
    static public final Color COLOR_WARN = new Color(255, 255, 100);
    static public final Color COLOR_ERR = new Color(255, 160, 160);
    static public final Color COLOR_DISABLED = new Color(255, 200, 255);
    static public final Color COLOR_OFF = new Color(160, 200, 255);
    static public final Color COLOR_POPUP = new Color(255, 255, 160);
    static public final Color COLOR_MULTI = Color.LIGHT_GRAY;
    static public final Color COLOR_NA = COLOR_BG;
    
    static public final EnumMap<DataProviderState,Color> COLOR_STATE = new EnumMap(DataProviderState.class);
    static {
        COLOR_STATE.put(DataProviderState.OFF_LINE, COLOR_OFF);
        COLOR_STATE.put(DataProviderState.DISABLED, COLOR_DISABLED);
        COLOR_STATE.put(DataProviderState.NOT_VALIDATED, COLOR_BG);
        COLOR_STATE.put(DataProviderState.NOMINAL, COLOR_GOOD);
        COLOR_STATE.put(DataProviderState.WARNING, COLOR_WARN);
        COLOR_STATE.put(DataProviderState.ALARM, COLOR_ERR);
    }
    
    static public final String TEXT_MULTI = "---";
    static public final int HA_MULTI = SwingConstants.CENTER;
    
// -- Fields : -----------------------------------------------------------------
    
    protected String DEFAULT_FLOAT_FORMAT = "%14.5f";
    protected String DEFAULT_INT_FORMAT = "%10d";
    protected String DEFAULT_DT_FORMAT = "%1$tF %1$tT";
    protected int LENGTH_LIMIT = 40;
    
    static private final Pattern pFormatConversion = Pattern.compile("(?:\\d+\\$)?[^ ]*?\\d*(?:\\.\\d*)?([a-zA-Z]{1,2})(?: .*)?");
    
    
// -- Formatting : -------------------------------------------------------------
    
    /**
     * Produces formatted value, given the field and the list of contributing channels. <br>
     * The implementation provided by this class calls {@link #format(MonitorField field, AgentChannel channel)}
     * for each channel, then merges the results using {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param field Monitored field.
     * @param channels Contributing channels.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, List<AgentChannel> channels) {
        
        // Ask the field format itself based on the list of channels:
        
        FormattedValue fv = field.format(channels);
        
        // If the field cannot, ask it to format each channel separately, then merge:
        
        if (fv == null) {
            List<FormattedValue> fvList = channels.stream().map(ch -> format(field, ch)).collect(Collectors.toList());
            fv = merge(field, fvList);
        }
        
        // If the formatted value is incomplete, finish formatting:
        
        if (!fv.isValid()) {
            fv = format(fv);
        }
        
        // If the formatted value is still incomplete, return indication that the data is currently unavailable:
        
        if (!fv.isValid()) {
            fv = FormattedValue.NA;
        }
        
        return fv;
    }
    
    /**
     * Produces formatted value, given the monitored field and the data channel whose value needs to be formatted. <br>
     * The implementation provided by this class first tries to call {@link MonitorField#format(AgentChannel channel)}.
     * If the returned value is not valid, it is forwarded to {@link #format(FormattedValue)} to finish formatting.
     * 
     * @param field Monitored field.
     * @param channel Data channel.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, AgentChannel channel) {
        
        // Ask the field format itself:
        
        FormattedValue fv = field.format(channel);
        if (fv.isValid()) return fv;
        
        // If the formatted value is incomplete, finish formatting:
        
        fv = format(fv);
        if (fv.isValid()) return fv;
        
        // If the formatted value is still incomplete, return indication that the data is currently unavailable:
        
        return FormattedValue.NA;
    }

    /**
     * Completes formatting of a partially formatted value.
     * The instance passed to this method should have its {@code value} and optionally {@code format}
     * fields set to the unformatted value and the suggested format string, respectively.
     * 
     * @param fv Partially formatted value.
     * @return Formatted value.
     */
    protected FormattedValue format(FormattedValue fv) {
        
        if (fv.isValid()) return fv;
        
        // If configurable parameter, color foreground:
        
        if (fv.value instanceof ConfigurationParameterInfo) {
            ConfigurationParameterInfo conf = (ConfigurationParameterInfo) fv.value;
            fv.value = conf.getCurrentValue();
            if (fv.fgColor == null) fv.fgColor = conf.isDirty() ? COLOR_CHANGED : COLOR_FG;
            if (fv.editable == null) fv.editable = !conf.isFinal();
            if (fv.toolTip == null) fv.toolTip = conf.getDescription();
            return format(fv);
        }
        
        // Format raw value:
        
        if (fv.text == null) {
            
            // if format is known, try to use it
            
            if (fv.format != null) {
                
                char conv = getFormatConversion(fv.format).charAt(0);
                try {
                    switch (conv) {
                        case 'd': case 'o': case 'x': case 'X':  // integer
                            if (!(fv.value instanceof Integer || fv.value instanceof Long || fv.value instanceof Short || fv.value instanceof Byte || fv.value instanceof BigInteger)) {
                                fv.value = Long.parseLong(fv.value.toString());
                            }
                            break;
                        case 'f': case 'e': case 'E': case 'g': case 'G':
                            if (!(fv.value instanceof Double || fv.value instanceof Float || fv.value instanceof BigInteger)) {
                                fv.value = Double.parseDouble(fv.value.toString());
                            }
                            break;
                    }
                    fv.text = String.format(fv.format, fv.value);
                } catch (NumberFormatException | IllegalFormatException x) {
                }
            }
            
            // otherwise, if a number, try to use default format
            
            if (fv.text == null) {
                if (fv.value instanceof Double || fv.value instanceof Float) {
                    try {
                        fv.text = String.format(DEFAULT_FLOAT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Integer) {
                    try {
                        fv.text = String.format(DEFAULT_INT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                } else if (fv.value instanceof Instant) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, ZonedDateTime.ofInstant((Instant)fv.value, ZoneId.systemDefault()));
                    } catch (IllegalFormatException | DateTimeException x) {
                    }
                } else if (fv.value instanceof ZonedDateTime) {
                    try {
                        fv.text = String.format(DEFAULT_DT_FORMAT, fv.value);
                    } catch (IllegalFormatException x) {
                    }
                }
            }

            // otherwise, convert to String:
            
            if (fv.text == null) {
                fv.text = fv.value.toString();
            }
            
            // limit text length:
            
            if (fv.text.length() > LENGTH_LIMIT) {
                if (fv.toolTip == null) {
                    fv.toolTip = fv.text;
                } else {
                    fv.toolTip = "<html><b>"+ fv.text +"</b><br>"+ fv.toolTip;
                }
            }
            
            // pad with spaces:
            
            if (!fv.text.startsWith(" ")) fv.text = " "+ fv.text;
            if (!fv.text.endsWith(" ")) fv.text = fv.text +" ";

        }
        
        fv.value = null;        
        return fv;
    }
    
    
// -- Convenience methods: -----------------------------------------------------
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell. <br>
     * The actual formatting is done by {@link #format(MonitorField field, List channels)}.
     * 
     * @param cell Cell to be formatted.
     * @return True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell) {
        List<AgentChannel> channels = cell.getChannels().stream()
                .flatMap(h -> h.getChannels().stream())
                .collect(Collectors.toList());
        FormattedValue fv = format(cell.getField(), channels);
        if (Objects.equals(fv, cell.getFormattedValue())) {
            return false;
        } else {
            cell.setFormattedValue(fv);
            return true;
        }
    }
    
    /**
     * Convenience method that updates {@link FormattedValue} associated with the specified cell,
     * assuming that the only possible reason for a change are changes in the specified channel. <br>
     * The implementation provided by this class forwards the call to {@link #format(MonitorCell cell)}, 
     * ignoring the {@code channelHandle} argument. Subclasses might implement more efficient formatting
     * for cells that depend on multiple channels.
     * 
     * @param cell Cell to be formatted.
     * @param channelHandle Handle of the data channel that might have changed.
     * @return  True if the formatted value associated with the cell has changed as a result of this call.
     */
    public boolean format(MonitorCell cell, DisplayChannel channelHandle) {
        return format(cell);
    }
    
    /**
     * Convenience method that produces formatted value, given the field and the channel handle. <br>
     * The actual formatting is done by {@link #format(MonitorField field, AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channelHandle Handle of a data channel whose value needs to be formatted.
     * @return Formatted value.
     */
    public FormattedValue format(MonitorField field, DisplayChannel channelHandle) {
        return format(field, channelHandle.getChannels());
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Merges zero or more formatted values.
     * 
     * @param field Monitored field.
     * @param values Formatted values to merge.
     * @return Merged formatted value.
     */
    public FormattedValue merge(MonitorField field, List<FormattedValue> values) {
        switch (values.size()) {
            case 0:
                return new FormattedValue(field.getTitle(), COLOR_FG, COLOR_BG, SwingConstants.CENTER, null, false);
            case 1:
                return values.get(0);
            default:
                Iterator<FormattedValue> it = values.iterator();
                FormattedValue first = it.next();
                String text = first.getText();
                Color fgColor = first.getFgColor();
                Color bgColor = first.getBgColor();
                int horizontalAlignment = first.getHorizontalAlignment();
                String toolTip = first.getToolTip();
                while (it.hasNext()) {
                    FormattedValue other = it.next();
                    if (!Objects.equals(text, other.getText())) {
                        text = TEXT_MULTI;
                    }
                    fgColor = mergeConfigColor(fgColor, other.getFgColor());
                    bgColor = mergeStateColor(bgColor, other.getBgColor());
                    if (horizontalAlignment != other.getHorizontalAlignment()) {
                        horizontalAlignment = HA_MULTI;
                    }
                    if (!Objects.equals(toolTip, other.getToolTip())) {
                        toolTip = null;
                    }
                }
                return new FormattedValue(text, fgColor, bgColor, horizontalAlignment, toolTip, false);
        }
    }
    
    /**
     * Merges colors used to indicate a state of a configurable parameter. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeConfigColor(Color c1, Color c2) {
        return COLOR_FG.equals(c1) ? c2 : c1; 
    }
    
    /**
     * Merges colors used to indicate the channel state. <br>
     * Used internally by {@link #merge(MonitorField field, List formattedValues)}.
     * 
     * @param c1 First color.
     * @param c2 Second color.
     * @return Color of the merged value.
     */
    public Color mergeStateColor(Color c1, Color c2) {
        if (c1 == null) return c2;
        if (c2 == null) return c1;
        for (Color c : COLOR_STATE.values()) {
            if (c.equals(c1)) {
                return c2;
            } else if (c.equals(c2)) {
                return c1;
            }
        }
        return c1;
    }
    
    /**
     * Extracts unformatted value corresponding to the specified field from a channel. <br>
     * The implementation provided by this class forwards the call to {@code field.getValue(AgentChannel channel)}.
     * 
     * @param field Monitored field.
     * @param channel Data channel.
     * @return Unformatted value,or {@code null} if the value cannot be extracted.
     */
    public Object getValue(MonitorField field, AgentChannel channel) {
        return field.getValue(channel);
    }
    
    /**
     * Extracts format conversion symbol from a format string.
     * This horrible code is designed to make best guess for all the misspelled formats in existing groovy files. // FIXME
     * 
     * @param format Format string, ideally as specified in https://docs.oracle.com/javase/8/docs/api/java/util/Formatter.html#syntax .
     * @return Format conversion characters, or a string with a single space character if failed to guess.
     */
    static private String getFormatConversion(String format) {
        String[] ss = format.split("%");
        for (String s : ss) {
            Matcher m = pFormatConversion.matcher(s);
            if (m.matches()) {
                return m.group(1);
            }
        }
        return " ";
    }
    
    
// -- Specialized subclusses : -------------------------------------------------
    
    static public final class DefaultMonitorFormat extends MonitorFormat {
        public final String FLOAT_FORMAT = super.DEFAULT_FLOAT_FORMAT;
        public final String INT_FORMAT = super.DEFAULT_INT_FORMAT;
        public final String DT_FORMAT = super.DEFAULT_DT_FORMAT;
        public final int LENGTH_LIMIT = super.LENGTH_LIMIT;
    };
    
    
// -- Test : -------------------------------------------------------------------
    
//    public static void main(String... args) {
//        System.out.println("---------------"+ getFormatConversion(".3g") +"-------------");
//    }
    
    
}
