package org.lsst.ccs.gconsole.base.filter;

import java.util.Collections;
import java.util.List;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;

/**
 * Regular expression based channels filter.
 * The filter selects channels that match a regular expression and leaves the paths unchanged.
 *
 * @author onoprien
 */
public class RegExFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final Pattern pattern;
    private final boolean wholePath;
    private final String regex;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "Regular expression filter",
            path = "Built-In/Regular Expression",
            description = "Channels filter that uses a regular expression to select channels and leaves their paths unchanged.")
    public RegExFilter(
            @Par(def = Par.NULL, desc = "Filter name. Optional.") String name,
            @Par(desc = "Regular expression to be applied to the channel path.") String regex,
            @Par(def = "false", desc = "Match the whole path. If checked, a channel is accepted only if the provided regular expression matches its entire path.") boolean wholePath)
    {
        this.name = name;
        if (regex == null || regex.isEmpty()) throw new IllegalArgumentException("The regular expression cannot be empty");
        pattern = Pattern.compile(regex);
        this.wholePath = wholePath;
        if (!wholePath) {
            regex = ".*"+ regex +".*";
        }
        this.regex = regex;
    }
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getOriginChannels() {
        return Collections.singletonList(regex);
    }

    @Override
    public String getDisplayPath(String originPath) {
        boolean accept;
        if (wholePath) {
            accept = pattern.matcher(originPath).matches();
        } else {
            accept = pattern.matcher(originPath).find();
        }
        return accept ? originPath : null;
    }

    @Override
    public String getOriginPath(String displayPath) {
        return displayPath;
    }
    
}
