package org.lsst.ccs.gconsole.plugins.monitor;

import javax.swing.JComponent;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.base.Console;

/**
 * View of monitoring data.
 * {@code AgentStatusListener} methods are called on the {@code AgentStatusAggregator} thread.
 * All other methods declared by this interface should be called on AWT Event Dispatch Thread.
 *
 * @author onoprien
 */
public interface MonitorView extends AgentStatusListener {
    
    /**
     * Category for general purpose {@link Persistable Persistable} views.
     */
    static public String CATEGORY = "MonitorView";
    
// -- Getters and setters : ----------------------------------------------------
    
    /**
     * Returns the graphical component maintained by this view.
     * @return Graphical component to be displayed by the GUI.
     */
    JComponent getPanel();
    
    /**
     * Returns the name of this view.
     * @return  View name.
     */
    default String getName() {
        JComponent panel = getPanel();
        return panel == null ? "" : panel.getName();
    }
    
    /**
     * Sets the name of this view.
     * @param name View name.
     */
    default void setName(String name) {
        JComponent panel = getPanel();
        if (panel != null) panel.setName(name);
    }
    
    /**
     * Returns the filter associated with this view, or {@code null} if there is none.
     * @return Channels filter.
     */
    AgentChannelsFilter getFilter();
    
    /**
     * Sets the channels filter to be used by this view.
     * The filter determines what channels are displayed by this view, and how they are displayed.
     * @param filter Filter to be associated with this view.
     */
    void setFilter(AgentChannelsFilter filter);
    
    /**
     * Returns formatter associated with this view, if any.
     * @return Formatter associated with this view, or {@code null} if there is no formatter.
     */
    default MonitorFormat getFormater() {
        return null;
    }
    
    /**
     * Associates a formatter with this view.
     * The exact way the formatter is used - if at all - depends on specific implementation.
     * @param formatter Formatter to be associated with this view.
     */
    default void setFormatter(MonitorFormat formatter) {
    }
    
    /**
     * Returns {@code true} if this view has no content to display at the moment.
     * This can be used report that the view can be closed, for example.
     * The default implementation returns {@code false}.
     * @return True if this view has no content to display.
     */
    default boolean isEmpty() {
        return false;
    }
    
    
// -- Installation : -----------------------------------------------------------
    
    /**
     * Installs this view, connecting it to the data source.
     */
    default void install() {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) {
            filter = AgentChannelsFilter.ALL;
            setFilter(filter);
        }
        Console.getConsole().getStatusAggregator().addListener(this, filter);
    }
    
    /**
     * Un-installs this view, disconnecting it from the data source.
     * Uninstalling a view that has not been installed should have no effect.
     */
    default void uninstall() {
        Console.getConsole().getStatusAggregator().removeListener(this);
    }
            
}
