package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.io.Serializable;
import java.time.Duration;
import java.util.Arrays;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;

/**
 * Graphics panel that contains all other panels for a specific subsystem.
 * Used by both {@link BrowserFull} and {@link BrowserOneSubsystem}.
 * Contains {@code ComponentTreePanel}, {@code CommandListPanel}, {@code ArgInputPanel},
 * "Send" button and the results panel.
 *
 * @author onoprien
 */
public final class AgentPanel extends JSplitPane implements Savable, ChangeListener {

// -- Fields : -----------------------------------------------------------------
    
    private AgentStatus agent;
    
    final ComponentTree componentTree;
    private final CommandListPanel commandListPanel;
    private final ArgInputPanel argInputPanel;
    private final HTMLTextPane resultPane;
    
    private final CommandSender sender;
    private final JCheckBox hideLockedBox, hideSystemBox;
    private final JTextField timeoutField;
    private final JProgressBar progressBar;
    
    Descriptor descriptor;
    
    private CommandTask lastCommand;
    private long lastCommandStart;

// -- Life cycle : -------------------------------------------------------------
    
    public AgentPanel() {
        
        super(VERTICAL_SPLIT);
        setContinuousLayout(true);
        descriptor = new Descriptor();
        
        sender = Console.getConsole().getSingleton(CommandService.class).getSender();
        sender.setCommandHandle(new BrowserCommandHandle());
        
        // Results panel:

        JPanel topPane = new JPanel(new GridLayout(1, 3));
        setTopComponent(topPane);
        resultPane = new HTMLTextPane();
        setBottomComponent(new JScrollPane(resultPane));
        
        // Components tree panel:
        
        componentTree = new ComponentTree(this);
        topPane.add(new JScrollPane(componentTree));
        
        // Commands list panel:

        commandListPanel = new CommandListPanel(this);
        topPane.add(new JScrollPane(commandListPanel));
        componentTree.addTreeSelectionListener(commandListPanel);
        
        // Command description and argument input panel:

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPane.add(topPaneRightPanel);
        
        argInputPanel = new ArgInputPanel();
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        commandListPanel.addListSelectionListener(argInputPanel);
        
        // Settings and send button panel:
        
        JPanel controlPane = new JPanel(new GridBagLayout());
        topPaneRightPanel.add(controlPane, BorderLayout.SOUTH);
        controlPane.setBorder(BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, 0, Const.HSPACE));
        
        hideLockedBox = new JCheckBox("Hide locked");
        GridBagConstraints c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 0;
        c.anchor = GridBagConstraints.WEST;
        controlPane.add(hideLockedBox, c);
        hideLockedBox.setToolTipText("Hide commands unavailable due to agent lock state");
        hideLockedBox.setSelected(descriptor.isHideLocked());
        hideLockedBox.addActionListener(e -> {
            descriptor.setHideLocked(((JCheckBox)e.getSource()).isSelected());
            componentTree.updateData(agent);
        });
        
        hideSystemBox = new JCheckBox("Hide system");
        c.gridx = 0;
        c.gridy = 1;
        controlPane.add(hideSystemBox, c);
        hideSystemBox.setToolTipText("Hide system commands");
        hideSystemBox.setSelected(descriptor.isHideSystem());
        hideSystemBox.addActionListener(e -> {
            descriptor.setHideSystem(((JCheckBox)e.getSource()).isSelected());
            componentTree.updateData(agent);
        });
        
        JButton sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("ic_send_black_24dp.png")));
        c = new GridBagConstraints();
        c.gridx = 1;
        c.gridy = 0;
        c.gridheight = 2;
        c.weightx = 1.0;
        c.weighty = 1.0;
        c.anchor = GridBagConstraints.EAST;
        controlPane.add(sendCmdButton, c);
        sendCmdButton.setText("Send");
        sendCmdButton.setEnabled(false);
        sendCmdButton.addActionListener(e -> sendCommand());
        
        Box timeoutPanel = Box.createHorizontalBox();
        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 2;
        c.gridwidth = 2;
        c.weightx = 1.;
        c.weighty = 0.;
        c.insets = new Insets(Const.VSPACE, 0, Const.VSPACE, 0);
        c.anchor = GridBagConstraints.WEST;
        c.fill = GridBagConstraints.NONE;
        controlPane.add(timeoutPanel, c);
        timeoutPanel.add(new JLabel("Timeout:"));
        timeoutPanel.add(Box.createRigidArea(Const.HDIM));
        timeoutField = new JTextField(Integer.toString(descriptor.getTimeout()), 4);
        timeoutPanel.add(timeoutField);
        timeoutField.addActionListener(e -> setTimeout());
        timeoutField.addFocusListener(new FocusAdapter() {
            @Override
            public void focusLost(FocusEvent e) {
                setTimeout();
            }
        });
        timeoutPanel.add(new JLabel(" seconds."));
        timeoutPanel.add(Box.createHorizontalGlue());
        
        progressBar = new JProgressBar(0, descriptor.getTimeout());
        c = new GridBagConstraints();
        c.gridx = 0;
        c.gridy = 3;
        c.gridwidth = 2;
        c.weightx = 1.;
        c.weighty = 0.;
        c.anchor = GridBagConstraints.CENTER;
        c.fill = GridBagConstraints.HORIZONTAL;
        controlPane.add(progressBar, c);
//        progressBar.setStringPainted(true);
        
        // Enabling send button on command selection:

        commandListPanel.addListSelectionListener(e -> {
            boolean en = commandListPanel.isVisibleCommandSelected();
            sendCmdButton.setEnabled(en);
            argInputPanel.setEnabled(en);
        });
    }

    
// -- Receiving external events : ----------------------------------------------
    
    void setAgent(AgentStatus agent) {
        if (this.agent != null) {
            this.agent.removeListener(this);
        }
        this.agent = agent;
        if (agent != null) {
            agent.addListener(this);
        }
        componentTree.updateData(agent);
    }

    /**
     * Called in response to changes in agent data.
     * @param e Event
     */
    @Override
    public void stateChanged(ChangeEvent e) {
        if (agent == null || !agent.isAdjusting()) {
            componentTree.updateData(agent);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private void sendCommand() {
        String path = componentTree.getSelectedPathAsString();
        DictionaryCommand cmd = commandListPanel.getSelectedValue();
        if (path != null && cmd != null) {
            
            String s = argInputPanel.getFormattedArgumentValues();
            Object[] args = argInputPanel.getFormattedValues();
            String commandName = cmd.getCommandName();
            String command = path +"/"+ commandName;
            resultPane.insertHTML("<b>> " + path +" "+ commandName +" "+ s +" ...</b>");
            
            progressBar.setMaximum(descriptor.getTimeout() * 1000);
            progressBar.setValue(0);
            progressBar.setString(null);
            
            lastCommandStart = System.currentTimeMillis();
            lastCommand = sender.execute(command, args);
            
            Progress prog = new Progress(lastCommand);
            PeriodicTask task = new PeriodicTask(Console.getConsole().getScheduler(), prog, true, "Com browser progress bar", Level.SEVERE, 500, TimeUnit.MILLISECONDS);
            prog.setPeriodicTask(task);
            task.start(500, TimeUnit.MILLISECONDS);
        }
    }
    
    private void setTimeout() {
        try {
            int t = Integer.parseInt(timeoutField.getText());
            if (t > 0 && t < 9999) {
                descriptor.setTimeout(t);
                sender.setTimeout(Duration.ofSeconds(t));
            } else {
                timeoutField.setText(Integer.toString(descriptor.getTimeout()));
            }
        } catch (NumberFormatException x) {
            timeoutField.setText(Integer.toString(descriptor.getTimeout()));
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class BrowserCommandHandle implements CommandHandle {

        @Override
        public void onSendingFailure(Throwable t, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            if (t == null) {
                sb.append("Cannot send command.");
            } else {
                sb.append(shorten(t.getMessage()));
            }
            resultPane.insertHTML(sb.append("</font>").toString());
            resetProgressBar(source);
        }

        @Override
        public void onAck(CommandAck ack, CommandTask source) {
            if (source == lastCommand) {
                progressBar.setForeground(Color.GREEN);
                progressBar.setString("received");
                Duration customTimeout = ack.getTimeout();
                if (customTimeout != null) {
                    long current = System.currentTimeMillis();
                    int total = (int) (current - lastCommandStart + customTimeout.toMillis());
                    progressBar.setMaximum(total);
                    int progress = (int) (current - lastCommandStart);
                    progressBar.setValue(progress);
                }
            }
        }

        @Override
        public void onNack(CommandNack nack, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            sb.append("Command rejected");
            if (nack != null) {
                String message = shorten(nack.getReason());
                if (message != null && !message.isEmpty()) {
                    sb.append(": ").append(message);
                }
            }
            resultPane.insertHTML(sb.append("</font>").toString());
            resetProgressBar(source);
        }

        @Override
        public void onSuccess(Object res, CommandTask source) {
            if (res == null) {
                resultPane.insertHTML("Done.");
            } else {
                if (res.getClass().isArray()) {
                    try {
                        res = Arrays.deepToString((Object[]) res);
                    } catch (ClassCastException x) {
                    }
                }
                resultPane.insertHTML(res.toString().replaceAll("(\\r\\n|\\n)", "<br />"));
//                resultPane.insertHTML("<pre>"+ res.toString() +"</pre>");
            }
            resetProgressBar(source);
        }

        @Override
        public void onExecutionFailure(Throwable t, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            if (t == null) {
                sb.append("Failed.");
            } else {
                sb.append(shorten(t.getMessage()));
            }
            resultPane.insertHTML(sb.append("</font>").toString());
            resetProgressBar(source);
        }

        @Override
        public void onTimeout(TimeoutException x, CommandTask source) {
            resultPane.insertHTML("<font color=\"#FF0000\">"+ shorten(x.getMessage()) +"</font>");
            resetProgressBar(source);
        }

        @Override
        public void onCancel(CancellationException exception, CommandTask source) {
            resultPane.insertHTML("Cancelled.");
            resetProgressBar(source);
        }

        @Override
        public void onResult(Object res, CommandTask source) {
            resultPane.insertHTML("Result: "+ (res == null ? "" : res)); // never
            resetProgressBar(source);
        }
        
        private void resetProgressBar(CommandTask source) {
            if (source == lastCommand) {
                progressBar.setValue(0);
                lastCommand = null;
            }
        }
        
    }
    
    private class Progress implements Runnable {
        
        CommandTask commandTask;
        PeriodicTask periodicTask;
        
        Progress(CommandTask commandTask) {
            this.commandTask = commandTask;
        }
        
        void setPeriodicTask(PeriodicTask periodicTask) {
            this.periodicTask = periodicTask;
        }

        @Override
        public void run() {
            SwingUtilities.invokeLater(() -> {
                if (lastCommand == commandTask) {
                    int progress = (int) (System.currentTimeMillis() - lastCommandStart);
                    progressBar.setValue(progress);
                } else {
                    periodicTask.cancel(false);
                }
            });
        }
    }
    
// -- Saving/restoring : -------------------------------------------------------    
    
    @Override
    public void restore(Serializable descriptor) {
        
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
        
        hideLockedBox.setSelected(this.descriptor.isHideLocked());
        hideSystemBox.setSelected(this.descriptor.isHideSystem());
        int timeout = this.descriptor.getTimeout();
        timeoutField.setText(Integer.toString(timeout));
        sender.setTimeout(Duration.ofSeconds(timeout));
    }

    @Override
    public Descriptor save() {
        return descriptor.clone();
    }
    
    static public class Descriptor implements Serializable, Cloneable {

        private boolean hideLocked;
        private boolean hideSystem = true;
        private int timeout = 10;

        public boolean isHideLocked() {
            return hideLocked;
        }

        public void setHideLocked(boolean hideLocked) {
            this.hideLocked = hideLocked;
        }

        public boolean isHideSystem() {
            return hideSystem;
        }

        public void setHideSystem(boolean hideSystem) {
            this.hideSystem = hideSystem;
        }

        public int getTimeout() {
            return timeout;
        }

        public void setTimeout(int timeout) {
            this.timeout = timeout;
        }
        
        @Override
        public Descriptor clone() {
            try {
                return (Descriptor) super.clone();
            } catch (CloneNotSupportedException x) {
                throw new RuntimeException(); // never
            }
        }
        
    }
    
}
