package org.lsst.ccs.gconsole.services.rest;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.ws.rs.core.MediaType;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.localdb.statusdb.server.Data;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.Datas;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.statusdb.server.TrendingResult;
import org.lsst.ccs.messaging.AgentPresenceListener;

/**
 * Service plugin for retrieving data from the REST server.
 * This service provides methods for retrieving trending data, and accepts registration 
 * of listeners to be notified when the service is (re)connected to a new REST server.
 *
 * @author onoprien
 */
@Plugin(name="LSST Rest Service Plugin",
        id="rest-service",
        description="Plugin that handles connection to the REST server.")
public class LsstRestService extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    public static final String FROM_BUS_PROPERTY = "fromBus";
    public static final String HOST_PROPERTY = "server";
    public static final String PORT_PROPERTY = "port";
    public static final String CONNECT_ON_STARTUP_PROPERTY = "connectOnStartup";
    
    private Client client; // guarded by "this"
    private WebResource server;  // guarded by "this" 
    private boolean refreshInProgress; // guarded by "this"
    private boolean refreshRequested; // guarded by "this"
    private AgentPresenceListener agentsListener;  // guarded by "this"
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    
// -- Life cycle : -------------------------------------------------------------

    @Override
    public void initialize() {
        
        getServices().addProperty(FROM_BUS_PROPERTY, true);
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, null, "${"+ FROM_BUS_PROPERTY +"} Use REST server found on CCS buses.");
        
        getServices().addProperty(HOST_PROPERTY, "localhost");
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, "URL", "Server: ${"+ HOST_PROPERTY +"#history=5}");
        getServices().addProperty(PORT_PROPERTY, 8080);
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, "URL", "Port: ${"+ PORT_PROPERTY +"}");
        
        getServices().addProperty(CONNECT_ON_STARTUP_PROPERTY, true);
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, null, "${"+ CONNECT_ON_STARTUP_PROPERTY +"} Connect on startup.");
        
        getConsole().getLogger().setLevel(Level.FINE);
    }

    @Override
    public void start() {
        if ((Boolean) getServices().getProperty(CONNECT_ON_STARTUP_PROPERTY)) {
            refreshConnection();
        }
    }

    @Override
    synchronized public void shutdown() {
        if (client != null) client.destroy();
    }

    
// -- Connecting : -------------------------------------------------------------
    
    /**
     * Called by the framework in response to change in preferences.
     * This method should not be called by service clients.
     */
    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        Set<String> keys = changes.keySet();
        if (keys.contains(HOST_PROPERTY) || keys.contains(PORT_PROPERTY) || keys.contains(FROM_BUS_PROPERTY)) {
            triggerRefreshConnection();
        }
    }
    
    /**
     * Requests reestablishing connection based on current preferences.
     * If the connection is already being reestablished, this call is ignored.
     * This method returns immediately, connecting is done on a dedicated thread,
     * listeners are notified on EDT once the connection is established.
     */
    synchronized public void refreshConnection() {
        if (!refreshInProgress) {
            triggerRefreshConnection();
        }
    }
    
    /** Sets {@code refreshRequested} flag or triggers refresh, depending on whether refresh is already in progress. */
    synchronized private void triggerRefreshConnection() {
        if (!refreshInProgress) {
            refreshRequested = false;
            refreshInProgress = true;
            Thread t = new Thread(this::connect, "REST Connector");
            t.setDaemon(true);
            t.start();
        } else {
            refreshRequested = true;
        }
    }
    
    private void connect() {
        String url = null;
        try {
            url = getURL();
            if (url == null) {
                server = null;
            } else {
                getConsole().getLogger().fine("Connecting to REST server " + url);
                server = client.resource(url).path("dataserver");
                server.head();
                getConsole().getLogger().fine("REST server connection successful.");
            }
        } catch (RuntimeException e) {
            server = null;
            getConsole().getLogger().warn("Unable to connect to REST server at " + url, e);
        } finally {
            boolean repeat = false;
            synchronized(this) {
                if (refreshRequested) {
                    repeat = true;
                    refreshRequested = false;
                } else {
                    notifyAll();
                    refreshInProgress = false;
                    fireEvent();
                }
            }
            if (repeat) connect();
        }
    }

    private String getURL() {
        AgentPresenceListener oldListener;
        synchronized(this) {
            oldListener = agentsListener;
            if (client == null) client = Client.create(new DefaultClientConfig());
        }
        if (oldListener != null) {
            getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(oldListener);
        }
        String url = null;
        if ((Boolean) getServices().getProperty(FROM_BUS_PROPERTY)) {
            String agentName = null;
            for (AgentInfo agent : getConsole().getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
                if (AgentCategory.REST_SERVER.name().equals(agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY))) {
                    String host = agent.getAgentProperty("rest-service-addr");
                    String port = agent.getAgentProperty("rest-service-port");
                    String entryPoint = agent.getAgentProperty("rest-service-entrypoint");
                    url = String.format("http://%s:%d%s", host, Integer.valueOf(port), entryPoint);
                    agentName = agent.getName();
                    break;
                }
            }
            synchronized(this) {
                agentsListener = agentName == null ? new AgentConnectListener() : new AgentDisconnectListener(agentName);
            }
            getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(agentsListener);
        } else {
            String host = null;
            try {
                host = getServices().getProperty(HOST_PROPERTY).toString();
                int port = (Integer) getServices().getProperty(PORT_PROPERTY);
                url = String.format("http://%s:%d/rest/data", host, port);
            } catch (RuntimeException x) {
                getConsole().getLogger().warn("Invalid server " + host + " or port " + getServices().getProperty(PORT_PROPERTY));
                throw x;
            }
        }
        return url;
    }
    
    private class AgentConnectListener implements AgentPresenceListener {
        @Override
        public void connected(AgentInfo... agents) {
            for (AgentInfo agent : agents) {
                if (AgentCategory.REST_SERVER.name().equals(agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY))) {
                    triggerRefreshConnection();
                }
            }
        }
    }
    
    private class AgentDisconnectListener implements AgentPresenceListener {
        private final String agent;
        AgentDisconnectListener(String agent) {
            this.agent = agent;
        }
        @Override
        public void disconnecting(AgentInfo agent) {
            if (agent.getName().equals(this.agent)) {
                triggerRefreshConnection();
            }
        }        
    }
    
    
// -- Handling listeners : -----------------------------------------------------
    
    /**
     * Registers a listener to be notified when the service is (re)connected to a new REST server.
     * Listeners are notified on EDT.
     * 
     * @param listener Listener to be registered.
     */
    public void addListener(ChangeListener listener) {
        listeners.add(listener);
    }
    
    /**
     * Removes a listener.
     * 
     * @param listener Listener to be removed.
     */
    public void removeListener(ChangeListener listener) {
        listeners.remove(listener);
    }
    
    /**
     * Removes all listeners.
     */
    public void removeAllListeners() {
        listeners.clear();
    }
    
    private void fireEvent() {
        SwingUtilities.invokeLater(() ->  {
            ChangeEvent event = new ChangeEvent(this);
            listeners.forEach(listener -> listener.stateChanged(event));
        });
    }
    
    
// -- Retrieving data : --------------------------------------------------------
// FIXME: this mostly duplicates functionality in TrendingConnectionUtils and trending tool classes
    
    /**
     * Retrieves and returns the list of channels currently available from the REST server.
     * If the REST service has not been started earlier, if will be initialized before the data is retrieved.
     * 
     * @return List of channels, or {@code nill} if the REST server is not available.
     * @throws UniformInterfaceException If the status of the HTTP response is greater than or equal to 300.
     * @throws ClientHandlerException On failure to process the request or response.
     */
    public DataChannel.DataChannelList getChannelList() {
        WebResource resource = getServer();
        if (resource == null) return null;
        resource = resource.path("listchannels");
        return resource.accept(MediaType.TEXT_XML).get(DataChannel.DataChannelList.class);
    }
    
    /**
     * Retrieves and returns the list of channels currently available from the REST server.
     * Only channels that have data recorded in the last {@code begin} seconds are included.
     * If the REST service has not been started earlier, if will be initialized before the data is retrieved.
     * 
     * @param begin Maximum time with no data, in seconds.
     * @return List of channels, or {@code nill} if the REST server is not available.
     * @throws UniformInterfaceException If the status of the HTTP response is greater than or equal to 300.
     * @throws ClientHandlerException On failure to process the request or response.
     */
    public DataChannel.DataChannelList getChannelList(long begin) {
        WebResource resource = getServer();
        if (resource == null) return null;
        resource = resource.path("listchannels");
        resource = resource.queryParam("maxIdleSeconds", String.valueOf(begin));
        return resource.accept(MediaType.TEXT_XML).get(DataChannel.DataChannelList.class);
    }

    /**
     * Retrieves and returns the latest recorded data for the specified channel.
     * If the REST service has not been started earlier, if will be initialized before the data is retrieved.
     * 
     * @param path Data channel path.
     * @return Latest data, or {@code nill} if the REST server is not available.
     * @throws UniformInterfaceException If the status of the HTTP response is greater than or equal to 300.
     * @throws ClientHandlerException On failure to process the request or response.
     */
    // FIXME: how is the illegal path reported?
    public TrendingData getLatestData(String path) {
        WebResource resource = getServer();
        if (resource == null) return null;
        resource = resource.path("data/latest").queryParam("path", path);
        return resource.accept(MediaType.TEXT_XML).get(TrendingData.class);
    }
    
    /**
     * Retrieves and returns the data for the specified channel and time window.
     * If the REST service has not been started earlier, if will be initialized before the data is retrieved.
     * 
     * @param begin Start of the time window, milliseconds since January 1, 1970, 00:00:00 GMT.
     * @param end End of the time window, milliseconds since January 1, 1970, 00:00:00 GMT.
     * @param flavor Data flavor, "raw" or "stat".
     * @param nBins Requested number of bins. If non-positive, the number of bins is not specified in the request.
     * @param paths Paths of the data channels.
     * @return Data (including all available metadata), or {@code nill} if the REST server is not available.
     * @throws UniformInterfaceException If the status of the HTTP response is greater than or equal to 300.
     * @throws ClientHandlerException On failure to process the request or response.
     */
    public Datas getData(long begin, long end, String flavor, int nBins, String... paths) {
        WebResource resource = getServer();
        if (resource == null) return null;
        resource = resource.path("data/search");
        resource = resource.queryParam("t1", String.valueOf(begin)).queryParam("t2", String.valueOf(end));
        if (flavor != null) {
            resource = resource.queryParam("flavor", flavor);
        }
        if (nBins > 0) {
            resource = resource.queryParam("n", Integer.toString(nBins));
        }
        for (String path : paths) {
            resource = resource.queryParam("path", path);
        }
        WebResource.Builder a = resource.accept(MediaType.TEXT_XML);
        return a.get(Datas.class);
//        Data b = a.get(Data.class);
//        TrendingResult tr = b.getTrendingResult();
//        return tr;
    }
    
    /**
     * Retrieves and returns the data for the specified channel and time window.
     * If the REST service has not been started earlier, if will be initialized before the data is retrieved.
     * 
     * @param begin Start of the time window, milliseconds since January 1, 1970, 00:00:00 GMT.
     * @param end End of the time window, milliseconds since January 1, 1970, 00:00:00 GMT.
     * @param flavor Data flavor, "raw" or "stat".
     * @param nBins Requested number of bins. If non-positive, the number of bins is not specified in the request.
     * @param ids IDs of the data channels.
     * @return Data (including all available metadata), or {@code nill} if the REST server is not available.
     * @throws UniformInterfaceException If the status of the HTTP response is greater than or equal to 300.
     * @throws ClientHandlerException On failure to process the request or response.
     */
    public TrendingResult getData(long begin, long end, String flavor, int nBins, Integer... ids) {
        WebResource resource = getServer();
        if (resource == null) return null;
        resource = resource.path("data");
        resource = resource.queryParam("t1", String.valueOf(begin)).queryParam("t2", String.valueOf(end));
        if (flavor != null) {
            resource = resource.queryParam("flavor", flavor);
        }
        if (nBins > 0) {
            resource = resource.queryParam("n", Integer.toString(nBins));
        }
        for (Integer id : ids) {
            resource = resource.queryParam("id", id.toString());
        }
        return resource.accept(MediaType.TEXT_XML).get(Data.class).getTrendingResult();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    synchronized private WebResource getServer() {
        while (refreshInProgress) {
            try {
                wait();
            } catch (InterruptedException x) {
                return null;
            }
        }
        return server;
    }
    

}
