package org.lsst.ccs.gconsole.services.persist;

import java.awt.Component;
import java.io.Serializable;
import java.util.concurrent.CancellationException;
import org.lsst.ccs.gconsole.base.Console;

/**
 * A component that can be persisted between console sessions using {@link PersistenceService}.
 * 
 * The {@code save()} and {@code restore()} methods save/restore the component state to/from a JavaBean.
 * Exactly what is being saved and restored is up to a specific component - in general, one should not
 * expect that saving and then restoring a component will result in an identical object. Descriptors
 * returned by {@code save()} and taken by {@code restore()} should be independent of the internal state 
 * of this {@code Persistable}.
 * 
 * The {@code getDescriptor()} method provides access to the descriptor maintained by this component.
 * Implementations are expected to ensure that any properties set on the descriptor returned by this
 * method are reflected by descriptors subsequently returned by {@code save()}, unless modified
 * between the two calls. Descriptors should extend {@link Persistable.Descriptor} - see that class
 * documentation for additional requirements.
 *
 * @author onoprien
 */
public interface Persistable extends Savable {

    /**
     * Overrides {@code Savable.restore(Serializable)} to forward the call to {@link #restore(Descriptor)}
     * if the provided argument value is of type {@link Descriptor}, and do nothing otherwise.
     * 
     * @param descriptor Descriptor.
     */
    @Override
    public default void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            restore((Descriptor) descriptor);
        }
    }
    
    /**
     * Restores this component to the state specified by the provided JavaBean.
     * The default implementation does nothing.
     * 
     * @param descriptor JavaBean encapsulating the state.
     */
    default void restore(Descriptor descriptor) {
    }
    
    /**
     * Returns JavaBean describing the current state of this component.
     * The default implementation returns a clone of {@code getDescriptor()}.
     * 
     * @return JavaBean encapsulating the state, or {@code null} if no state information needs to be saved.
     */
    @Override
    default Descriptor save() {
        return getDescriptor().clone();
    }
    
    /**
     * Returns a reference to the descriptor maintained by this component.
     * Implementations are expected to ensure that this method never returns {@code null}, and
     * that any properties set on the descriptor returned by this method are reflected by
     * descriptors subsequently returned by {@code save()}, unless modified between the two calls.
     * 
     * @return JavaBean describing this component.
     */
    Descriptor getDescriptor();
    
    /**
     * Returns the category this {@code Persistable} belongs to.
     * The default implementation extracts the category from the descriptor.
     * @return Category of this {@code Persistable}.
     */
    default String getCategory() {
        return getDescriptor().getCategory();
    }
    
//    /**
//     * Returns the name of this {@code Persistable} instance.
//     * The default implementation returns the name set on the descriptor.
//     * @return Name of this {@code Persistable}.
//     */
//    default String getName() {
//        return getDescriptor().getName();
//    }
    
//    /**
//     * Sets the name of this {@code Persistable} instance.
//     * The default implementation sets the name on the descriptor.
//     * @param name Name to be assigned.
//     */    
//    default void setName(String name) {
//        getDescriptor().setName(name);
//    }
    
    /**
     * Takes user input and returns a modified {@code Persistable} instance.
     * This method should be called on EDT.
     * The default implementation returns {@code null}.
     * 
     * @param title User-interaction dialog title, or {@code null} if the default title should be used.
     * @param parent Graphical component to be use as a parent for user-interaction dialog(s).
     * @throws CancellationException If editing is canceled by the user.
     * @throws RuntimeException If editing fails for any reason.
     * @return Modified instance (may or may not be {@code this}).
     *         May return {@code null} to indicate that the {@code Persistable} should
     *         be re-instantiated, letting the user select among available {@code Creator}
     *         factories for the given category.
     */
    default Persistable edit(String title, Component parent) {
        return null;
    }
    
    
// -- Descriptor class : -------------------------------------------------------
    
    /**
     * JavaBean that describes a state of a {@link Persistable} component.
     * Provides a property that describes {@link Creator} invocation, as well as
     * properties used when saving or loading a {@code Persistable} component.
     * All classes implementing {@code Persistable} should use descriptors that extend this class.
     */
    static public class Descriptor implements Serializable, Cloneable {
        
        // -- Properties :

        private Creator.Descriptor creator;
        private String name;
        private String category;
        private String path;
        private String description;
        
        // -- Convenience methods :
        
        public void saveOrigin(Descriptor other) {
            if (other == null) return;
            setCreator(other.getCreator());
            setName(other.getName());
            setCategory(other.getCategory());
            setPath(other.getPath());
            setDescription(other.getDescription());
        }
        
        // -- Property setters/getters :

        public Creator.Descriptor getCreator() {
            return creator;
        }

        public void setCreator(Creator.Descriptor creator) {
            this.creator = creator;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getCategory() {
            if (category == null) {
                return creator == null ? null : creator.getCategory();
            } else {
                return category;
            }
        }

        public void setCategory(String category) {
            this.category = category;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public String getDescription() {
            return description;
        }

        public void setDescription(String description) {
            this.description = description;
        }
        
        // -- Cloning :

        @Override
        public Descriptor clone() {
            try {
                Descriptor clone = (Descriptor) super.clone();
                if (creator != null) {
                    clone.creator = creator.clone();
                }
                return clone;
            } catch (CloneNotSupportedException x) {
                return null;
            }
        }
        
    }
    
}
