package org.lsst.ccs.gconsole.services.aggregator;

import java.time.Instant;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;

/**
 * Collects status data from local and remote agents and notifies clients of relevant changes.
 * A singleton instance of this class is created by the graphical console.
 * <p>
 * The aggregator maintains a "watch list" that determines what data is being collected.
 * The watch list is empty on the aggregator creation, and is modified when
 * listeners are added or removed. See 
 * {@link #addListener(AgentStatusListener listener, AgentChannelsFilter filter)} and
 * {@link #addListener(AgentStatusListener listener, Collection agents, Collection channels)}
 * method descriptions for explanations of how filters and explicit lists of agents,
 * channels, templates, and selectors are used to determine what events are delivered 
 * to each listener.
 * <p>
 * When a filter fires a change event, this aggregator watch list is updated if necessary.
 * <p>
 * The aggregator modifies its state (in response to status messages received on the buses)
 * and notifies listeners on a single dedicated thread. No new messages will be processed,
 * and the aggregator state will not change, until all listeners process the current event.
 * Listeners that need to perform any time consuming or potentially blocking operations
 * should do that on other threads to avoid clogging the aggregator.
 * <p>
 * Methods that retrieve current status of agents and channels can be called on any thread - they
 * are guaranteed to report consistent information that reflects the aggregator state that existed
 * at some point between processing two consecutive status messages. There is no guarantee, however,
 * that the aggregator will remain in that state by the time these methods return. There is also no
 * guarantee that the state of the aggregator reflects the actual current state of an Agent,
 * due to the asynchronous nature of message delivery and delays in processing.
 *
 * @author onoprien
 */
public class AgentStatusAggregator {
    
// -- Fields : -----------------------------------------------------------------
    
    // Accessible to AgentHandle:
    
    private Agent localAgent; // local Agent
    private ConcurrentMessagingUtils messenger;
    final ThreadPoolExecutor exec; // single-threaded executor for processing status change events and notifying listeners
    final ArrayList<GlobalListenerHandle> listeners = new ArrayList<>(0); // all registered listeners
    
    // Private:

    private final AgentPresenceListener agentPresenceListener;
    private final StatusMessageListener statusMessageListener;

    private final ConcurrentHashMap<String,Instant> times = new ConcurrentHashMap<>(); // Agent name to local time of last message
    private final ConcurrentHashMap<String,AgentHandle> agentHandles = new ConcurrentHashMap<>(); // Agent name to handle (all agents present on buses are in this map)
    
    
// -- Life cycle : -------------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">

    public AgentStatusAggregator() {
        
        // Single-threaded executor that updates data structures and notifies listeners.
        
        ThreadFactory threadFactory = (run) -> {
            Thread thread = new Thread(run, "AgentStatusAggregator");
            thread.setDaemon(true);
            return thread;
        };
        exec = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, new LinkedBlockingQueue(), threadFactory) ;
        
        // AgentPresenceListener for responding to agents disconnect notifications
        
        agentPresenceListener = new AgentPresenceListener() {
            @Override
            public void disconnected(AgentInfo... agents) {
                exec.execute(() -> {
                    for (AgentInfo agent : agents) {
                        processDisconnect(agent.getName());
                    }
                });
            }
        };
        
        // StatusMessageListener for receiving status updates
        
        statusMessageListener = this::onStatusMessage;
    }
    
    /**
     * Initializes this status aggregator.
     * @param agent Local localAgent associated with this aggregator.
     */
    public void initialize(Agent agent) {
        this.localAgent = agent;
        AgentMessagingLayer messagingAccess = agent.getMessagingAccess();
        messenger = new ConcurrentMessagingUtils(messagingAccess);
        messagingAccess.addStatusMessageListener(statusMessageListener);
        messagingAccess.getAgentPresenceManager().addAgentPresenceListener(agentPresenceListener);
    }
    
    /**
     * Shuts down this status aggregator.
     * Once shut down, the aggregator cannot be restarted (this restriction can be removed in future versions if necessary).
     */
    public void shutdown() {
        AgentMessagingLayer messagingAccess = localAgent.getMessagingAccess();
        messagingAccess.removeStatusMessageListener(statusMessageListener);
        messagingAccess.getAgentPresenceManager().removeAgentPresenceListener(agentPresenceListener);
        exec.shutdownNow();
        messenger = null;
        localAgent = null;
    }

// </editor-fold>    
    
// -- Public getters : ---------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    /**
     * Returns a reference to the local agent that running this status aggregator.
     * @return Local agent.
     */
    public Agent getAgent() {
        return localAgent;
    }
    
    /**
     * Returns an instance of {@code ConcurrentMessagingUtils} that can be used to send commands to remote agents.
     * @return Instance of {@code ConcurrentMessagingUtils}.
     */
    public ConcurrentMessagingUtils getMessenger() {
        return messenger;
    }
    
    /**
     * Lists currently connected agents.
     * @return List of connected agents.
     */
    public List<AgentInfo> getAgents() {
        return agentHandles.values().stream().map(a -> a.getAgent()).collect(Collectors.toList());
    }
    
    /**
     * Return {@code AgentInfo} instance for the specified {@link Agent}.
     * 
     * @param agentName Name of the localAgent.
     * @return {@code AgentInfo} instance, or {@code null} is no localAgent with the specified name is connected to the buses.
     */
    public AgentInfo getAgent(String agentName) {
        AgentHandle data = agentHandles.get(agentName);
        return  data == null ? null : data.getAgent();
    }

    /**
     * Returns {@code true} if the specified localAgent is present on the buses.
     * 
     * @param agentName Agent name.
     * @return {@code true} if the localAgent is connected.
     */
    public boolean hasAgent(String agentName) {
        return agentHandles.get(agentName) != null;
    }
    
    /**
     * Returns a list of currently watched channels identified by the specified paths, templates, and selectors.
     * If no names are given, returns a list of all watched channels.
     * 
     * @param channels Paths, templates, selectors.
     * @return List of channels.
     */
    public List<AgentChannel> getChannels(String... channels) {
        ArrayList<AgentChannel> out = new ArrayList<>();
        for (AgentHandle agentHandle : agentHandles.values()) {
            synchronized (agentHandle) {
                if (channels.length > 0) {
                    for (AgentChannel c : agentHandle.getChannels()) {
                        for (String s : channels) {
                            if (AgentHandle.isSelector(s)) {
                                if (AgentHandle.acceptSelector(c, s)) out.add(c);
                            } else {
                                if (AgentHandle.matchTemplate(c.getPath(), s)) out.add(c);
                            }
                        }
                    }
                } else {
                    out.addAll(agentHandle.getChannels());
                }
            }
        }
        out.trimToSize();
        return out;
    }
    
    /**
     * Returns the channel specified by the given path.
     * 
     * @param path Data channel path.
     * @return Requested channel, or {@code null} if the channel is not being watched by this aggregator.
     */
    public AgentChannel getChannel(String path) {
        int i = path.indexOf("/");
        if (i == -1) return null;
        String agentName = path.substring(0, i);
        AgentHandle agentHandle = agentHandles.get(agentName);
        return agentHandle == null ? null : agentHandle.getChannel(path.substring(i+1));
    }
    
    /**
     * Returns the current state of the specified {@link Agent}.
     * 
     * @param agentName Name of the localAgent.
     * @return State of the specified localAgent.
     */
    public StateBundle getAgentState(String agentName) {
        AgentHandle agentHandle = agentHandles.get(agentName);
        return agentHandle == null ? null : agentHandle.getState();
    }
    
// </editor-fold>    

// -- Handling watched channels and listeners : --------------------------------
    
// <editor-fold defaultstate="collapsed">

    /**
     * Adds a listener to be notified of changes in agents and channels specified by the filter.
     * <p>
     * The listener is added
     * with the lists of agents and channels returned by the filter's {@code getAgents()} and
     * {@code getOriginChannels()} methods, respectively, unless any of those methods returns
     * {@code null} and the {@code getDisplayChannels()} method does not. In the latter case, the 
     * {@code null} list is replaced by the list reconstructed by applying {@code getOriginPath(...)}
     * to the output of {@code getDisplayChannels()}.
     * <p>
     * If the filter fires change event, the lists are re-evaluated, and the listener 
     * is notified through a configuration event.
     * 
     * @param listener Listener to be added.
     * @param filter Channels filter.
     * @throws NullPointerException if either argument is {@code null}.
     */
    public void addListener(AgentStatusListener listener, AgentChannelsFilter filter) {
        List<String> agents = filter.getAgents();
        List<String> channels = filter.getOriginChannels();
        if (agents == null || channels == null) {
            List<String> display = filter.getDisplayChannels();
            if (display != null) {
                ArrayList<String> reconChannels = new ArrayList<>(display.size());
                for (String d : display) {
                    String path = filter.getOriginPath(d);
                    if (path != null) {
                        reconChannels.add(path);
                    }
                }
                if (agents == null) {
                    agents = new ArrayList<>();
                    for (String path : reconChannels) {
                        String name = path.substring(0, path.indexOf("/"));
                        if (!agents.contains(name)) {
                            agents.add(name);
                        }
                    }
                }
                if (channels == null) {
                    channels = reconChannels;
                }
            }
        }
        List<String> finalAgents = agents == null ? null : new ArrayList<>(agents);
        List<String> finalChannels = channels == null ? Collections.singletonList("/") : new ArrayList<>(channels);
        exec.submit(() -> executeAddListener(listener, finalAgents, finalChannels));
    }

    /**
     * Adds a listener to be notified of changes in specified agents and data
     * channels. Note that calling this method with both {@code agents} and
     * {@code channels} arguments equal to {@code null} forces this aggregator
     * to watch all channels from all agents.
     * <p>
     * The list of channels passed to this method may contain:<ul>
     * <li>Complete channel paths.
     * <li>Path templates in the "{@code [Agent]/[partial path][/]}"
     * format. Templates that omit the Agent name are expanded against all
     * agents whose names are in the {@code agents} list (or all agents present
     * on the buses if that list is {@code null}). Templates that end with "/"
     * match all channels that start with "{@code Agent/partial path/}".
     * <li>Selector strings in the
     * "{@code [agent.name=value&][agent.type=value&][agent.key[=value]&…&agent.key[=value]&][key[=value]&…&key[=value]]}"
     * format, where {@code key} and {@code value} are names and values of
     * Agent properties or static channel attributes. If {@code value} is
     * omitted, the existence of the attribute is checked. Note that, unlike
     * templates, selectors are only applied to channels and their attributes as
     * listed in the agent channels dictionary. Any channels not appearing
     * in the dictionary are ignored, as are attribute changes.</ul>
     * <p>
     * The listener is notified of agent connection, configuration, and
     * disconnection events if the agent name is in the supplied list of
     * channels, or the list is {@code null}, or the Agent name is
     * explicitly contained in any of the templates and selectors.
     * <p>
     * The listener is notified of value changes in data channels if:<ul>
     * <li>The channel path is contained in the supplied list of channels.
     * <li>The channel path matches any of the supplied templates.
     * <li>The channel satisfies any of the supplied selectors.</ul>
     * <p>
     * Unless the agent list is {@code null}, templates and selectors that do not explicitly
     * specify agent names are only applied to agents whose names are on the Agent list.
     *
     * @param listener Listener to be added.
     * @param agents Names of agents the listener is interested in.
     *               If {@code null}, the listener will be notified of changes in all agents.
     * @param channels Paths, templates, and selectors specifying channels the listener is interested in.
     *                 If {@code null}, the listener is notified of changes in all channels from the
     *                 specified agents. Supplying an empty list ensures that the listener is only
     *                 notified of agent connections and disconnections.
     * @throws NullPointerException if the {@code listener} argument is {@code null}.
     */
    public void addListener(AgentStatusListener listener, Collection<String> agents, Collection<String> channels) {
        List<String> finalAgents = agents == null ? null : new ArrayList<>(agents);
        List<String> finalChannels = channels == null ? Collections.singletonList("/") : new ArrayList<>(channels);
        exec.submit(() -> executeAddListener(listener, finalAgents, finalChannels));
    }

    /**
     * Removes the specified listener.
     * Also stops watching channels that were only watched because of this listener.
     * 
     * @param listener Listener to be removed.
     */
    public void removeListener(AgentStatusListener listener) {
       exec.submit(() -> executeRemoveListener(listener));
    }
    
    private void executeAddListener(AgentStatusListener listener, List<String> agents, List<String> channels) {
        GlobalListenerHandle handle = new GlobalListenerHandle(listener, agents, channels);
        listeners.add(handle);
        for (AgentHandle agentHandle : agentHandles.values()) {
            agentHandle.addListener(handle);
        }
    }
    
    private void executeRemoveListener(AgentStatusListener listener) {
        Set<GlobalListenerHandle> removed = Collections.newSetFromMap(new IdentityHashMap<GlobalListenerHandle,Boolean>());
        Iterator<GlobalListenerHandle> it = listeners.iterator();
        while (it.hasNext()) {
            GlobalListenerHandle global = it.next();
            if (global.listener == listener) {
                it.remove();
                if (!removed.contains(global)) {
                    removed.add(global);
                    for (AgentHandle agentHandle : agentHandles.values()) {
                        agentHandle.removeListener(global);
                    }
                }
            }
        }
    }

// </editor-fold>
    
// -- Processing status messages: ----------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    private void onStatusMessage(StatusMessage message) {
        AgentInfo info = message.getOriginAgentInfo();
        if (info == null || info.getType() == AgentType.CONSOLE) return;
        String agentName = message.getOriginAgentInfo().getName();
        if (agentName == null) return;
        times.put(agentName, Instant.now());
        exec.execute(() -> processStatusMessage(message));
    }
    
    /** Runs on aggregator thread. */
    private void processStatusMessage(StatusMessage message) {
        
        AgentInfo agentInfo = message.getOriginAgentInfo();
        String agentName = agentInfo.getName();
        
        // Agent went offline - destroy handle and notify listeners, no other changes:
        
        if (message.getState().isInState(PhaseState.OFF_LINE)) {
            processDisconnect(agentName);
            return;
        }
        
        // Ignore messages while the remote agent is initializing:
        
        if (!(message.getState().isInState(PhaseState.OPERATIONAL) || message.getState().isInState(PhaseState.CLOSING))) {
            return; // May reconsider this, but as of toolkit 2.5.2, this is the only simple way to awoid asking for the dictionary too early
        }
        
        AgentHandle agentHandle = agentHandles.get(agentName);
        
        // First message from this agent - create handle and notify listeners:
        
        if (agentHandle == null) {
            agentHandle = new AgentHandle(agentInfo, this);
            agentHandle.onConnect(message);
            agentHandles.put(agentName, agentHandle);
        }
        
        // Process status changes:
        
        agentHandle.onMessage(message);
    }
    
    /** Runs on aggregator thread. */
    private void processDisconnect(String agentName) {
        AgentHandle agentHandle = agentHandles.remove(agentName);
        if (agentHandle != null) agentHandle.onDisconnect();
    }

// </editor-fold>
    
// -- Auxiliary classes : ------------------------------------------------------
    
// <editor-fold defaultstate="collapsed">
    
    /** Listener and associated data for use in the list of listeners maintained by aggregator. */
    class GlobalListenerHandle {
        
        AgentStatusListener listener;
        List<String> agents;
        List<String> channels;
        
        GlobalListenerHandle(AgentStatusListener listener, List<String> agents, List<String> channels) {
            this.listener = listener;
            this.agents = agents;
            this.channels = channels;
        }
    }
    
// </editor-fold>
    
}
