package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;

/**
 * {@link MonitorView} that splits all channels accepted by the filter into a small number of groups.
 *
 * @author onoprien
 */
public interface GroupView extends MonitorView {
    
    static public final MonitorField DEFAULT_FIELD = MonitorField.AVERAGE_VALUE;
    static public final MonitorField[] AVAILABLE_FIELDS = {
        MonitorField.AVERAGE_VALUE, 
        MonitorField.MAX_VALUE, 
        MonitorField.MIN_VALUE, 
        MonitorField.MEDIAN_VALUE
    };
    
    /**
     * Computes group name based on the display path returned by the filter.
     * The implementation provided by this class splits around the last "/".
     * 
     * @param displayPath Display path.
     * @return Group name, or {@code null} if this display path does not belong to any group.
     */
    default String getGroup(String displayPath) {
        int i = displayPath.lastIndexOf("/");
        return displayPath.substring(i+1);
    }
    
    /**
     * Computes a list of display channels groups.
     * @return List of groups, or {@code null} is the list is unknown at this time.
     */
    default List<String> getGroups() {
        List<String> channels = getFilter().getDisplayChannels();
        if (channels == null) return null;
        LinkedHashSet<String> groups = new LinkedHashSet<>();
        channels.forEach(displayPath -> {
            String group = getGroup(displayPath);
            if (group != null) groups.add(group);
        });
        return new ArrayList<>(groups);
    }
    
    /**
     * Retrieves a list {@code MonitorField} for display channel groups from the filter.
     * The assumption is that the list of corresponding strings is returned by {@code AgentChannelsFilter.getFields(true)}.
     * 
     * @return List of fields for display channel groups, or {@code null} if the filter does not provide the list.
     */
    default List<MonitorField> getFields() {
        AgentChannelsFilter filter = getFilter();
        if (filter == null) return null;
        List<String> channels = filter.getDisplayChannels();
        if (channels == null) return null;
        List<String> fields = filter.getFields(true);
        if (fields == null || fields.size() != channels.size()) return null;
        return fields.stream().map(s -> stringToField(s)).collect(Collectors.toList());
    }
    
    default MonitorField stringToField(String field) {
        for (MonitorField mf : AVAILABLE_FIELDS) {
            if (mf.toString().equals(field)) {
                return mf;
            }
        }
        return DEFAULT_FIELD;
    }
    
    default String fieldToString(MonitorField field) {
        return field.toString();
    }
    
}
