package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Represents a channel displayed by a monitoring views.
 * Each {@code DisplayChannel} is identified by a display path unique within a monitoring view,
 * and is linked to one or more original data channels that contribute to it.
 * Optionally, {@code DisplayChannel} can be assigned an {@link Updatable} target that
 * will be automatically updated whenever there are changes in the channel data.
 * 
 * @author onoprien
 */
public interface DisplayChannel {

    /**
     * Returns the display path associated with this channel.
     * @return Display path.
     */
    String getPath();

    /**
     * Returns the object that should be notified whenever there are changes to data channels associated with this display channel.
     * @return Target to be notified of changes.
     */
    Updatable getTarget();

    /**
     * Set the object that should be notified whenever there are changes to data channels associated with this display channel.
     * @param target Target to be notified of changes.
     */
    void setTarget(Updatable target);

    /**
     * Returns the original published data channel associated with this display channel.
     * It is up to implementations to decide what to return if multiple original channels are mapped to this display channel.
     * 
     * @return Data channel, or {@code null} if no data channels are associated with this display channel.
     */
    default AgentChannel getChannel() {
        List<AgentChannel> out = getChannels();
        switch (out.size()) {
            case 0: return null;
            case 1: return out.get(0);
            default: return null;
        }
    }

    /**
     * Returns an unmodifiable list of original published data channels associated with this display channel.
     * @return Data channel.
     */
    default List<AgentChannel> getChannels() {
        AgentChannel ch = getChannel();
        return ch == null ? Collections.emptyList() : Collections.singletonList(ch);
    }

    /**
     * Sets the original published data channel associated with this display channel.
     * 
     * @param channel Data channel.
     */
    void setChannel(AgentChannel channel);

    /**
     * Sets the list of original published data channels associated with this display channel.
     * Any previously associated channels are removed.
     * 
     * @param channels Data channels to be associated with this display channel.
     * @throws UnsupportedOperationException If the provided list contains multiple channels but the implementation only supports single channel association.
     */
    default void setChannels(Collection<AgentChannel> channels) {
        switch (channels.size()) {
            case 0: setChannel(null); break;
            case 1: setChannel(channels.iterator().next()); break;
            default: throw new UnsupportedOperationException("This display channel cannot be associated with multiple original channels.");
        }
    }
    
    /**
     * Adds an original published data channel to the list of channels associated with this display channel.
     * If a channel with the same path already exists, it is replaced.
     * 
     * @param channel Data channel.
     * @throws UnsupportedOperationException If the implementation only supports single channel association.
     */
    default void addChannel(AgentChannel channel) {
        throw new UnsupportedOperationException("This display channel cannot be associated with multiple original channels.");
    }
    
    /**
     * Removes an original published data channel from the list of channels associated with this display channel.
     * 
     * @param path Original path of the channel to be removed.
     * @return True if this display channel no longer has any data channels associated with it.
     * @throws UnsupportedOperationException If the implementation only supports single channel association.
     */
    default boolean removeChannel(String path) {
        throw new UnsupportedOperationException("This display channel cannot be associated with multiple original channels.");
    }
    
    /**
     * Removes all offline channels from the specified agent.
     * The default implementation does nothing and returns {@code false}.
     * 
     * @param agentName Agent name.
     * @return True if this display channel no longer has any data channels associated with it.
     */
    default boolean purgeOfflineChannels(String agentName) {
        return false;
    }
    
    /**
     * Updates the target.
     * The default implementation converts the list of attributes into the list of monitored fields using the target's
     * {@code getAffectedFields(...)} method, then calls the appropriate {@code update(...)} method of the target.
     * Most classes implementing this interface will not override this method.
     * 
     * @param attributes List of attributes whose values might have changed, or {@code null} if all attribute values might have changed.
     */
    default void update(List<String> attributes) {
        Updatable target = getTarget();
        if (target != null) {
            if (attributes == null) {
                target.update(this);
            } else {
                List<MonitorField> fields = target.getAffectedFields(attributes);
                if (fields == null) {
                    target.update(this);
                } else if (!fields.isEmpty()) {
                    target.update(this, fields);
                }
            }
        }
    }

}
