package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;

/**
 * Data model for a graphical component that displays a single monitored value.
 * There is a single {@link MonitorField} instance associated with this model, 
 * describing the type of the displayed value, and a list of contributing {@code DisplayChannel} objects.
 * The model also contains the current {@link FormattedValue}.
 * 
 * @author onoprien
 */
public class DefaultMonitorCell implements MonitorCell {

// -- Fields : -----------------------------------------------------------------

    static public DefaultMonitorCell EMPTY = new DefaultMonitorCell(Collections.emptyList(), MonitorField.NULL, FormattedValue.EMPTY);
    
    protected List<DisplayChannel> channels;
    protected MonitorField field;
    
    protected FormattedValue formattedValue;


// -- Life cycle : -------------------------------------------------------------

    /**
     * Constructs a cell with no associated channels, {@code NULL} field, and no formatted current value.
     */
    public DefaultMonitorCell() {
        this(Collections.emptyList(), MonitorField.NULL);
    }

    /**
     * Constructs a cell affected by a single display channel, with no current formatted value.
     * @param channel Handle of the channel to be displayed, or {@code null} if cell is unaffected by any channels.
     * @param field Field to be displayed.
     */
    public DefaultMonitorCell(DisplayChannel channel, MonitorField field) {
        this(channel == null ? Collections.emptyList() : Collections.singletonList(channel), field);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param channels Handles of the channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     */
    public DefaultMonitorCell(List<? extends DisplayChannel> channels, MonitorField field) {
        this(channels, field, null);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param channels Display channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     * @param data Current formatted value.
     */
    public DefaultMonitorCell(List<? extends DisplayChannel> channels, MonitorField field, FormattedValue data) {
        this.channels = channels == null ? Collections.emptyList() : (List<DisplayChannel>) channels;
        this.field = field == null ? MonitorField.NULL : field;
        this.formattedValue = data;
    }
    
    
// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the list of handles for data channels that affect this cell.
     * @return List of contributing channels.
     */
    @Override
    public List<DisplayChannel> getChannels() {
        return channels;
    }

    /**
     * Returns the descriptor of the monitored field this cell displays.
     * @return Field descriptor.
     */
    @Override
    public MonitorField getField() {
        return field;
    }

    /**
     * Sets the current formatted value to be displayed by this cell.
     * @param formattedValue New formatted value.
     */
    @Override
    public void setFormattedValue(FormattedValue formattedValue) {
        this.formattedValue = formattedValue;
    }

    /**
     * Returns the current formatted value to be displayed by this cell.
     * @return Current formatted value.
     */
    @Override
    public FormattedValue getFormattedValue() {
        return formattedValue;
    }

}
