package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.services.aggregator.MutableAgentChannel;

/**
 * Adapter class that simplifies implementing monitor views.
 * <p>
 * Classes extending {@code AbstractMonitorView2} should implement {@link #resetChannels()}
 * and, optionally,{@link #createChannelHandle(String, AgentChannel, Updatable)},
 * {@link #connect(AgentInfo)}, and {@link #disconnect(AgentInfo)} methods.
 * <p>
 * As long as implementations of {@link AgentStatusListener} methods, 
 * {@link #addChannels(AgentInfo agent, Map)}, {@link #removeChannels(AgentInfo, List)},
 * and {@link #updateChannels(AgentInfo, Map)} provided by this adapter are left unchanged,
 * the view will be updated automatically in response to status aggregator events.
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView2 extends AbstractMonitorView1 {

// -- Fields : -----------------------------------------------------------------
    
    protected final LinkedHashMap<String,DisplayChannel> path2data = new LinkedHashMap<>(); // display path to display channel, in addition order

// -- Life cycle : -------------------------------------------------------------
    
// -- Getters and setters : ----------------------------------------------------
    
    @Override
    public boolean isEmpty() {
        return path2data.isEmpty();
    }
    
    
// -- Implementing AbstractMonitorView1 : --------------------------------------

    /**
     * Adds channels to display.
     * @param agent Source agent.
     * @param channels Map of paths to channels.
     */
    @Override
    final protected void addChannels(AgentInfo agent, Map<String,AgentChannel> channels) {
        for (Map.Entry<String, AgentChannel> e : channels.entrySet()) {
            String path = e.getKey();
            DisplayChannel handle = createChannelHandle(path, e.getValue(), null);
            path2data.put(path, handle);
        }
        resetChannels();
    }

    /**
     * Removes a channels from this view.
     * @param agent Source agent.
     * @param paths Paths to remove.
     */
    @Override
    final protected void removeChannels(AgentInfo agent, List<String> paths) {
        for (String path : paths) {
            DisplayChannel handle = path2data.get(path);
            if (handle != null) {
                AgentChannel channel = handle.getChannel();
                if (channel instanceof MutableAgentChannel) {
                    ((MutableAgentChannel)channel).set(AgentChannel.Key.STATE, DataProviderState.OFF_LINE);
                    handle.update(null);
                }
            }
        }
    }
    
    /**
     * Called to notify this view that the specified channels have been updated.
     * @param agent Source agent.
     * @param channels Map of paths to key-value pairs, where the key is as {@code AgentChannel} object
     *                 and the value is a list of changed attributes.
     */
    @Override
    final protected void updateChannels(AgentInfo agent, Map<String,Map.Entry<AgentChannel,List<String>>> channels) {
        channels.forEach((path, e) -> {
            DisplayChannel handle = path2data.get(path);
            if (handle != null && handle.getTarget() != null) {
                if (!Objects.equals(handle.getChannel(), e.getKey())) {
                    handle.setChannel(e.getKey());
                    handle.update(null);
                } else {
                    handle.update(e.getValue());
                }
            }
        });
    }


// -- EDT hooks for updating the graphical component : -------------------------
    
    /**
     * Should be implemented to completely rebuild the view based on {@code path2data} field.
     * Called on EDT whenever channels have been added or removed.
     */
    abstract protected void resetChannels();
    
    /**
     * Creates a new instance of {@code DisplayChannel}.
     * Called on EDT whenever channels are being added. Handles created by this method are put into {@code path2data} map.
     * <p>
     * The default implementation provided by this class creates an instance of {@code DefaultChannelHandle}.
     * Subclasses may override this method to return a different implementation of {@code DisplayChannel}.
     * 
     * @param path Display path.
     * @param channel Agent channel.
     * @param view View displaying the channel.
     * @return New instance of {@code DisplayChannel}.
     */
    protected DisplayChannel createChannelHandle(String path, AgentChannel channel, Updatable view) {
        return new DisplayChannelSingle(path, channel, view);
    }
        
}
