package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.io.Serializable;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import javax.swing.SwingWorker;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.services.AgentCommandDictionaryService;
import org.lsst.ccs.services.AgentLockService;

/**
 * Tracks status of a remote {@link Agent}, notifies listeners of changes.
 * <ul>
 * <li>Keeps track of the locking state of a target agent, accepts requests for changes.
 * <li>Keeps track of connections and disconnections.
 * <li>Accepts {@code ChangeListener} registration. Listeners are notified whenever lock or level change,
 *     lock or level changes are requested through this {@code AgentWatch}, or the set of known commands
 *     changes (on agent connection and disconnection).
 * </ul>
 * All access to instances of this class should happen on EDT.
 * 
 * @author onoprien
 */
class AgentStatus implements Serializable {
    
// -- Fields : -----------------------------------------------------------------
        
    private final String name;
    private AgentInfo info;
    
    private final LsstCommandBrowserPlugin plugin;
    private final AgentLockService lockService;
    private final AgentCommandDictionaryService.AgentCommandDictionaryListener dictionaryListener;
    
    final Map<String, Dictionary> dictionaries = new HashMap<>();
    
    private AgentLock lock; // current lock
    private int level; // current lock level
    private boolean online; // true if the remote agent is currently connected
    private boolean adjusting; // true if a change in lock or level has been requested but not completed
    
    private final CopyOnWriteArrayList<ChangeListener> listeners = new CopyOnWriteArrayList<>();
    private final ChangeEvent changeEvent = new ChangeEvent(this);
    
// -- Life cycle : -------------------------------------------------------------
    
    AgentStatus(String agent) {
        this.name = agent;
        plugin = Console.getConsole().getSingleton(LsstCommandBrowserPlugin.class);
        lockService = Console.getConsole().getAgentService(AgentLockService.class);
        
        dictionaryListener = e -> {
            if (name.equals(e.getAgentInfo().getName())) {
                ThreadUtil.invokeLater(() -> {
                    switch (e.getEventType()) {
                        case ADDED:
                            online = true;
                            info = e.getAgentInfo();
                            dictionaries.clear();
                            dictionaries.putAll(e.getDictionary());
                            break;
                        case REMOVED:
                            online = false;
                            dictionaries.clear();
                            break;
                        case UPDATED:
                            lock = lockService.getLockForAgent(name);
                            level = lockService.getLevelForAgent(name);
                    }
                    listeners.forEach(listener -> listener.stateChanged(changeEvent));
                });
            }
        };
        
        online = false;
    }
    
    AgentStatus(AgentInfo agentInfo) {
        this(agentInfo.getName());
        info = agentInfo;
        online = true;
    }
    
    void init() {
        Console.getConsole().getAgentService(AgentCommandDictionaryService.class).addAgentCommandDictionaryListener(dictionaryListener);
    }
    
    void shutdown() {
        Console.getConsole().getAgentService(AgentCommandDictionaryService.class).removeAgentCommandDictionaryListener(dictionaryListener);
    }
    
// -- Getters : ----------------------------------------------------------------
    
    public String getName() {
        return name;
    }
    
    AgentInfo getInfo() {
        return info;
    }

    AgentLock getLock() {
        return lock;
    }

    int getLevel() {
        return level;
    }

    boolean isAdjusting() {
        return adjusting;
    }

    Map<String, Dictionary> getDictionaries() {
        return dictionaries;
    }

    boolean isOnline() {
        return online;
    }
    
    boolean isLockedByMe() {
        return lock != null && lock.getOwnerAgentName().equals(Console.getConsole().getAgentInfo().getName());
    }

    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Requests to acquire or release the lock.
     * When this method is called, the state of this AgentState changes to "adjusting",
     * listeners are notified, and the request is forwarded to {@code AgentLockService}.
     * 
     * @param locked If {@code true}, request the lock; if {@code false}, release the lock.
     */
    void setLock(boolean locked) {
        adjusting = true;
        notifyListeners();
        new SwingWorker<Object,Object>() {
            @Override
            protected Object doInBackground() throws Exception {
                if (locked) {
                    lockService.lockAgent(name);
                } else {
                    lockService.unlockAgent(name);
                }
                return null;
            }
            @Override
            protected void done() {
                try {
                    get();
                } catch (InterruptedException x) {
                    Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.");
                } catch (ExecutionException x) {
                    try {
                        Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.", (Exception) x.getCause());
                    } catch (ClassCastException xx) {
                        Console.getConsole().error("Unable to "+ (locked ? "" : "un") +"lock the "+ name +" subsystem.", x);
                    }
                }
                adjusting = false;
                notifyListeners();
            }
        }.execute();
    }

    /**
     * Requests the desired lock level.
     * When this method is called, the state of this AgentState changes to "adjusting",
     * listeners are notified, and the request is forwarded to {@code AgentLockService}.
     * 
     * @param locked If {@code true}, request the lock; if {@code false}, release the lock.
     */
    public void setLevel(int level) {
        adjusting = true;
        notifyListeners();
        new SwingWorker<Object,Object>() {
            @Override
            protected Object doInBackground() throws Exception {
                lockService.setLevelForAgent(name, level);
                return null;
            }
            @Override
            protected void done() {
                try {
                    get();
                } catch (InterruptedException x) {
                    Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.");
                } catch (ExecutionException x) {
                    try {
                        Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.", (Exception) x.getCause());
                    } catch (ClassCastException xx) {
                        Console.getConsole().error("Unable to set level "+ level +" for "+ name +" subsystem.", x);
                    }
                }
                adjusting = false;
                notifyListeners();
            }
        }.execute();
    }


// -- Handling listeners : -----------------------------------------------------
    
    public boolean addListener(ChangeListener listener) {
        return listeners.addIfAbsent(listener);
    }
    
    public boolean removeListener(ChangeListener listener) {
        return listeners.remove(listener);
    }
    
    public boolean removeAllListeners() {
        boolean out = ! listeners.isEmpty();
        listeners.clear();
        return out;
    }
    
    private void notifyListeners() {
        listeners.forEach(listener -> listener.stateChanged(changeEvent));
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true;
        if (obj instanceof AgentStatus) {
            return name.equals(((AgentStatus)obj).name);
        } else {
            return false;
        }
    }
    
    
}
