package org.lsst.ccs.gconsole.agent.command;

import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;

/**
 * Interface to be implemented by classes that process responses to commands sent to remote subsystems.
 *
 * @author onoprien
 */
public interface CommandHandle  {
    
    /**
     * Empty {@code CommandHandle} that does no processing.
     */
    static public final CommandHandle NONE = new CommandHandle() {};
    
    /**
     * Called if the command cannot be sent to the target subsystem.
     * The default implementation forwards the call to {@link #onResult}.
     * @param exception Exception thrown while attempting to send the command.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onSendingFailure(Throwable exception, CommandTask source) {
        onResult(exception, source);
    }
    
    /**
     * Called when an ACK is received.
     * @param ack ACK.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onAck(CommandAck ack, CommandTask source) {
    }
    
    /**
     * Called when the command is rejected.
     * The default implementation forwards the call to {@link #onResult}.
     * @param nack NACK.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onNack(CommandNack nack, CommandTask source) {
        onResult(nack, source);
    }        
    
    /**
     * Called when the command result that is not an exception is received.
     * The default implementation forwards the call to {@link #onResult}.
     * @param result Result.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onSuccess(Object result, CommandTask source) {
        onResult(result, source);
    }
    
    /**
     * Called when the target subsystem either throws an uncaught exception while trying
     * to execute the command, or returns an exception as the result of execution.
     * The default implementation forwards the call to {@link #onResult}.
     * @param exception Exception thrown while attempting to execute the command.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onExecutionFailure(Throwable exception, CommandTask source) {
        onResult(exception, source);
    }
    
    /**
     * Called if the command times out.
     * The default implementation forwards the call to {@link #onResult}.
     * @param exception Exception thrown.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onTimeout(TimeoutException exception, CommandTask source) {
        onResult(exception, source);
    }
    
    /**
     * Called if the command is successfully canceled.
     * The default implementation forwards the call to {@link #onResult}.
     * @param exception Exception thrown.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onCancel(CancellationException exception, CommandTask source) {
        onResult(exception, source);
    }
    
    /**
     * Called when either the result of the command execution is received, or
     * the command is canceled or times out. This method is guaranteed to be 
     * called for every command unless one of the more specific handlers -
     * {@link #onNack}, {@link #onSuccess}, {@link #onExecutionFailure}, {@link #onTimeout}, {@link #onCancel}
     * - is implemented.
     * The default implementation does nothing.
     * 
     * @param result Result of the command submission. Either an object returned
     * by the target subsystem, or a locally generated {@code TimeoutException}
     * or {@code InterruptedException}.
     * @param source {@code CommandTask} that executed the command.
     */
    default void onResult(Object result, CommandTask source) {
    }
    
    /**
     * Shortens messages longer than 200 characters.
     * This method is specifically tuned to deal with annoying command responses, like "Could not find command " + stack trace.
     * 
     * @param message Original message.
     * @return Shortened message
     */
    default String shorten(String message) {
        if (message != null && message.length() > 200) {
            Matcher m = Pattern.compile("\\s[A-Z]").matcher(message);
            int end = m.find(20) ? m.start() : 150;
            message = message.substring(0, end) + " ...";
        }
        return message;
    }
    
    
}
