package org.lsst.ccs.gconsole.services.persist;

import java.awt.Component;
import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.CancellationException;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.ConsoleService;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Describe;

/**
 * Service that facilitates persisting data between the graphical console sessions.
 *
 * @author onoprien
 */
@Plugin(name = "Persistence Service Plugin",
        id="persistence-service",
        description = "Service that facilitates persisting data between the graphical console sessions.")
public class PersistenceService extends ConsolePlugin implements ConsoleService {

// -- Fields : -----------------------------------------------------------------
    
    private final ArrayList<Creator> factories = new ArrayList<>();

// -- Life cycle : -------------------------------------------------------------
    
    /** Called by the framework to start the service. */
    @Override
    public void startService() {
        try {
            ClassLoader loader = getClass().getClassLoader();
            Enumeration<URL> e = loader.getResources(Create.RESOURCE);
            while (e.hasMoreElements()) {
                URL url = e.nextElement();
                try (
                        InputStream ins = url.openStream();
                        BufferedReader in = new BufferedReader(new InputStreamReader(ins));
                    ) 
                {
                    String line;
                    while ((line = in.readLine()) != null) {
                        try {
                            Class<?> clazz = Class.forName(line.trim());
                            for (Method m : clazz.getDeclaredMethods()) {
                                int mod = m.getModifiers();
                                if (Modifier.isPublic(mod) && Modifier.isStatic(mod)) {
                                    Create an = m.getAnnotation(Create.class);
                                    if (an != null) {
                                        factories.add(new CreatorExecutable(m, null));
                                    }
                                }
                            }
                            for (Constructor<?> c : clazz.getConstructors()) {
                                int mod = c.getModifiers();
                                if (Modifier.isPublic(mod)) {
                                    Create an = c.getAnnotation(Create.class);
                                    if (an != null) {
                                        factories.add(new CreatorExecutable(c, null));
                                    }
                                }
                            }
                        } catch (ClassNotFoundException | SecurityException x) {
                        }
                    }
                } catch (IOException x) {
                }
            }
        } catch (IOException x) {
        }
    }

// -- Creating instances : -----------------------------------------------------
    
    /**
     * Creates a {@code Persistable} instance based on the provided descriptor.
     * No user interaction. An instance of {@code Persistable} is constructed as follows:
     * <ul>
     * <li>if the descriptor's class has a target class, identified either explicitly with
     *     {@link Describe} annotation or by being an enclosing class, and the target class has
     *     a constructor that takes the descriptor as the only argument, that constructor is used;
     * <li>otherwise, if the descriptor has "creator" property and a {@code Creator} factory
     *     identified by it can be found, that factory is used;
     * <li>otherwise, if the descriptor's class has a target class with a default constructor,
     *     that constructor is used.
     * </ul>
     * After the {@code Persistable} instance is created, its {@code restore(...)} method is called.
     * 
     * @param descriptor JavaBean that describes the {@code Persistable} type and state.
     * @return Newly created {@code Persistable}, or {@code null} if it cannot be created.
     */
    public Persistable make(Persistable.Descriptor descriptor) {
        
        // Try to use target (enclosing or named in annotation) class constructor that takes a descriptor as an argument:
        
        Persistable out = null;
        Class<?> targetClass = null;
        Class<?> descriptorClass = descriptor.getClass();
        Describe an = descriptorClass.getAnnotation(Describe.class);
        if (an != null) {
            try {
                targetClass = Class.forName(an.className());
            } catch (ClassNotFoundException | LinkageError x) {
            }
        } else {
            targetClass = descriptorClass.getEnclosingClass();
        }
        boolean hasTargetClass = targetClass != null && Persistable.class.isAssignableFrom(targetClass);
        if (hasTargetClass) {
            try {
                Constructor c = targetClass.getConstructor(descriptorClass);
                out = (Persistable) c.newInstance(descriptor);
                return out;
            } catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException | InvocationTargetException x) {
            }
        }
        
        // Try to use creator:
        
        Creator.Descriptor desc = descriptor.getCreator();
        if (desc != null) {
            Creator creator = getFactory(desc.getCategory(), desc.getPath());
            if (creator != null) {
                try {
                    out = creator.make(desc.getParameters());
                } catch (Exception x) {
                }
            }
        }
        
        // Try to use target class default constructor:
        
        if (out == null && hasTargetClass) {
            try {
                Constructor c = targetClass.getConstructor();
                out = (Persistable) c.newInstance();
            } catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException | InvocationTargetException x) {
            }
            
        }
        
        // Copy Persistable descriptor properties:
 
        if (out != null) {
            Persistable.Descriptor d = out.getDescriptor();
            d.setCategory(descriptor.getCategory());
            d.setPath(descriptor.getPath());
            d.setName(descriptor.getName());
            d.setDescription(descriptor.getDescription());
            d.setCreator(desc);
            
        // Restore post-creation configuration
            
            out.restore(descriptor);
        }
        
        // Return created Persistable:
        
        return out;
    }
    
    /**
     * Creates a {@code Persistable} instance. 
     * 
     * @param category Persistable category.
     * @param path Path of the Creator that should be used.
     * @param parameters Parameters to be passed to the Creator.
     * @return Newly created {@code Persistable}, or {@code null} if it cannot be created.
     */
    public Persistable make(String category, String path, Object... parameters) {
        Creator creator = getFactory(category, path);
        if (creator != null) {
            try {
                return creator.make(parameters);
            } catch (Exception x) {
            }
        }
        return null;
    }
    
    /**
     * Creates a {@code Persistable} instance by letting the user to select one of
     * the available creators in the given category, and using the provided
     * descriptor (if any) as the starting point.
     * Returns {@code null} if the user cancels creation, or an instance cannot be created for any reason.
     * 
     * @param descriptor Descriptor to be used as a starting point.
     * @param title User-interaction dialog title, or {@code null} if the default title should be used.
     * @param parent Graphical component to be use as a parent for user-interaction dialog(s).
     * @param category Persistable category. If {@code null}, the category will be extracted from the provided descriptor.
     * @throws IllegalArgumentException if the category is not provided and cannot be inferred from the descriptor.
     * @return Newly created {@code Persistable}, or {@code null} if the object creation was canceled by the user.
     */
    public Persistable make(Persistable.Descriptor descriptor, String title, Component parent, String category) {
        if (category == null) {
            if (descriptor != null) category = descriptor.getCategory();
            if (category == null) throw new IllegalArgumentException("Persistable category is not defined");
        }
        ArrayList<Creator> ff = getSaved(category);
        ff.addAll(getFactories(category));
        return CreationDialog.make(descriptor, title, parent, ff);
    }
    
    /**
     * Saves the specified descriptor to disk, making it available in future console sessions.
     * The method will display an error message if the descriptor does not have category or path set.
     * 
     * @param descriptor Descriptor to be saved.
     */
    public void save(Persistable.Descriptor descriptor) {
        try {
            String category = descriptor.getCategory();
            String path = descriptor.getPath();
            ArrayList<Persistable.Descriptor> dd = getSavedDescriptors(category);
            Iterator<Persistable.Descriptor> it = dd.iterator();
            while (it.hasNext()) {
                Persistable.Descriptor d = it.next();
                if (path.equals(d.getPath())) {
                    it.remove();
                }
            }
            dd.add(descriptor);
            Path filePath = getConsole().getHomeDirectory().resolve(categoryToFileName(category));
            if (Files.notExists(filePath)) Files.createDirectories(filePath.getParent());
            BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(filePath));
            try (XMLEncoder encoder = new XMLEncoder(out)) {
                for (Persistable.Descriptor d : dd) {
                    encoder.writeObject(d);
                }
            }
        } catch (NullPointerException | IOException x) {
            getConsole().error("Unable to save descriptor", x);
        }
    }
    
    /**
     * Saves the descriptor to disk, letting the user specify path, name, and description.
     * The saved descriptor will be available in future console sessions.
     * The method will display an error message if the descriptor cannot be saved.
     * 
     * @param descriptor Descriptor to be saved.
     * @param title User interaction dialog title.
     * @param parent User interaction dialog parent component, or {@code null} if the parent is the console window.
     */
    public void saveAs(Persistable.Descriptor descriptor, String title, Component parent) {
        try {
            descriptor = SaveDescriptorDialog.show(descriptor, title, parent);
            save(descriptor);
        } catch (CancellationException x) {
        }
    }
    
    /**
     * Deletes the specified descriptor from storage.
     * 
     * @param descriptor Descriptor to be removed.
     */
    public void delete(Persistable.Descriptor descriptor) {
        try {
            String category = descriptor.getCategory();
            if (category == null) {
                category = descriptor.getCreator().getCategory();
            }
            String path = descriptor.getPath();
            ArrayList<Persistable.Descriptor> dd = getSavedDescriptors(category);
            Iterator<Persistable.Descriptor> it = dd.iterator();
            while (it.hasNext()) {
                Persistable.Descriptor d = it.next();
                if (path.equals(d.getPath())) {
                    it.remove();
                }
            }
            Path filePath = getConsole().getHomeDirectory().resolve(categoryToFileName(category));
            if (Files.notExists(filePath)) Files.createDirectories(filePath.getParent());
            BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(filePath));
            try (XMLEncoder encoder = new XMLEncoder(out)) {
                for (Persistable.Descriptor d : dd) {
                    encoder.writeObject(d);
                }
            }
        } catch (NullPointerException | IOException x) {
            getConsole().error("Unable to delete descriptor", x);
        }
    }
    
    /**
     * Lets the user select one of the saved descriptors in the specified category.
     * 
     * @param category {@code Persistable} object category.
     * @param title User interaction dialog title.
     * @param parent  User interaction dialog parent component, or {@code null} if the parent is the console window.
     * @return Chosen descriptor.
     */
    public Persistable.Descriptor load(String category, String title, Component parent) {
        ArrayList<Creator> ff = getSaved(category);
        Creator c = CreationDialog.select(title, parent, ff);
        if (c instanceof CreatorFromDescriptor) {
            return ((CreatorFromDescriptor)c).getDescriptor();
        } else {
            return null;
        }
    }
    
    
// -- Utilities : --------------------------------------------------------------
    
    /**
     * Utility method that returns an instance creator identified by category and path.
     * 
     * @param category Creator category.
     * @param path Creator path.
     * @return Desired factory, or {@code null} if no such factory is known to this service.
     */
    public Creator getFactory(String category, String path) {
        for (Creator f : factories) {
            if (path.equals(f.getPath())) {
                if (category == null || category.equals(f.getCategory())) {
                    return f;
                }
            }
        }
        return null;
    }
    
    /**
     * Utility method that returns all registered instance creators in the specified category.
     * 
     * @param category {@code Persistable} object category.
     * @return List of known creators.
     */
    public List<Creator> getFactories(String category) {
        if (category == null) {
            return new ArrayList<>(factories);
        } else {
            return factories.stream().filter(f -> category.equals(f.getCategory())).collect(Collectors.toList());
        }
    }
    
    /**
     * Returns a list of saved descriptors in the specified category.
     * 
     * @param category {@code Persistable} object category.
     * @return List of saved descriptors.
     */
    public ArrayList<Persistable.Descriptor> getSavedDescriptors(String category) {
        ArrayList<Persistable.Descriptor> out = new ArrayList<>();
        try {
            Path filePath = getConsole().getHomeDirectory().resolve(categoryToFileName(category));
            try (XMLDecoder decoder = new XMLDecoder(new BufferedInputStream(Files.newInputStream(filePath)), null, e -> {})) {
                try {
                    while (true) {
                        Persistable.Descriptor d = (Persistable.Descriptor) decoder.readObject();
                        out.add(d);
                    }
                } catch (ArrayIndexOutOfBoundsException x) {
                }
            } catch (IOException | ClassCastException | ArrayIndexOutOfBoundsException x) {
            }
        } catch (NullPointerException | IllegalArgumentException x) {
        }
        return out;
    }
    
    private ArrayList<Creator> getSaved(String category) {
        List<Persistable.Descriptor> dd = getSavedDescriptors(category);
        ArrayList<Creator> out = new ArrayList<>(dd.size());
        for (Persistable.Descriptor d : dd) {
            out.add(new CreatorFromDescriptor(d));
        }
        return out;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private String categoryToFileName(String category) {
        StringBuilder sb = new StringBuilder("saved/");
        boolean start = true;
        for (char c : category.toLowerCase().toCharArray()) {
            if (start) {
                if (Character.isLetter(c)) {
                    sb.append(c);
                    start = false;
                }
            } else {
                if (Character.isLetterOrDigit(c) || c == '.' || c == '_') {
                    sb.append(c);
                }
            }
        }
        if (start) throw new IllegalArgumentException("Cannot convert category "+ category +" to file name");
        sb.append(".xml");
        return sb.toString();
    }
    
}
