package org.lsst.ccs.gconsole.services.persist;

import java.io.Serializable;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import org.lsst.ccs.gconsole.util.tree.HasPath;
import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;
import org.lsst.ccs.utilities.conv.TypeUtils;

/**
 * An object that can instantiate a {@link Persistable}.
 *
 * @author onoprien
 */
public interface Creator extends HasPath {
    
// -- Getters : ----------------------------------------------------------------
    
    /**
     * Returns the category this factory belongs to.
     * The category determines the set of factories the user is prompted to select from
     * when interactively creating a {@link Persistable}.
     * 
     * @return Category ID.
     */
    String getCategory();
    
    /**
     * Returns the slash-separated path to this factory.
     * The path determines placement in the tree of factories the user is prompted
     * to select from when interactively creating a {@link Persistable}.
     * The path must be unique among factories in the same category.
     * 
     * @return Path to this factory.
     */
    @Override
    String getPath();
    
    /**
     * The name of this factory.
     * The name is presented to the user along with the description but is not used internally.
     * 
     * @return Human-readable name.
     */
    String getName();
    
    /**
     * The description of this factory.
     * 
     * @return Description (html allowed).
     */
    String getDescription();
    
    /**
     * The list of parameters of this factory.
     * 
     * @return List of parameters.
     */
    Parameter[] getParameters();
    
    /**
     * The list of the names of the parameters of this factory.
     * @return List of parameter names.
     */
    String[] getParameterDescriptions();
    
    
// -- Creating an instance : ---------------------------------------------------
    
    /**
     * Creates an instance of a {@code Persistable}.
     * Implementations must ensure that the created instance has correct creator properties set on its descriptor.
     * 
     * @param parameters Creator parameters to be used.
     * @return The created component.
     * @throws Exception If the instance cannot be created. The exception should indicate the reason for the failure.
     */
    Persistable make(Object... parameters) throws Exception;
    
    /**
     * Creates an instance of a {@code Persistable}.
     * Implementations must ensure that the created instance has correct factory properties set on its descriptor.
     * 
     * @param parametersAsStrings String representations of the factory parameters to be used.
     * @return The created component.
     * @throws Exception If the instance cannot be created. The exception should indicate the reason for the failure.
     */
    Persistable make(String... parametersAsStrings) throws Exception;
    
    
// -- Utilities : --------------------------------------------------------------
    
    /**
     * Converts a parameter value to its string representation.
     * 
     * @param value Parameter value.
     * @param valueType Parameter type.
     * @return String representation of the parameter value.
     * @throws IllegalArgumentException If the provided arguments are incompatible.
     */
    static public String encode(Object value, Type valueType) {
        try {
            return TypeUtils.stringify(value);
        } catch (RuntimeException x) {
            throw new IllegalArgumentException(x);
        }
    }
    
    /**
     * Converts a string representation of a parameter value to the actual value.
     * 
     * @param encodedValue String representation of a parameter value.
     * @param valueType Parameter type.
     * @return  Parameter value.
     * @throws IllegalArgumentException If the provided string cannot be converted to an object of the specified type.
     */
    static public Object decode(String encodedValue, Type valueType) {
        try {
            return InputConversionEngine.convertArgToType(encodedValue, valueType);
        } catch (TypeConversionException x) {
            throw new IllegalArgumentException(x);
        }
    }

    
// -- Descriptor class : -------------------------------------------------------
    
    /**
     * JavaBean that describes an invocation of a {@link Creator}.
     * This descriptor identifies a factory by category and path, and contains the values of parameters.
     * 
     * // FIXME: Add special values for category, like CLASS.
     */
    static public class Descriptor implements Serializable, HasPath, Cloneable {

        private String category;
        private String path;
        private String[] parameters;
        
        public Descriptor() {
        }
        
        public Descriptor(String category, String path, String... parameters) {
            this.category = category;
            this.path = path;
            if(parameters.length != 0) this.parameters = parameters;
        }

        public String getCategory() {
            return category;
        }

        public void setCategory(String category) {
            this.category = category;
        }

        @Override
        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public String[] getParameters() {
            return parameters;
        }

        public void setParameters(String[] parameters) {
            this.parameters = parameters;
        }

        @Override
        protected Descriptor clone() throws CloneNotSupportedException {
            return (Descriptor) super.clone();
        }
        
    }

}
