package org.lsst.ccs.gconsole.plugins.trending;

import jas.hist.DataSource;
import jas.hist.HasStyle;
import jas.hist.HistogramUpdate;
import jas.hist.JASHist1DHistogramStyle;
import jas.hist.JASHistAxis;
import jas.hist.JASHistStyle;
import jas.hist.XYDataSource;
import java.awt.Color;
import java.awt.Component;
import java.awt.Point;
import java.io.Serializable;
import java.util.*;
import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JSeparator;
import org.freehep.jas.plugin.plotter.DefaultPlotter;
import org.freehep.jas.plugin.plotter.DefaultRegion;
import org.freehep.jas.plugin.plotter.JAS3DataSource;
import org.freehep.jas.plugin.plotter.JAS3Plot;
import org.freehep.jas.plugin.tree.FTreePath;
import org.freehep.jas.services.PlotPage;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.Plotter;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;

/**
 * Plotter for {@link Trend} objects.
 * All methods should be called on EDT.
 *
 * @author onoprien
 */
public final class TrendPlotter implements Plotter {
    
    /** Option that control plotting methods. */
    enum Option {
        /** Replace an existing plot for the same channel if possible. */ EXIST,
        /** Plot in a new page unless an existing plot is being replaced. */ NEWPAGE,
        /** Plot in a new region on the current page unless an existing plot is being replaced. */ NEWPLOT,
        /** Overlay on top of the current region */ OVERLAY,
        /** Fetch data at low priority. */ SLOW,
        /** Regions are redrawn whether or not any datasets they contain have been refreshed. */ REPLOT
    }

// -- Fields : -----------------------------------------------------------------
    
    /** Option to not plot any off-point metadata. */
    public static final String NO_META = "no-meta";
    
    private static final Color[] colors = {Color.BLUE, Color.RED, Color.GREEN, Color.MAGENTA, Color.ORANGE, Color.CYAN, Color.DARK_GRAY, Color.LIGHT_GRAY};

    private final LsstTrendingPlugin plugin;
    private final DefaultPlotter plotter;
    private final ArrayList<TrendInfo> trends = new ArrayList<>(1); // currently displayed trends
    
    private PlotRegion region;
    
    private TimeWindow timeWindow;
    private long timestamp; // last refresh, or -1 if in the process of refreshing    
    
    private Trend.Meta metaOnPoint;
    private EnumSet<Trend.Meta> metaOffPoint;
    private boolean isCustomRangeT, isCustomRangeV;
    private double[] rangeT, rangeV; // last automatically computed range; custom range is kept by the plot itself.
    private boolean ignoreMeta;


// -- Life cycle : -------------------------------------------------------------
    
    public TrendPlotter(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
        reset();
        try {
            plotter = (DefaultPlotter) plugin.getPlotFactory().createPlotterFor(JAS3DataSource.class);
            plotter.getPlot().setShowStatistics(false);
            plotter.getPlot().getXAxis().setRangeAutomatic(false);
            plotter.getPlot().getYAxis().setRangeAutomatic(false);
        } catch (ClassCastException x) {
            throw new RuntimeException("TrendPlotter is designed to wrap DefaultPlotter", x);
        }
    }


// -- Getters and setters : ----------------------------------------------------

    public PlotRegion getRegion() {
        return region;
    }
    
    public void setRegion(PlotRegion region) {
        this.region = region;
        try {
            DefaultRegion dr = (DefaultRegion) region;
            dr.addPopupItems(new Popup());
        } catch (ClassCastException x) {
        }
    }
    
    public EnumSet<Trend.Meta> getMeta() {
        EnumSet<Trend.Meta> out = metaOnPoint == null ? EnumSet.noneOf(Trend.Meta.class) : EnumSet.of(metaOnPoint);
        out.addAll(metaOffPoint);
        return out;
    }
    
    public void setMeta(EnumSet<Trend.Meta> extras) {
        throw new UnsupportedOperationException("Not yet implemented.");
    }
    
    /**
     * Returns {@code true} if this plotter is currently managed by the graphical console.
     * The plotter's viewable component does not have to be visible for the plotter to be active.
     * 
     * @return {@code True} if this plotter is active.
     */
    public boolean isActive() {
        if (region != null) {
            try {
                return ((JComponent)viewable()).getTopLevelAncestor() != null;
            } catch (ClassCastException|NullPointerException x) {
            }
        }
        return false;
    }
    
    /**
     * Lists trends currently displayed by this plotter.
     * @return The list of currently displayed trends.
     */
    public List<Trend> getTrends() {
        ArrayList<Trend> out = new ArrayList<>(trends.size());
        trends.forEach(info -> out.add(info.trend));
        return out;
    }
    
    /**
     * Returns {@code true} if this plotter is empty.
     * @return {@code true} if this plotter does not display any trends.
     */
    public boolean isEmpty() {
        return trends.isEmpty();
    }


// -- Implementing Plotter : ---------------------------------------------------
    
    /**
     * Plots a trend. No style or options.
     * See {@link #plot(Object,int,Object,String) plot(Object timeHistory, int mode, Object style, String options)} for details.
     * 
     * @param timeHistory  An instance of {@link Trend} to be plotted.
     * @param mode Plotter.NORMAL or Plotter.OVERLAY.
     */
    @Override
    public void plot(Object timeHistory, int mode) {
        plot(timeHistory, mode, null, null);
    }
    
    /**
     * Plot the specified trends, without resetting the plotter.
     */
    /**
     * Plot the specified trends.
     * 
     * @param trendList The list of trends to plot.
     * @param mode Plotter.NORMAL or Plotter.OVERLAY.
     * See {@link #plot(Object,int,Object,String) plot(Object timeHistory, int mode, Object style, String options)} for details.
     */
    public void plot(List<Trend> trendList, int mode) {
        if (trendList.size() == 1) {
            plot(trendList.get(0), mode);
        } else {
            trendList.forEach(trend -> plot(trend, Plotter.OVERLAY, null, NO_META));
        }
    }

    /**
     * Plots a trend.
     * <p>
     * If a trend is plotted in NORMAL mode, the existing content of this plotter 
     * is erased and all properties are reset. In OVERLAY mode, the existing content 
     * and custom properties are retained except that any datasets representing off-point
     * metadata are removed.
     * 
     * @param timeHistory An instance of {@link Trend} to be plotted.
     * @param mode Plotter.NORMAL or Plotter.OVERLAY.
     * @param style Ignored at the moment.
     * @param options Comma-separated list of options:<br>
     *                &nbsp; NO_META ("no-meta") - do not plot any off-point metadata.
     */
    @Override
    public void plot(Object timeHistory, int mode, Object style, String options) {
        
        if (!(timeHistory instanceof Trend && (mode == Plotter.NORMAL || mode == Plotter.OVERLAY))) return;
        Set<String> opt;
        if (options == null || options.trim().isEmpty()) {
            opt = Collections.emptySet();
        } else {
            opt = new HashSet<>(Arrays.asList(options.split("\\s*,\\s*")));
        }
        
        // Deal with existing content of this plotter
        
        if (mode == Plotter.NORMAL) { // replacing previous plot (clean and reset)
            clear();
        } else {
            checkRange();
            if (!trends.isEmpty() && trends.get(0).hasOffPoint()) { // overlaying plot with off-point metadata (re-plot the on-point dataset only)
                clearPlotter(true);
                trends.get(0).removeOffPoint();
                plotter.plot(trends.get(0).data.get(0), Plotter.OVERLAY);
            }
        }
        
        // Add Trend
        
        Trend trend = (Trend) timeHistory;
        trend.setPlotter(this);
        TrendInfo trendInfo = new TrendInfo(trend);
        int trendIndex = trends.size();
        trends.add(trendInfo);
        
        // Create on-point dataset
        
        Jas3Source ds = new Jas3Source(trendIndex, metaOnPoint, null);
        trendInfo.data.add(ds);
        
        // Create off-point metadata
        
        if (!metaOffPoint.isEmpty() && trends.size() == 1 && !opt.contains(NO_META)) {
            for (Trend.Meta meta : metaOffPoint) {
                for (String key : meta.getKeys()) {
                    ds = new Jas3Source(trendIndex, meta, key);
                    trendInfo.data.add(ds);
                }
            }
        }
        
        // Set plotter properties; plot datasets
        
        plotter.plot(trendInfo.data.get(0), Plotter.OVERLAY);
        for (int i=1; i<trendInfo.data.size(); i++) {
            plotter.plot(trendInfo.data.get(i), Plotter.OVERLAY);
        }
        setPlotProperties();
        
    }

    /** Not implemented. */
    @Override
    public void remove(Object data) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** Removes all plotted data and resets properties. */
    @Override
    public void clear() {
        clearData();
        reset();
        clearPlotter(false);
    }

    /**
     * Returns the graphic component of this plotter.
     * @return The GUI component that contains the plot.
     */
    @Override
    public Component viewable() {
        return plotter.viewable();
    }

    /**
     * Returns the list of trends displayed by this plotter.
     * @return The list of {@link Trend} instances displayed by this plotter.
     */
    @Override
    public List<Object> getData() {
        ArrayList<Object> out = new ArrayList<>(trends.size());
        trends.forEach(info -> out.add(info.trend));
        return out;
    }
    
    
// -- Additional operations : --------------------------------------------------
    
    /**
     * Makes sure the page and region that contain this plotter are visible and selected.
     */
    public void toFront() {
        if (region != null) {
            PlotPage page = region.getPage();
            if (page != null) {
                page.setCurrentRegion(region);
                page.showPage();
            }
        }
    }
    
    
// -- Receive notifications from Trends : --------------------------------------
    
    /**
     * Called by Trend when it starts loading new data.
     * @param trend Source trend.
     */
    void onRefreshRequest(Trend trend) {
        for (TrendInfo info : trends) {
            if (info.trend == trend) {
                info.messageID = plugin.getConsole().setStatusMessage("Loading trending data...", 0);
                break;
            }
        }
    }
    
    void onDataChange(Trend trend) {
        TrendInfo trendInfo = null;
        for (TrendInfo info : trends) {
            if (info.trend == trend) {
                trendInfo = info;
                break;
            }
        }
        if (trendInfo != null) {
            trendInfo.data.forEach(ds -> ds.update());
            plugin.getConsole().setStatusMessage(null, trendInfo.messageID);
            setPlotProperties();
            trendInfo.data.forEach(ds -> ds.replot());
        }
    }
    
    void onTimeWindowChange(Trend trend) {
        isCustomRangeT = false;
        rangeT = null;
        setPlotProperties();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /** Re-plot trends currently displayed by this plotter, taking current properties into account. */
    private void replot() {
        if (trends.size() > 1) {
            trends.forEach(info -> info.data.get(0).update());
            setPlotProperties();
            trends.forEach(info -> info.data.get(0).replot());
        } else if (trends.size() == 1) {
            Trend trend = trends.get(0).trend;
            clearData();
            clearPlotter(true);
            plot(trend, Plotter.OVERLAY);
        }
    }
    
    /** Removes all data from this plotter without resetting properties. */
    private void clearData() {
        trends.forEach(info -> info.trend.setPlotter(null));
        trends.clear();
    }
    
    /** Resets all properties for this plotter to their default values. Does not re-plot.*/
    private void reset() {
        
        TrendingPreferences pref = plugin.getPreferences();
        
        metaOnPoint = null;
        metaOffPoint = EnumSet.noneOf(Trend.Meta.class);
        pref.getDrawMeta().forEach(m -> {
            if (m.isOnPoint()) {
                metaOnPoint = m;
            } else {
                metaOffPoint.add(m);
            }
        });
        
        ignoreMeta = pref.isIgnoreMetaRange();
        
        isCustomRangeT = false;
        isCustomRangeV = false;
        rangeT = null;
        rangeV = null;
    }
    
    private void setPlotProperties() {
        
        JASHistAxis axis = plotter.getPlot().getXAxis();
        double[] r = new double[]{axis.getMin(), axis.getMax()};
        if (isCustomRangeT) {
            if (!isRangeEqual(rangeT, r)) {
                axis.setRange(rangeT[0], rangeT[1]);
            }
        } else {
            if (true || rangeT == null || axis.getRangeAutomatic() || isRangeEqual(rangeT, r)) { // custom T-range is currently disabled
                long begin = Long.MAX_VALUE;
                long end = 0L;
                for (TrendInfo info : trends) {
                    Trend trend = info.trend;
                    long[] range = trend.getTimeRange();
                    if (range[0] < begin) begin = range[0];
                    if (range[1] > end) end = range[1];
                }
                if (end < begin) end = begin = System.currentTimeMillis();
                if (end == begin) {
                    begin -= 1000L;
                    end += 1000L;
                }
                rangeT = new double[]{begin / 1000., end / 1000.};
                axis.setRange(rangeT[0], rangeT[1]);
            } else {
                isCustomRangeT = true;
                rangeT = r;
            }
        }
        
        axis = plotter.getPlot().getYAxis();
        r = new double[]{axis.getMin(), axis.getMax()};
        if (isCustomRangeV) {
            if (!isRangeEqual(rangeV, r)) {
                axis.setRange(rangeV[0], rangeV[1]);
            }
        } else {
            if (rangeV == null || axis.getRangeAutomatic() || isRangeEqual(rangeV, r)) {
                double min = Double.POSITIVE_INFINITY;
                double max = Double.NEGATIVE_INFINITY;
                for (TrendInfo info : trends) {
                    List<Jas3Source> data = info.data;
                    if (!data.isEmpty()) {
                        if (ignoreMeta) {
                            double[] range = data.get(0).getValueRange();
                            if (range[0] < min) min = range[0];
                            if (range[1] > max) max = range[1];
                        } else {
                            for (Jas3Source dataset : data) {
                                double[] range = dataset.getValueRange();
                                if (range[0] < min) min = range[0];
                                if (range[1] > max) max = range[1];
                            }
                        }
                    }
                }
                if (max < min) {
                    min = 0.;
                    max = 1.;
                } else if (min == max) { // check computation, must guarantee the same double value
                    if (min < 0.) {
                        min *= 2.;
                        max = 0.;
                    } else if (min > 0) {
                        min = 0.;
                        max *= 2.;
                    } else {
                        min = -1.;
                        max = 1.;
                    }
                } else {
                    double padding = (max - min) * .05;
                    min -= padding;
                    max += padding;
                }
                rangeV = new double[]{min, max};
                axis.setRange(min, max);
            } else {
                isCustomRangeV = true;
                rangeV = r;
            }
        }
        
        if (trends.size() == 1) {
            plotter.getPlot().setTitle(trends.get(0).trend.getDescriptor().getTitle());
        }
        
    }
    
    private void checkRange() {
        if (!isCustomRangeT && rangeT != null) {
            JASHistAxis a = plotter.getPlot().getXAxis();
            double[] r = new double[]{a.getMin(), a.getMax()};
            if (! (a.getRangeAutomatic() || isRangeEqual(r, rangeT)) ) {
                isCustomRangeT = true;
                rangeT = r;
            }
        }
        if (!isCustomRangeV && rangeV != null) {
            JASHistAxis a = plotter.getPlot().getYAxis();
            double[] r = new double[]{a.getMin(), a.getMax()};
            if (! (a.getRangeAutomatic() || isRangeEqual(r, rangeV)) ) {
                isCustomRangeV = true;
                rangeV = r;
            }
        }
    }
    
    private boolean isRangeEqual(double[] r1, double[] r2) {
        double tolerance = (r1[1] - r1[0])/10.;
        return Math.abs(r1[0] - r2[0]) < tolerance && Math.abs(r1[1] - r2[1]) < tolerance;
    }
    
    /**
     * Clears underlying DefaultPlotter.
     * @param keepProperties If {@code true}, axes ranges and title are kept intact.
     */
    private void clearPlotter(boolean keepProperties) {
        JAS3Plot plot = ((DefaultPlotter)plotter).getPlot();
        if (keepProperties) {
            String title = plot.getTitle();
            JASHistAxis a = plot.getXAxis();
            double tMin = a.getMin();
            double tMax = a.getMax();
            a = plot.getYAxis();
            double vMin = a.getMin();
            double vMax = a.getMax();
            plotter.clear();
            if (! ( title == null || title.equals(plot.getTitle()) ) ) {
                plot.setTitle(title);
            }
            a = plot.getXAxis();
            if (tMin != a.getMin() || tMax != a.getMax()) {
                a.setRange(tMin, tMax);
            }
            a = plot.getYAxis();
            if (vMin != a.getMin() || vMax != a.getMax()) {
                a.setRange(vMin, vMax);
            }
        } else {
            plotter.clear();
        }
        plot.setShowStatistics(false);
        plot.getXAxis().setRangeAutomatic(false);
        plot.getYAxis().setRangeAutomatic(false);
    }
    
    
// -- Class that encapsulates per-trend information kept by the plotter : ------
    
    private class TrendInfo {
        
        Trend trend;
        List<Jas3Source> data;
        int messageID;
        
        TrendInfo(Trend trend, Jas3Source dataset) {
            this.trend = trend;
            data = new ArrayList<>(1);
            data.add(dataset);
        }
        
        TrendInfo(Trend trend) {
            this.trend = trend;
            data = new ArrayList<>(1);
        }
        
        boolean hasOffPoint() {
            return data.size() > 1;
        }
        
        void removeOffPoint() {
            Jas3Source onPoint = data.get(0);
            data = new ArrayList<>(1);
            data.add(onPoint);
        }
    }
    
// -- Jas3 adapter class : -----------------------------------------------------
    
    /** Objects of this class can be plotted directly by the Jas3-provided DefaultPlotter. */
    private class Jas3Source extends Observable implements JAS3DataSource, XYDataSource, HasStyle {
        
        private final int trendIndex;
        private final String title;
        private final String key;
        private long[] t;
        private double[] y, plusError, minusError;
        private final JASHist1DHistogramStyle style = new JASHist1DHistogramStyle();
        
        private double[] vRange;
        
        Jas3Source(int trendIndex, Trend.Meta meta, String key) {
            this.trendIndex = trendIndex;
            this.key = key;
            Trend trend = trends.get(trendIndex).trend;
            Color color;
            if (key == null) {
                title = trend.getDescriptor().getTitle();
                color = colors[trendIndex%(colors.length)];
                style.setShowDataPoints(true);
                style.setShowErrorBars(true);
            } else {
                title = meta.toString();
                color = meta.getColor();
                style.setShowDataPoints(false);
                style.setShowErrorBars(false);
            }
            style.setDataPointColor(color);
            style.setErrorBarColor(color);
            style.setLineColor(color);
            style.setDataPointStyle(JASHist1DHistogramStyle.SYMBOL_DOT);
            style.setShowHistogramBars(false);
            style.setShowLinesBetweenPoints(true);
            populate();
        }
        
        // Implement JAS3DataSource :

        @Override
        public void destroy() {
        }

        @Override
        public void modifyPopupMenu(JPopupMenu jPopupMenu, Component component) {
        }

        @Override
        public DataSource dataSource() {
            return this;
        }

        @Override
        public FTreePath path() {
            return new FTreePath(getTitle());
        }

        @Override
        public String[] axisLabels() {
            return null;
        }

        @Override
        public void setAxisType(int type) {
        }
        
        // Implement XYDataSource :

        @Override
        public int getNPoints() {
            return t == null ? 0 : t.length;
        }

        @Override
        public double getX(int i) {
            return t[i]/1000.;
        }

        @Override
        public double getY(int i) {
            return y[i];
        }

        @Override
        public double getPlusError(int i) {
            return plusError == null ? 0. : plusError[i];
        }

        @Override
        public double getMinusError(int i) {
            return minusError == null ? 0. : minusError[i];
        }

        @Override
        public int getAxisType() {
            return DataSource.DATE;
        }

        @Override
        public String getTitle() {
            return title;
        }
        
        // Implement HasStyle :

        @Override
        public JASHistStyle getStyle() {
            return style;
        }
        
        // Local methods :
        
        private void populate() {
            TrendData td = trends.get(trendIndex).trend.getData();
            if (td == null) {
                t = null;
                y = plusError = minusError = null;
            } else if (key == null) {
                t = td.getTime();
                y = td.getValue();
                if (metaOnPoint == null) {
                    plusError = null;
                    minusError = null;
                } else {
                    List<String> keys = metaOnPoint.getKeys();
                    int n = keys.size();
                    double[][] in = new double[n+1][];
                    in[0] = y;
                    for (int i=0; i<n; i++) {
                        in[i+1] = td.getValue(keys.get(i));
                    }
                    double[][] out = metaOnPoint.getBars(in);
                    minusError = out[0];
                    plusError = out[1];
                }
            } else {
                t = td.getTime(key);
                y = td.getValue(key);
            }
            vRange = null;
        }
        
        private void computeValueRange() {
            int n = getNPoints();
            double min = Double.POSITIVE_INFINITY;
            double max = Double.NEGATIVE_INFINITY;
            if (ignoreMeta) {
                for (int i=0; i<n; i++) {
                    double v = getY(i);
                    if (v < min) min = v;
                    if (v > max) max = v;
                }
            } else {
                for (int i=0; i<n; i++) {
                    double v = getY(i);
                    double d = v - getMinusError(i);
                    if (d < min) min = d;
                    d = v + getPlusError(i);
                    if (d > max) max = d;
                }
            }
            vRange =  new double[] {min, max};
        }
                
        private void update() {
            populate();
            setChanged();
        }
        
        private void replot() {
            notifyObservers(new HistogramUpdate(HistogramUpdate.DATA_UPDATE, false));
        }
        
        private double[] getValueRange() {
            if (vRange == null) {
                computeValueRange();
            }
            return vRange;
        }
        
    }
    
    
// -- Popup menu handler class : -----------------------------------------------
    
    private class Popup implements HasPopupItems {

        @Override
        public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component componentt, Point point) {
            
            menu.insert(new JSeparator(), 0);
            
            JMenu rangeMenu = new JMenu("Axes range");
            JMenuItem it = new JCheckBoxMenuItem("Ignore metadata", ignoreMeta);
            it.addActionListener(e -> {
                JCheckBoxMenuItem box = (JCheckBoxMenuItem) e.getSource();
                boolean isSelected = box.isSelected();
                if (ignoreMeta != isSelected) {
                    ignoreMeta = isSelected;
                    isCustomRangeV = false;
                    rangeV = null;
                    replot();
                }
            });
            rangeMenu.add(it);
            it = new JMenuItem("Reset");
            it.addActionListener(e -> {
                isCustomRangeT = false;
                rangeT = null;
                isCustomRangeV = false;
                rangeV = null;
                replot();
            });
            rangeMenu.add(it);
            menu.insert(rangeMenu, 0);
            
            JMenu metaMenu = new JMenu("Show metadata");
            ButtonGroup bg = new ButtonGroup();
            Trend.Meta.getOnPointSet().forEach(meta -> {
                JRadioButtonMenuItem item = new JRadioButtonMenuItem(meta.toString(), metaOnPoint == meta);
                bg.add(item);
                item.addActionListener(e -> {
                    Trend.Meta selected = Trend.Meta.valueOf(e.getActionCommand());
                    if (selected != metaOnPoint) {
                        metaOnPoint = selected;
                    } else {
                        metaOnPoint = null;
                    }
                    if (trends.size() == 1) {
                        Jas3Source dataset = trends.get(0).data.get(0);
                        dataset.update();
                        setPlotProperties();
                        dataset.replot();
                    } else {
                        trends.forEach(info -> info.data.get(0).update());
                        setPlotProperties();
                        trends.forEach(info -> info.data.get(0).replot());
                    }
                });
                metaMenu.add(item);
            });
            metaMenu.addSeparator();
            Trend.Meta.getOffPointSet().forEach(meta -> {
                JCheckBoxMenuItem item = new JCheckBoxMenuItem(meta.toString(), metaOffPoint.contains(meta));
                item.setEnabled(trends.size() < 2);
                item.addActionListener(e -> {
                    Trend.Meta m = Trend.Meta.valueOf(e.getActionCommand());
                    boolean wasSelected = metaOffPoint.contains(m);
                    JCheckBoxMenuItem mi = (JCheckBoxMenuItem) e.getSource();
                    boolean isSelected = mi.isSelected();
                    if (wasSelected != isSelected) {
                        if (wasSelected) {
                            metaOffPoint.remove(m);
                        } else {
                            metaOffPoint.add(m);
                        }
                        replot();
                    }
                });
                metaMenu.add(item);
            });
            menu.insert(metaMenu, 0);
            
            return menu;
        }

    }
    
    
// -- Style class : ------------------------------------------------------------
    
    public static class Style {
        
        Trend.Meta onPoint;
        EnumSet<Trend.Meta> offPoint;
        
        public Style() {
        }
        
        public Style(Trend.Meta onPoint, EnumSet<Trend.Meta> offPoint) {
            this.onPoint = onPoint;
            this.offPoint = offPoint;
        }

        public Trend.Meta getOnPoint() {
            return onPoint;
        }

        public void setOnPoint(Trend.Meta onnPoint) {
            this.onPoint = onnPoint;
        }

        public EnumSet<Trend.Meta> getOffPoint() {
            return offPoint;
        }

        public void setOffPoint(EnumSet<Trend.Meta> offPoint) {
            this.offPoint = offPoint;
        }

    }
    
    
// -- Saving/Restoring : -------------------------------------------------------
    
    Descriptor save() {
        Descriptor desc = new Descriptor();
        
        if (!trends.isEmpty()) {
            int n = trends.size();
            Trend.Descriptor[] tds = new Trend.Descriptor[n];
            for (int i = 0; i < n; i++) {
                tds[i] = trends.get(i).trend.save();
            }
            desc.setTrends(tds);
        }
        
        if (timeWindow != null) {
            desc.setTimeWindow(timeWindow.toCompressedString());
        }
        
        if (metaOnPoint != null) {
            desc.setOnPointMeta(metaOnPoint.name());
        }
        if (!metaOffPoint.isEmpty()) {
            StringBuilder sb = new StringBuilder();
            metaOffPoint.forEach(m -> sb.append(m.name()).append("+"));
            desc.setOffPointMeta(sb.substring(0, sb.length()-1));
        }
        
        desc.setCustomRangeT(isCustomRangeT);
        desc.setRangeT(isCustomRangeT ? Arrays.copyOf(rangeT, rangeT.length) : null);
        desc.setCustomRangeV(isCustomRangeV);
        desc.setRangeV(isCustomRangeV ? Arrays.copyOf(rangeV, rangeV.length) : null);
                
        return desc;
    }
    
    void restore(Descriptor desc) {
        
        isCustomRangeT = desc.isCustomRangeT();
        rangeT = desc.getRangeT();
        isCustomRangeV = desc.isCustomRangeV();
        rangeV = desc.getRangeV();
        
        String ms = desc.getOnPointMeta();
        if (ms != null) metaOnPoint = Trend.Meta.valueOf(ms);
        ms = desc.getOffPointMeta();
        metaOffPoint = EnumSet.noneOf(Trend.Meta.class);
        if (ms != null) {
            String[] ss = ms.split("\\+");
            for (String s : ss) {
                metaOffPoint.add(Trend.Meta.valueOf(s));
            }
        }
        
        String tw = desc.getTimeWindow();
        timeWindow = tw == null ? plugin.getSelectedTimeWindow() : plugin.getTimeWindowSelector().getTimeWindow(tw);
        
        Trend.Descriptor[] tds = desc.getTrends();
        if (tds != null) {
            ArrayList<Trend> trendList = new ArrayList<>(tds.length);
            for (Trend.Descriptor td : tds) {
                Trend trend = new Trend(td);
                TimeWindow timeWindow = plugin.getTimeWindowSelector().getTimeWindow(td.getTimeWindow());
                trend.setTimeWindow(timeWindow);
                trendList.add(trend);
            }
            if (!trendList.isEmpty()) {
                plot(trendList, Plotter.NORMAL);
                plugin.refresh(getTrends());
                toFront();
            }
        }
    }
    
    public static class Descriptor implements Serializable {

        private Trend.Descriptor[] trends;
        private String timeWindow;
        private String onPointMeta;
        private String offPointMeta;
        private double[] rangeT, rangeV;
        private boolean customRangeT, customRangeV;

        public boolean isCustomRangeV() {
            return customRangeV;
        }

        public void setCustomRangeV(boolean customRangeV) {
            this.customRangeV = customRangeV;
        }

        public boolean isCustomRangeT() {
            return customRangeT;
        }

        public void setCustomRangeT(boolean customRangeT) {
            this.customRangeT = customRangeT;
        }
        
        public double[] getRangeV() {
            return rangeV;
        }

        public void setRangeV(double[] rangeV) {
            this.rangeV = rangeV;
        }

        public double[] getRangeT() {
            return rangeT;
        }

        public void setRangeT(double[] rangeT) {
            this.rangeT = rangeT;
        }

        public String getOffPointMeta() {
            return offPointMeta;
        }

        public void setOffPointMeta(String offPointMeta) {
            this.offPointMeta = offPointMeta;
        }

        public String getOnPointMeta() {
            return onPointMeta;
        }

        public void setOnPointMeta(String onPointMeta) {
            this.onPointMeta = onPointMeta;
        }

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public Trend.Descriptor[] getTrends() {
            return trends;
        }

        public void setTrends(Trend.Descriptor[] trends) {
            this.trends = trends;
        }
        
    }
    
    
}
