package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.util.*;
import javax.swing.JComponent;
import javax.swing.JScrollPane;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.treetable.TreeTableModel;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.gconsole.util.tree.SModel;
import org.lsst.ccs.gconsole.util.tree.SNode;

/**
 * {@link MonitorView} that displays a {@code TreeTable} with columns defined by last segments of display channel paths. 
 *
 * @author onoprien
 */
public class SummaryTableView extends AbstractMonitorView3 implements PersistableMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    static private final MonitorField DEFAULT_FIELD = MonitorField.AVERAGE_VALUE;
    
    private final Descriptor descriptor = new Descriptor();
    
    private volatile AbstractChannelsFilter filter;
    private JScrollPane scrollPane;
    private int nColumns;
    private ArrayList<MonitorField> columnFields;
    private ArrayList<String> columnNames;
    private Model model;
    private Table table;
    private final Set<SNode<Cell[]>> changedNodes = Collections.newSetFromMap(new IdentityHashMap<>());

// -- Life cycle : -------------------------------------------------------------
    
    public SummaryTableView() {
    }

    @Override
    public JComponent getPanel() {
        if (scrollPane == null) {
            scrollPane = new JScrollPane(new MonitorDisplay.EMPTY());
        }
        return scrollPane;
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (!(filter instanceof AbstractChannelsFilter)) throw new IllegalArgumentException("SummaryTableView only accepts AbstractChannelsFilter filters");
        this.filter = (AbstractChannelsFilter) filter;
    }

    @Override
    public void install() {
        ThreadUtil.invokeLater(this::start);
        super.install();
    }

    @Override
    public void uninstall() {
        super.uninstall();
        ThreadUtil.invokeLater(this::stop);
    }
    
    private void start() {
        if (filter == null) throw new IllegalStateException("Trying to initialize SummaryTableView without a filter.");
    }
    
    private void stop() {
        if (model != null) {
            scrollPane.setViewportView(new MonitorDisplay.EMPTY());
            table = null;
            model.destroy();
            model = null;
        }
    }
    
    
// -- Updates : ----------------------------------------------------------------

    @Override
    protected void resetChannels() {
        if (model != null) {
            model.destroy();
        }
        model = new Model();
        table = new Table(model);
        scrollPane.setViewportView(table);
    }

    @Override
    protected void update() {
        changedNodes.forEach(node -> model.nodeChanged(node));
        changedNodes.clear();
    }

    
// -- TreeTableModel : ---------------------------------------------------------
    
    private class Model extends SModel<Cell[]> implements TreeTableModel {
        
        Model() {
            
            // Compile lists of MonitorFields and column names:
            
            columnFields = getGroupFields();
            columnNames = getGroups();
            nColumns = columnNames.size();
            
            // Compile a map for building the tree:
            
            LinkedHashMap<String,Cell[]> path2row = new LinkedHashMap<>();
            data.forEach((displayPath, displayChannel) -> {
                int i = displayPath.lastIndexOf("/");
                String path = displayPath.substring(0, i);
                String columnName = displayPath.substring(i+1);
                int columnIndex = columnNames.indexOf(columnName);
                if (columnIndex != -1) {
                    Cell[] row = path2row.get(path);
                    if (row == null) {
                        row = new Cell[nColumns];
                        for (int col=0; col<nColumns; col++) {
                            row[col] = new Cell(null, col, new ArrayList<>());
                        }
                        path2row.put(path, row);
                    }
                    row[columnIndex].channels.add(displayChannel);
                }
            });
            
            // Build the tree:
            
            update(path2row);
            
            // Wire cells for updates:
            
            SNode<Cell[]> rootNode = this.getRoot();
            if (rootNode != null) {
                Enumeration<SNode<Cell[]>> en = rootNode.postorderEnumeration();
                while (en.hasMoreElements()) {
                    SNode<Cell[]> node = en.nextElement();
                    if (node != rootNode) {
                        Cell[] cells = node.getUserObject();
                        SNode<Cell[]> parentNode = node.getParent();
                        if (parentNode == rootNode) {
                            for (Cell cell : cells) {
                                cell.setNode(node);
                            }
                        } else {
                            Cell[] parentCells = parentNode.getUserObject();
                            if (parentCells == null) {
                                parentCells = new Cell[nColumns];
                                for (int col = 0; col < nColumns; col++) {
                                    Cell cell = cells[col];
                                    parentCells[col] = new Cell(null, col, new ArrayList<>(cell.channels));
                                    cell.setNode(node);
                                }
                                parentNode.setUserObject(parentCells);
                            } else {
                                for (int col = 0; col < nColumns; col++) {
                                    Cell cell = cells[col];
                                    parentCells[col].channels.addAll(cell.channels);
                                    cell.setNode(node);
                                }
                            }
                        }
                    }
                }
            }
        }
        
        void destroy() {
            Enumeration<SNode<Cell[]>> en = getRoot().depthFirstEnumeration();
            while (en.hasMoreElements()) {
                Cell[] cc = en.nextElement().getUserObject();
                if (cc != null) {
                    for (Cell c : cc) {
                        for (DisplayChannel dc : c.getChannels()) {
                            dc.setTarget(null);
                        }
                    }
                }
            }
        }
        
        // Implement TreeTableModel:

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return columnIndex == 0 ? Object.class : FormattedValue.class;
        }

        @Override
        public int getColumnCount() {
            return columnFields.size() + 1;
        }

        @Override
        public String getColumnName(int column) {
            return column == 0 ? "" : columnNames.get(column-1);
        }

        @Override
        public int getHierarchicalColumn() {
            return 0;
        }

        @Override
        public Object getValueAt(Object node, int column) {
            if (column == 0) {
                return node;
            } else {
                SNode<Cell[]> sNode = (SNode<Cell[]>) node;
                Cell[] cells = sNode.getUserObject();
                if (cells == null) {
                    return FormattedValue.EMPTY;
                } else {
                    Cell c = cells[column-1];
                    FormattedValue fv = c.getFormattedValue();
                    if (fv == null) {
                        MonitorFormat.DEFAULT.format(c);
                        fv = c.getFormattedValue();
                    }
                    return fv;
                }
            }
        }

        @Override
        public boolean isCellEditable(Object node, int column) {
            return false;
        }

        @Override
        public void setValueAt(Object value, Object node, int column) {
        }
                
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class Cell implements MonitorCell, Updatable {
        
        private SNode<Cell[]> node;
        private final int column;
        private final ArrayList<DisplayChannel> channels;
        private FormattedValue formattedValue;
        
        Cell(SNode<Cell[]> node, int column, ArrayList<DisplayChannel> channels) {
            this.node = node;
            this.column = column;
            this.channels = channels;
        }

        @Override
        public FormattedValue getFormattedValue() {
            return formattedValue;
        }

        @Override
        public void setFormattedValue(FormattedValue formattedValue) {
            this.formattedValue = formattedValue;
        }

        @Override
        public MonitorField getField() {
            return columnFields == null ? DEFAULT_FIELD : columnFields.get(column);
        }

        @Override
        public List<DisplayChannel> getChannels() {
            return channels;
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            formattedValue = null;
            SNode<Cell[]> parentNode = node.getParent();
            if (parentNode != model.getRoot()) {
                parentNode.getUserObject()[column].update(channelHandle);
            }
            changedNodes.add(node);
        }
        
        void setNode(SNode<Cell[]> node) {
            this.node = node;
            channels.trimToSize();
            if (node.isLeaf()) {
                channels.forEach(dc -> dc.setTarget(this));
            }
        }
    }
    
    /**
     * Customized JXTreeTable (hack around hack around DefaultTableCellRenderer "color memory").
     */
    static private class Table extends JXTreeTable {
        Table(TreeTableModel model) {
            super(model);
            setDefaultRenderer(FormattedValue.class, new MonitorTableCellRenderer());
            setRowSelectionAllowed(true);
            setColumnSelectionAllowed(false);
            setShowGrid(true);
        }
        @Override
        protected void resetDefaultTableCellRendererColors(Component renderer, int row, int column) {
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

}
