package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.concurrent.TimeUnit;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.MutableAgentChannel;
import org.lsst.ccs.gconsole.base.Console;

/**
 * Adapter class that simplifies implementing monitor views.
 * <p>
 *
 * @author onoprien
 */
abstract public class AbstractMonitorView3 implements MonitorView {

// -- Fields : -----------------------------------------------------------------
    
    protected final LinkedHashMap<String,DisplayChannel> data = new LinkedHashMap<>(); // in addition order
    
    private long resetRequested = -2;
    private final long RESET_DELAY = 1000; // reset delay in milliseconds

// -- Life cycle : -------------------------------------------------------------
    
// -- SPI for subclasses (called on EDT) : -------------------------------------
    
    /**
     * Called on EDT whenever channels have been added or removed.
     * Should be implemented to completely rebuild the view based on {@code data} field.
     */
    abstract protected void resetChannels();
    
    /**
     * Called on EDT at the end of each update, after all DisplayChannel.update(...) methods.
     * Empty implementation is provided.
     */
    protected void update() {}

// -- Implement MonitorView : --------------------------------------------------
    
    @Override
    public boolean isEmpty() {
        return data.isEmpty();
    }
    
// -- Interacting with status aggregator : -------------------------------------

    @Override
    final public void configure(AgentStatusEvent event) {
        processEvent(event);
    }

    @Override
    final public void disconnect(AgentStatusEvent event) {
        processEvent(event);
    }

    @Override
    final public void statusChanged(AgentStatusEvent event) {
        processEvent(event);
    }
    
    private void processEvent(AgentStatusEvent event) {
        
        AgentChannelsFilter filter = getFilter();
            
        List<Map.Entry<String, AgentChannel>> addedChannels = new ArrayList<>();
        for (AgentChannel channel : event.getAddedChannels()) {
            for (String displayPath : filter.getDisplayPaths(channel)) {
                addedChannels.add(new HashMap.SimpleEntry<>(displayPath, channel));
            }
        }
        List<Map.Entry<String, AgentChannel>> removedChannels = new ArrayList<>();
        for (AgentChannel channel : event.getRemovedChannels()) {
            for (String displayPath : filter.getDisplayPaths(channel)) {
                removedChannels.add(new HashMap.SimpleEntry<>(displayPath, channel));
            }
        }
        List<Map.Entry<String, AgentChannel>> changedChannels = new ArrayList<>();
        for (AgentChannel channel : event.getStatusChanges().keySet()) {
            for (String displayPath : filter.getDisplayPaths(channel)) {
                changedChannels.add(new HashMap.SimpleEntry<>(displayPath, channel));
            }
        }
        
        if (!(changedChannels.isEmpty() && addedChannels.isEmpty() && removedChannels.isEmpty())) {
            SwingUtilities.invokeLater(() -> updateGUI(event, addedChannels, removedChannels, changedChannels));
        }
    }
    
    private void updateGUI(AgentStatusEvent event, 
            List<Map.Entry<String, AgentChannel>> addedChannels, 
            List<Map.Entry<String, AgentChannel>> removedChannels, 
            List<Map.Entry<String, AgentChannel>> changedChannels) {
        
        boolean needReset = false;
        
        // Removed channels:
        
        for (Map.Entry<String, AgentChannel> e : removedChannels) {
            DisplayChannel dc = data.get(e.getKey());
            if (dc != null) {
                AgentChannel channel = e.getValue();
                boolean mutable = channel instanceof MutableAgentChannel;
                if (mutable) {
                    ((MutableAgentChannel)channel).set(AgentChannel.Key.STATE, DataProviderState.OFF_LINE);
                    dc.addChannel(channel);
                    dc.update(null);
                } else {
                    if (dc.removeChannel(channel.getPath())) {
                        data.remove(dc.getPath());
                        needReset = true;
                    } else {
                        dc.update(null);
                    }
                }
            }
        }
        
        // Added channels:
        
        for (Map.Entry<String, AgentChannel> e : addedChannels) {
            String displayPath = e.getKey();
            AgentChannel channel = e.getValue();
            DisplayChannel dc = data.get(displayPath);
            if (dc == null) {
                dc = new DisplayChannelMulti(displayPath);
                dc.addChannel(channel);
                data.put(displayPath, dc);
                needReset = true;
            } else {
                dc.addChannel(channel);
                dc.update(null);
            }
        }
        
        // Changed channels (at the moment, treated the same way as added):
        
        for (Map.Entry<String, AgentChannel> e : changedChannels) {
            String displayPath = e.getKey();
            AgentChannel channel = e.getValue();
            DisplayChannel dc = data.get(displayPath);
            if (dc == null) {
                dc = new DisplayChannelMulti(displayPath);
                dc.addChannel(channel);
                data.put(displayPath, dc);
                needReset = true;
            } else {
                dc.addChannel(channel);
                dc.update(null);
            }
        }
        
        // Purge previously offline channels if new channels from the same agent have been added:

        if (!addedChannels.isEmpty()) {
            String agent = event.getSource().getName();
            Iterator<Map.Entry<String, DisplayChannel>> it  = data.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, DisplayChannel> e = it.next();
                if (e.getValue().purgeOfflineChannels(agent)) {
                    it.remove();
                    needReset = true;
                }
            }
        }
        
        // Call SPI methods :
        
        if (needReset) {
            scheduleReset();
        } else {
            update();
        }
    }
    
    private void scheduleReset() {
        if (RESET_DELAY == 0L) {
            resetChannels();
        } else if (resetRequested != -2) {
            if (resetRequested == -1) {
                Console.getConsole().getScheduler().schedule(() -> SwingUtilities.invokeLater(this::executeReset), RESET_DELAY, TimeUnit.MILLISECONDS);
            }
            resetRequested = System.currentTimeMillis();
        }
    }
    
    private void executeReset() {
        if (resetRequested != -2) {
            long delay = resetRequested + 900 - System.currentTimeMillis();
            if (delay > 0) {
                Console.getConsole().getScheduler().schedule(() -> SwingUtilities.invokeLater(this::executeReset), delay, TimeUnit.MILLISECONDS);
            } else {
                resetRequested = -1;
                resetChannels();
            }
        }
    }

    @Override
    public void install() {
        if (RESET_DELAY > 0) {
            SwingUtilities.invokeLater(() -> resetRequested = -1);
        }
        MonitorView.super.install();
    }

    @Override
    public void uninstall() {
        MonitorView.super.uninstall();
        if (RESET_DELAY > 0) {
            SwingUtilities.invokeLater(() -> resetRequested = -2);
        }
    }
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Retrieves a list {@code MonitorField} for display channel groups from the filter.
     * The assumption is that the list of corresponding strings is returned by {@code AgentChannelsFilter.getFields(false)}.
     * 
     * @return List of fields for display channel groups, or {@code null} if the filter does not provide the list.
     */
    protected ArrayList<MonitorField> getGroupFields() { // FIXME
        ArrayList<MonitorField> columnFields;
        List<String> fields = getFilter().getFields(false);
        if (fields == null) {
            columnFields = null;
        } else {
            columnFields = new ArrayList<>(fields.size());
            for (String f : fields) {
                columnFields.add(stringToField(f));
            }
        }
        return columnFields;
    }
    
    protected ArrayList<String> getGroups() { // FIXME: need to produce the list for filters that do not havce it ready.
//        ArrayList<String> columnNames;
        
        List<String> names = getFilter().getFields(true); // group names
//        LinkedHashMap<String,Boolean> gg = new LinkedHashMap<>();
//        if (names != null) {
//            names.forEach(g -> gg.put(g, false));
//        }
//        data.keySet().forEach(displayPath -> {
//            if (!columnNames.contains(displayPath)) {
//                columnNames.add(displayPath);
//            }
//        });       
//        
//        if (names == null) {
//            columnNames = new ArrayList<>();
//            data.keySet().forEach(displayPath -> {
//                if (!columnNames.contains(displayPath)) {
//                    columnNames.add(displayPath);
//                }
//            });
//        } else {
//            columnNames = new ArrayList<>(names);
//        }
        return new ArrayList<>(names);
    }
    
    protected MonitorField stringToField(String field) {
        switch (field) {
            case "Average":
                return MonitorField.AVERAGE_VALUE;
            case "Max":
                return MonitorField.MAX_VALUE;
            case "Min":
                return MonitorField.MIN_VALUE;
            default:
                return MonitorField.AVERAGE_VALUE;
        }
    }
    
    protected String fieldToString(MonitorField field) {
        return field.toString();
    }

}
