package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.io.Serializable;
import java.util.Arrays;
import java.util.concurrent.CancellationException;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.Box;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 * Graphic panel that provides command dictionary interface to a specific subsystem.
 *
 * @author onoprien
 */
public final class AgentPanel extends JSplitPane implements Savable, ChangeListener {

// -- Fields : -----------------------------------------------------------------
    
    private AgentHandle agent;
    
    private final DictionaryTreePanel dictionaryTree;
    private final CommandListPanel commandListPanel;
    private final ArgInputPanel argInputPanel;
    private final HTMLTextPane resultPane;
    
    private final CommandSender sender;
    
    Descriptor descriptor;

// -- Life cycle : -------------------------------------------------------------
    
    public AgentPanel() {
        
        super(VERTICAL_SPLIT);
        setContinuousLayout(true);
        descriptor = new Descriptor();

        JPanel topPane = new JPanel(new GridLayout(1, 3));
        setTopComponent(topPane);
        resultPane = new HTMLTextPane();
        setBottomComponent(new JScrollPane(resultPane));
        
        dictionaryTree = new DictionaryTreePanel(this);
        topPane.add(new JScrollPane(dictionaryTree));

        commandListPanel = new CommandListPanel(this);
        topPane.add(new JScrollPane(commandListPanel));
        dictionaryTree.addTreeSelectionListener(commandListPanel);

        JPanel topPaneRightPanel = new JPanel(new BorderLayout());
        topPane.add(topPaneRightPanel);        
        
        argInputPanel = new ArgInputPanel();
        topPaneRightPanel.add(argInputPanel, BorderLayout.CENTER);
        commandListPanel.addListSelectionListener(argInputPanel);

        Box buttonPane = Box.createHorizontalBox();
        topPaneRightPanel.add(buttonPane, BorderLayout.SOUTH);
        
        JCheckBox cb = new JCheckBox("Hide locked");
        buttonPane.add(cb);
        cb.setToolTipText("Hide commands unavailable due to agent lock state");
        cb.setSelected(descriptor.hideUnavailableCommands);
        cb.addActionListener(e -> {
            descriptor.setHideUnavailableCommands(cb.isSelected());
            dictionaryTree.updateData(agent);
        });
        
        buttonPane.add(Box.createRigidArea(Const.HDIM));
        buttonPane.add(Box.createHorizontalGlue());
        
        JButton sendCmdButton = new JButton(new ImageIcon(this.getClass().getResource("ic_send_black_24dp.png")));
        buttonPane.add(sendCmdButton);
        sendCmdButton.setText("Send");
        sendCmdButton.setEnabled(false);
        sendCmdButton.addActionListener(e -> sendCommand());

        commandListPanel.addListSelectionListener(e -> {
            boolean en = commandListPanel.isVisibleCommandSelected();
            sendCmdButton.setEnabled(en);
            argInputPanel.setEnabled(en);
        });
        
        sender = Console.getConsole().getSingleton(CommandService.class).getSender();
        sender.setCommandHandle(new BrowserCommandHandle());
    }

    
// -- Receiving external events : ----------------------------------------------
    
    void setAgent(AgentHandle agent) {
        if (this.agent != null) {
            this.agent.removeListener(this);
        }
        this.agent = agent;
        if (agent != null) {
            if (agent.getDictionaries().isEmpty()) {
                agent.updateDictionaries();
            }
            agent.addListener(this);
        }
        dictionaryTree.updateData(agent);
    }

    /**
     * Called in response to changes in agent data.
     * @param e Event
     */
    @Override
    public void stateChanged(ChangeEvent e) {
        if (agent == null || !agent.isAdjusting()) {
            dictionaryTree.updateData(agent);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------

    private void sendCommand() {
        TreeBranch<ComponentDictionary> destinationNode = (TreeBranch<ComponentDictionary>) dictionaryTree.getLastSelectedPathComponent();
        ComponentDictionary commandContext = destinationNode.getContent();
        DictionaryCommand cmd = commandListPanel.getSelectedValue();
        if (cmd != null) {
            String s = argInputPanel.getFormattedArgumentValues();
            Object[] args = argInputPanel.getFormattedValues();
            String commandName = cmd.getCommandName();
            String path = commandContext.path;
            String command = path +"/"+ commandName;
            resultPane.insertHTML("<b>> " + path +" "+ commandName +" "+ s +" ...</b>");
            sender.execute(command, args);
        }
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class BrowserCommandHandle implements CommandHandle {

        @Override
        public void onSendingFailure(Throwable t, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            if (t == null) {
                sb.append("Cannot send command.");
            } else {
                sb.append(shorten(t.getMessage()));
            }
            resultPane.insertHTML(sb.append("</font>").toString());
        }

        @Override
        public void onAck(CommandAck ack, CommandTask source) {
        }

        @Override
        public void onNack(CommandNack nack, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            sb.append("Command rejected");
            if (nack != null) {
                String message = shorten(nack.getReason());
                if (message != null && !message.isEmpty()) {
                    sb.append(": ").append(message);
                }
            }
            resultPane.insertHTML(sb.append("</font>").toString());
        }

        @Override
        public void onSuccess(Object res, CommandTask source) {
            if (res == null) {
                resultPane.insertHTML("Done.");
            } else {
                if (res.getClass().isArray()) {
                    try {
                        res = Arrays.deepToString((Object[]) res);
                    } catch (ClassCastException x) {
                    }
                }
                resultPane.insertHTML(res.toString().replaceAll("(\\r\\n|\\n)", "<br />"));
            }
        }

        @Override
        public void onExecutionFailure(Throwable t, CommandTask source) {
            StringBuilder sb = new StringBuilder("<font color=\"#FF0000\">");
            if (t == null) {
                sb.append("Failed.");
            } else {
                sb.append(shorten(t.getMessage()));
            }
            resultPane.insertHTML(sb.append("</font>").toString());
        }

        @Override
        public void onTimeout(TimeoutException x, CommandTask source) {
            resultPane.insertHTML("<font color=\"#FF0000\">"+ shorten(x.getMessage()) +"</font>");
        }

        @Override
        public void onCancel(CancellationException exception, CommandTask source) {
            resultPane.insertHTML("Cancelled.");
        }

        @Override
        public void onResult(Object res, CommandTask source) {
            resultPane.insertHTML("Result: "+ (res == null ? "" : res)); // should not execute
        }
        
    }
    
// -- Saving/restoring : -------------------------------------------------------    
    
    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
        } else if (descriptor == null) {
            this.descriptor = new Descriptor();
        }
    }

    @Override
    public Descriptor save() {
        return descriptor;
    }
    
    static public class Descriptor implements Serializable {

        private boolean hideUnavailableCommands;

        public boolean isHideUnavailableCommands() {
            return hideUnavailableCommands;
        }

        public void setHideUnavailableCommands(boolean hideUnavailableCommands) {
            this.hideUnavailableCommands = hideUnavailableCommands;
        }
        
    }
    
}
