package org.lsst.ccs.gconsole.util.tree;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JTree;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * 
 * @param <T> Type of objects that can be associated with nodes of this tree.
 *
 * @author onoprien
 */
public class STree<T> extends JTree implements Savable {

// -- Fields : -----------------------------------------------------------------

// -- Life cycle : -------------------------------------------------------------
    
    public STree() {
        setRootVisible(false);
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
    }
    
    public STree(SModel<T> model) {
        super(model);
        setRootVisible(false);
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
    }
    
    
// -- Overriding JTree : -------------------------------------------------------

    @Override
    public SNode<T> getLastSelectedPathComponent() {
        return (SNode<T>) super.getLastSelectedPathComponent();
    }

    @Override
    public SModel<T> getModel() {
        return (SModel<T>) super.getModel();
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    public void expandLevel(int level) {
        SNode<T> root = getModel().getRoot();
        if (root != null) {
            expandLevel(level-1, 0, root, new TreePath(root));
        }
    }
    
    private boolean expandLevel(int targetLevel, int currentLevel, SNode<T> node, TreePath path) {
        if (targetLevel == currentLevel) {
            if (node.isLeaf()) {
                return true;
            } else {
                expandPath(path);
            }
        } else {
            Enumeration e = node.children();
            if (e.hasMoreElements()) {
                currentLevel++;
                while (e.hasMoreElements()) {
                    SNode<T> child = (SNode<T>) e.nextElement();
                    boolean expandParent = expandLevel(targetLevel, currentLevel, child, path.pathByAddingChild(child));
                    if (expandParent) {
                        expandPath(path);
                        break;
                    }
                }
            } else {
                return true;
            }
        }
        return false;
    }
    
    
// -- Getters/Setters : --------------------------------------------------------
    
    public T getSelectedUserObject() {
        SNode<T> node = getLastSelectedPathComponent();
        return node == null ? null : node.getUserObject();
    }

    public void setSelectedPath(String path) {
        if (path == null) {
            setSelectionPath(null);
        } else {
            SModel<T> model = getModel();
            if (model != null) {
                setSelectionPath(model.getPath(path));
            }
        }
    }
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor implements Serializable {

        private String[] expansionState;

        public String[] getExpansionState() {
            return expansionState;
        }

        public void setExpansionState(String[] expansionState) {
            this.expansionState = expansionState;
        }
        
    }

    @Override
    public void restore(Serializable descriptor) {
        if (!(descriptor instanceof Descriptor)) return;
        Descriptor desc = (Descriptor) descriptor;
        String[] expansionState = desc.getExpansionState();
        for (String path : expansionState) {
            TreePath tp =  getModel().getPath(path);
            if (tp != null) expandPath(tp);
        }
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor();
        
        ArrayList<TreePath> expandedPaths = TreeUtil.saveExpansionState(this);
        String[] expansionState = expandedPaths.stream()
                .map(p -> ((SNode)(p.getLastPathComponent())).getPathAsString())
                .collect(Collectors.toList())
                .toArray(new String[expandedPaths.size()]);
        desc.setExpansionState(expansionState);
        
        return desc;
    }
 
}
