package org.lsst.ccs.gconsole.services.aggregator;

import java.util.*;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;

/**
 * 
 *
 * @author onoprien
 */
public class BasicChannel implements MutableAgentChannel {

// -- Fields : -----------------------------------------------------------------
    
    private final String path;
    private final AgentInfo agent;
    
    protected Object value;
    protected HashMap<String,Object> data;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs an instance.
     * @param path Unique path identifying this channel.
     * @param agent Agent this channel belongs to.
     */
    protected BasicChannel(String path, AgentInfo agent) {
        this.path = path;
        this.agent = agent;
    }
    
    /**
     * Copy constructor.
     * @param other Channel to copy.
     */
    public BasicChannel(BasicChannel other) {
        synchronized (other) {
            this.path = other.path;
            this.agent = other.agent;
            this.value = other.value;
            this.data = new HashMap<>(other.data);
        }
    }
    
    /**
     * Copy constructor.
     * @param other Channel to copy.
     */
    public BasicChannel(AgentChannel other) {
        synchronized (other) {
            this.path = other.getPath();
            this.agent = other.getAgent();
            this.value = other.get();
            this.data = new HashMap<>();
            other.getAttributes().forEach(key -> data.put(key, other.get(key)));
        }
    }
    
// -- Getters : ----------------------------------------------------------------

    /**
     * Return the unique path of this channel.
     * @return Channel path in (agent name)/(local path) form.
     */
    @Override
    public String getPath() {
        return path;
    }
    
    /**
     * Returns the name of the {@code Agent} this channel belongs to.
     * @return Agent name.
     */
    @Override
    public String getAgentName() {
        return agent.getName();
    }
    
    /**
     * Returns the descriptor of the {@code Agent} this channel belongs to. 
     * @return Agent descriptor.
     */
    @Override
    public AgentInfo getAgent() {
        return agent;
    }
    
    /**
     * Returns the path of this channel with the agent name and the following slash stripped.
     * Local paths are unique inside an agent.
     * @return Local path.
     */
    @Override
    public String getLocalPath() {
        return path.substring(path.indexOf("/")+1);
    }
    
    /**
     * Returns the value of the channel attribute identified by the {@code key}.
     * 
     * @param key Attribute key. If {@code VALUE} or {@code null}, the central value is returned.
     * @return Attribute value, or {@code null} if this channel does not have the specified attribute.
     */
    @Override
    @SuppressWarnings("unchecked")
    synchronized public <T> T get(Object key) {
        if (key == null) return get();
        String sKey = key.toString();
        if (Key.VALUE.equals(sKey)) return get();
        return data == null ? null : (T) data.get(sKey);
    }
    
    /**
     * Returns the current central value of this channel.
     * @return Current value.
     */
    @Override
    @SuppressWarnings("unchecked")
    synchronized public <T> T get() {
        return (T) value;
    }

    @Override
    synchronized public List<String> getAttributes() {
        return data == null ? Collections.emptyList() : new ArrayList<>(data.keySet());
    }
    

// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the value of the specified attribute.
     * The attribute value does not change if the new value is equal to the old one.
     * 
     * @param key The attribute is identified by the string returned by {@code toString()} method of the key object.
     *            If {@code key} is {@code null} or evaluates to {@link AgentChannel.Key#VALUE}, the central value is set.
     * @param value The new value.
     * @return {@code true} if the attribute value has changed as a result of this call.
     */
    @Override
    public synchronized boolean set(Object key, Object value) {
        if (key == null) return set(value);
        String sKey = key.toString();
        if (Key.VALUE.equals(sKey)) return set(value);
        if (data == null) {
            if (value == null) {
                return false;
            } else {
                data = new HashMap<>(4);
                data.put(sKey, value);
                return true;
            }
        } else {
            if (value == null) {
                Object oldValue = data.remove(sKey);
                if (data.isEmpty()) data = null;
                return oldValue != null;
            } else {
                Object oldValue = data.put(sKey, value);
                if (oldValue instanceof ConfigurationParameterInfo && !(value instanceof ConfigurationParameterInfo)) { // protect against replacing configurable with metadata
                    data.put(sKey, oldValue);
                    return false;
                }
                return !value.equals(oldValue);
            }
        }
    }
    
    /**
     * Sets the value of this channel.
     * The value will not change if the new value is equal to the old one.
     * 
     * @param value New value.
     * @return {@code true} if the value has changed as a result of this call.
     */
    @Override
    public synchronized boolean set(Object value) {
        boolean out = !Objects.equals(this.value, value);
        if (out) this.value = value;
        return out;
    }
    
    
// -- Overriding Object : ------------------------------------------------------

    @Override
    public String toString() {
        return path;
    }

    @Override
    public boolean equals(Object obj) {
        return path.equals(obj) || (obj instanceof AgentChannel && path.equals(((AgentChannel)obj).getPath()));
    }

    @Override
    public int hashCode() {
        return path.hashCode();
    }

}
