package org.lsst.ccs.gconsole.plugins.tracer;

import java.awt.Color;
import java.awt.Component;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.file.Path;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.gconsole.base.InstanceDialog;
import org.lsst.ccs.gconsole.plugins.tracer.filters.AlertSelector;
import org.lsst.ccs.gconsole.plugins.tracer.filters.DataDictionary;
import org.lsst.ccs.gconsole.plugins.tracer.filters.HeartBeatSelector;
import org.lsst.ccs.gconsole.plugins.tracer.filters.LegacyFormatter;
import org.lsst.ccs.gconsole.plugins.tracer.filters.StandardMessageFilter;
import org.openide.util.Lookup;
import org.openide.util.LookupListener;

/**
 * Maintains a collection of named message filters.
 * <p>
 * Filters come from four possible sources. Built-In filters are created by the graphical
 * console itself. Local filters are created and registered with the application lookup 
 * by the console plugins. User filters are defined by the user and stored in the 
 * "tracer/filters.xml" file in the graphical console home directory. Finally, additional
 * filters might be provided by other subsystem connected to the buses <i>(The mechanism
 * for doing this has yet to be finalized)</i>.
 * <p>
 * Each filter has a short name and a (possibly the same) full name. The full name is the
 * short name prefixed by {@code BUILTIN}, {@code LOCAL}, {@code USER}, or nothing,
 * depending on the filter origin.
 * The <tt>getName()</tt> method of any filter object returns its full name. All public 
 * methods of this class that take filter names as parameters also require full names.
 * Filters and filter factories registered with the lookup have their names modified to
 * add the {@code LOCAL} prefix.
 *
 * @author onoprien
 */
public class FilterRegistry {

// -- Fields : -----------------------------------------------------------------
    
    static final String USER = "User/";
    
    private final LsstTracerPlugin plugin;
    private final Map<String,MessageFilterFactory> filters = new ConcurrentHashMap<>(4, .75f, 1);
    private Lookup.Result result1;
    private final List<String> builtinFilterClasses = Arrays.asList(
            "org.lsst.ccs.gconsole.plugins.tracer.filters.StandardMessageFilter"
    );


// -- Construction and initialization : ----------------------------------------
    
    FilterRegistry(LsstTracerPlugin plugin) {
        this.plugin = plugin;
    }
    
    void init() {
        
        addFactory(new LegacyFormatter());
        addFactory(new HeartBeatSelector());
        addFactory(new AlertSelector());
        addFactory(new StandardMessageFilter.Default());
        addFactory(new StandardMessageFilter.All());
        addFactory(new DataDictionary());
        
        FreeHEPLookup lookup = plugin.getConsole().getConsoleLookup();
        LookupListener listener = e -> updateLocalFilters();
        Lookup.Template template = new Lookup.Template(MessageFilterFactory.class);
        result1 = lookup.lookup(template);
        result1.addLookupListener(listener);
        updateLocalFilters();
        
        addUserFilters(getUserFiltersFile());
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Returns the filter specified by the name.
     * 
     * @param name Full name of the required filter.
     * @return The filter specified by the name, or {@code null} if there is no such filter.
     * @throws IllegalArgumentException if the filter cannot be found or created.
     */
    public MessageFilter getFilter(String name) {
        MessageFilterFactory factory = filters.get(name);
        if (factory == null) throw new IllegalArgumentException("Unrecognized filter: "+ name);
        try {
            MessageFilter filter = factory.get(null);
            if (filter == null) throw new IllegalArgumentException("Unable to retrieve filter: "+ name);
            return filter;
        } catch (Exception x) {
            throw new IllegalArgumentException("Unable to retrieve filter: "+ name, x);
        }
    }
    
    /**
     * Saves user-defined filter to local storage.
     *
     * @param filter Filter to be saved.
     * @throws IOException if the filter cannot be saved.
     */
    public void saveFilter(UserFilter filter) throws IOException {
        try {
            
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document;
            try {
                document = reader.read(inputFile);
            } catch (DocumentException x) {
                document = DocumentHelper.createDocument();
                document.addElement( "filters" );
            }
            
            String name = filter.getPath();
            if (name.startsWith(USER)) {
                name = name.substring(USER.length(), name.length());
            }
            Element filtersElement = (Element) document.selectSingleNode("/filters");
            Element filterElement = (Element) filtersElement.selectSingleNode("filter[@name='"+ name +"']");
            if (filterElement != null) {
                filtersElement.remove(filterElement);
            }
            filterElement = filtersElement.addElement("filter");
            filterElement.addAttribute("name", name);
            String description = filter.getDescription();
            if (description != null) {
                filterElement.addElement("description").addText(description);
            }
            Element stepsElement = filterElement.addElement("steps");
            int nSteps = filter.filters.length;
            for (int step=0; step<nSteps; step++) {
                TracerFilter tf = filter.filters[step];
                Element stepElement = stepsElement.addElement("step");
                if (filter.or[step]) stepElement.addElement("or");
                stepElement.addElement("mode").addText(tf.getMode().name());
                if (tf.isInverted()) stepElement.addElement("invert");
                if (tf.isFormatting()) stepElement.addElement("format");
                stepElement.addElement("target").addText(tf.getTarget().name());
                stepElement.addElement("method").addText(tf.getMethod().name());
                stepElement.addElement("code").addText(tf.getCode());
                Color color = tf.getColor();
                if (color != null) {
                  stepElement.addElement("color").addText(encode(color));
                }
                FilteredMessage.Flag action = tf.getAction();
                if (action != null) {
                    stepElement.addElement("action").addText(action.name());
                }
            }
            
            OutputFormat format = OutputFormat.createPrettyPrint();
            if (!inputFile.exists()) {
                inputFile.getParentFile().mkdirs();
            }
            XMLWriter writer = new XMLWriter(new FileOutputStream(inputFile), format);
            writer.write(document);
            
            addFactory(new UserFilterFactory(name, description));
            
        } catch (NullPointerException|ClassCastException|IndexOutOfBoundsException|
                 IllegalArgumentException|IOException x) {
            throw new IOException("Unable to save filter "+ filter.getPath(), x);
        }
    }
    
    /**
     * Deletes user-defined filter from local storage.
     * Calling this method with a name of a non-existing filter has no effect.
     * 
     * @param name Full name of the filter to be deleted.
     * @throws IOException if the filter cannot be deleted.
     */
    public void deleteFilter(String name) throws IOException {
        
        if (!name.startsWith(USER)) return;
            
        filters.remove(name);
        name = name.substring(USER.length(), name.length());
        
        try {
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document = reader.read(inputFile);
            Element filtersElement = (Element) document.selectSingleNode("/filters");
            Element filterElement = (Element) filtersElement.selectSingleNode("filter[@name='"+ name +"']");
            if (filterElement != null) {
                filtersElement.remove(filterElement);
                OutputFormat format = OutputFormat.createPrettyPrint();
                XMLWriter writer = new XMLWriter(new FileOutputStream(inputFile), format);
                writer.write(document);
            }
        } catch (DocumentException|NullPointerException|ClassCastException|IllegalArgumentException|IOException x) {
            throw new IOException("Unable to delete filter "+ USER + name, x);
        }
    }
    
    
// -- Creating filters : -------------------------------------------------------
    
    public MessageFilter selectFilter(Component parent) {
        List<Object> factories = new ArrayList<>(filters.size() + builtinFilterClasses.size());
        factories.addAll(filters.values());
        factories.addAll(builtinFilterClasses);
        InstanceDialog d = InstanceDialog.show(MessageFilter.class, parent, "Select message filter", factories);
        MessageFilter filter = (MessageFilter) d.getInstance();
        if (filter != null) {
            filter.setParameters(d.getDescriptor().getParameters());
        }
        return filter;
    }
    
    public String selectFilterCode(Component parent) {
        MessageFilter filter = selectFilter(parent);
        if (filter == null) {
            return null;
        } else {
            InstanceDialog.Descriptor descriptor = new InstanceDialog.Descriptor(filter.getPath(), filter.getParameters());
            return descriptor.toString();
        }
    }
    
    public MessageFilter createFilter(String path, String[] parameters) {
        if (path == null) return null;
        MessageFilterFactory factory = filters.get(path);
        if (factory == null) {
            InstanceDialog.Descriptor descriptor = new InstanceDialog.Descriptor();
            descriptor.setPath(path);
            descriptor.setParameters(parameters);
            List<Object> factories = new ArrayList<>(filters.size() + builtinFilterClasses.size());
            factories.addAll(filters.values());
            factories.addAll(builtinFilterClasses);
            MessageFilter filter = InstanceDialog.getInstance(descriptor, MessageFilter.class, factories);
            if (filter != null) {
                filter.setParameters(parameters);
            }
            return filter;
        } else {
            return factory.get(parameters);
        }
    }
    
    public MessageFilter createFilter(InstanceDialog.Descriptor descriptor) {
        String path = descriptor.getPath();
        if (path == null) return null;
        MessageFilterFactory factory = filters.get(path);
        if (factory == null) {
            List<Object> factories = new ArrayList<>(filters.size() + builtinFilterClasses.size());
            factories.addAll(filters.values());
            factories.addAll(builtinFilterClasses);
            MessageFilter filter = InstanceDialog.getInstance(descriptor, MessageFilter.class, factories);
            if (filter != null) {
                filter.setParameters(descriptor.getParameters());
            }
            return filter;
        } else {
            return factory.get(descriptor.getParameters());
        }
    }
    
    public MessageFilter createFilter(String code) {
        return createFilter(new InstanceDialog.Descriptor(code));
    }


// -- Local methods : ----------------------------------------------------------
    
    void addFactory(MessageFilterFactory factory) {
        filters.put(factory.getPath(), factory);
    }
    
    private UserFilter openFilter(String shortName) {
        try {
            File inputFile = getUserFiltersFile();            
            SAXReader reader = new SAXReader();
            Document document = reader.read(inputFile);
            Node filterNode = document.selectSingleNode("/filters/filter[@name='"+ shortName +"']");
            List<Node> stepNodes = filterNode.selectNodes("steps/step");
            int nSteps = stepNodes.size();
            TracerFilter[] steps = new TracerFilter[nSteps];
            boolean[] ors = new boolean[nSteps];
            for (int step=0; step<nSteps; step++) {
                Node stepNode = stepNodes.get(step);
                ors[step] = stepNode.selectSingleNode("or") != null;
                String s = stepNode.selectSingleNode("mode").getText().trim();
                TracerFilter.Mode mode = TracerFilter.Mode.valueOf(s);
                boolean invert = stepNode.selectSingleNode("invert") != null;
                boolean format = stepNode.selectSingleNode("format") != null;
                s = stepNode.selectSingleNode("target").getText().trim();
                TracerFilter.Target target = TracerFilter.Target.valueOf(s);
                s = stepNode.selectSingleNode("method").getText().trim();
                TracerFilter.Method method = TracerFilter.Method.valueOf(s);
                String code = stepNode.selectSingleNode("code").getText();
                Node node = stepNode.selectSingleNode("color");
                Color color = null;
                if (node != null) {
                    s = node.getText().trim();
                    if (!s.isEmpty()) color = Color.decode("#"+ s);
                }
                node = stepNode.selectSingleNode("action");
                FilteredMessage.Flag action = null;
                if (node != null) {
                    s = node.getText().trim();
                    if (!s.isEmpty()) action = FilteredMessage.Flag.valueOf(s);
                }
                steps[step] = new TracerFilter(mode, invert, format, target, method, code, color, action, this);
            }
            return new UserFilter(USER + shortName, steps, ors);
        } catch (DocumentException|NullPointerException|ClassCastException|IndexOutOfBoundsException|IllegalArgumentException x) {
            throw new IllegalArgumentException("Unable to read filter "+ shortName +" from storage", x);
        }
    }
    
    private void updateLocalFilters() {
        filters.entrySet().removeIf(e -> e.getKey().startsWith("LOCAL"));  // FIXME
        Collection c = result1.allInstances();
        c.forEach(o -> {
            addFactory((MessageFilterFactory)o);
        });
    }
    
    private File getUserFiltersFile() {
        Path consoleHome = plugin.getConsole().getHomeDirectory();
        if (consoleHome == null) return null;
        return consoleHome.resolve("tracer/filters.xml").toFile();
    }
    
    private void addUserFilters(File file) {
        if (file == null) return;
        try {
            SAXReader reader = new SAXReader();
            Document document = reader.read(file);
            List<Node> nodes = document.selectNodes("/filters/filter");
            nodes.forEach(node -> {
                String name = node.valueOf("@name");
                Node descNode = node.selectSingleNode("description");
                String description = descNode == null ? null : descNode.getText().trim();
                addFactory(new UserFilterFactory(name, description));
            });
        } catch (DocumentException|NullPointerException x) {
//            plugin.getApplication().error("Error reading filters", x);
        }
    }
    
    static final String encode(Color color) {
        return String.format("%06x", color.getRGB() & 0x00FFFFFF);
    }


// -- Local classes : ----------------------------------------------------------

    private class UserFilterFactory implements MessageFilterFactory {
        
        private final String name;
        private final String desc;
        
        private UserFilterFactory(String shortName, String description) {
            name = shortName;
            desc = description;
        }

        @Override
        public String getPath() {
            return USER + name;
        }

        @Override
        public String getDescription() {
            return desc;
        }

        @Override
        public MessageFilter get(String[] parameters) {
            return openFilter(name);
        }
        
        public MessageFilter getInstance() {
            return openFilter(name);
        }

    }

}


