package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.util.concurrent.CancellationException;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;

/**
 * Dialog to select monitoring data view type and filter.
 * <p>
 * Currently implemented to look at these sources:<ul>
 * <li>Objects of type {@code AgentChannelsFilter} in console lookup.
 * <li>Strings with ID "AgentChannelsFilter" and "MonitorView" in console lookup.
 </ul>
 * @author onoprien
 */
final class MonitorPageDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private final String defaultName = "Monitor";
    private boolean nameNotEdited = true;
    private final PersistenceService service = Console.getConsole().getSingleton(PersistenceService.class);
    
    private PersistableMonitorView view;
    private PersistableAgentChannelsFilter filter;
    private MonitorPage.Descriptor descriptor;
    private MonitorPage page;
    
    private JTextField nameField, viewField, filterField;

// -- Life cycle : -------------------------------------------------------------
    
    private MonitorPageDialog(MonitorPage.Descriptor before, String title, Component parent) {
        super(parent == null ? null : SwingUtilities.getWindowAncestor(parent), 
                title == null ? "Open new monitoring page" : title,
                Dialog.ModalityType.APPLICATION_MODAL);
        descriptor = before;
        GridBagLayout gridbag = new GridBagLayout();
        setLayout(gridbag);
        
        GridBagConstraints c = new GridBagConstraints();
        c.insets = new Insets(3, 8, 3, 8);
        
        c.anchor = GridBagConstraints.EAST;
        
        c.gridx = 0;
        c.gridy = 0;
        JLabel label = new JLabel("Page name:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 1;
        label = new JLabel("View type:");
        gridbag.setConstraints(label, c);
        add(label);
        c.gridy = 2;
        label = new JLabel("Filter:");
        gridbag.setConstraints(label, c);
        add(label);
        
        c.fill = GridBagConstraints.HORIZONTAL;
        c.weightx = 1.0;
        
        c.gridx = 1;
        c.gridy = 0;
        nameField = new JTextField(defaultName, 50);
        nameField.addCaretListener(e -> {
            if (nameNotEdited) nameNotEdited = false;
        });
        gridbag.setConstraints(nameField, c);
        add(nameField);
        
        c.gridy = 1;
        viewField = new JTextField("Default view", 50);
        viewField.setEditable(false);
        gridbag.setConstraints(viewField, c);
        add(viewField);
        
        c.gridy = 2;
        filterField = new JTextField("Unfiltered", 50);
        filterField.setEditable(false);
        gridbag.setConstraints(filterField, c);
        add(filterField);
        
        c.fill = GridBagConstraints.NONE;
        c.weightx = 0.;
        
        c.gridx = 2;
        c.gridy = 1;
        JButton button = new JButton("...");
        button.setToolTipText("Select view type");
        button.addActionListener(e -> {
            PersistableMonitorView v = (PersistableMonitorView) service.make(descriptor.getView(), title, parent, MonitorView.CATEGORY);
            if (v != null) {
                view = v;
                String name = view.getName();
                if (name == null || name.trim().isEmpty()) {
                    Class<?> viewClass = view.getClass();
                    ConsoleLookup cl = viewClass.getAnnotation(ConsoleLookup.class);
                    if (cl != null) name = cl.name();
                    if (name == null || name.isEmpty()) name = viewClass.getSimpleName();
                } else if (nameNotEdited && nameField.getText().equals(defaultName)) {
                    nameField.setText(name);
                }
                viewField.setText(name);
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridy = 2;
        button = new JButton("...");
        button.setToolTipText("Select filter");
        button.addActionListener(e -> {
            PersistenceService serv = Console.getConsole().getSingleton(PersistenceService.class);
            filter = (AbstractChannelsFilter) serv.make(null, "Select filter", MonitorPageDialog.this, AgentChannelsFilter.CATEGORY);
            if (filter == null) {
                filterField.setText("Unfiltered");
            } else {
                String name = filter.getName();
                if (name == null || name.isEmpty()) {
                    filterField.setText(filter.getClass().getSimpleName());
                } else {
                    filterField.setText(name);
                    if (nameNotEdited) {
                        nameField.setText(name);
                    }
                }
            }
        });
        gridbag.setConstraints(button, c);
        add(button);
        
        c.gridx = 0;
        c.gridy = 3;
        c.anchor = GridBagConstraints.SOUTHEAST;
        c.gridwidth = GridBagConstraints.REMAINDER;
        c.weighty = 1.;
        c.insets = new Insets(16, 8, 4, 8);
        
        JPanel box = new JPanel();
        box.setLayout(new GridLayout(1,2,16,5));
        button = new JButton("Cancel");
        button.addActionListener(e -> {
            view = null;
            filter = null;
            page = null;
            dispose();
            throw new CancellationException();
        });
        box.add(button);
        button = new JButton("OK");
        button.addActionListener(e -> {
            if (done()) {
                dispose();
            }
        });
        box.add(button);
        gridbag.setConstraints(box, c);
        add(box);
    }
    
    static MonitorPage edit(MonitorPage page, String title, Component parentComponent) {
        MonitorPage.Descriptor before = page == null ? null : page.save();
        MonitorPageDialog dialog = new MonitorPageDialog(before, title, parentComponent);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        return dialog.page;
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private boolean done() {
        
        if (view == null) {
            PersistableMonitorView.Descriptor desc = descriptor.getView();
            if (desc == null) {
                view = new DefaultView();
            } else {
                view = (PersistableMonitorView) service.make(desc);
            }
        }
        if (view == null) return false;
        
        if (filter == null) {
            PersistableAgentChannelsFilter.Descriptor desc = descriptor.getFilter();
            if (desc == null) {
                filter = (PersistableAgentChannelsFilter) AgentChannelsFilter.ALL;
            } else {
                filter = (PersistableAgentChannelsFilter) service.make(desc);
            }
        }
                
        String name = nameField.getText();
        
        try {
            page = new MonitorPage(view, filter, name);
        } catch (RuntimeException x) {
            return false;
        }
        return true;
    }

}
