package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;
import java.util.stream.Collectors;
import javax.swing.SwingConstants;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannel;

/**
 * {@link MonitorTable} that displays channel values in a 2D grid.
 *
 * @author onoprien
 */
public class MeshTable extends MonitorTable {

// -- Fields : -----------------------------------------------------------------
    
    private List<MonitorField> extraColumns;
    private int firstExtraColumn;
    private List<MonitorField> updatableFields;
    
    private LinkedHashMap<String, List<Map.Entry<String[], DisplayChannel>>> sections;

// -- Life cycle : -------------------------------------------------------------
    
    static public MeshTable getInstance(Collection<Map.Entry<String,DisplayChannel>> channels, List<MonitorField> fields) {
        MeshTable instance = new MeshTable();
        instance.init(channels, fields);
        return instance;
    }
    
    private MeshTable() {
    }
    
    public void init(Collection<Map.Entry<String,DisplayChannel>> channels, List<MonitorField> fields) {
        
        // Columns and collapsed sections:

        List<DisplayChannel> handles = channels.stream().map(e -> e.getValue()).collect(Collectors.toList());
        extraColumns = trimAbsentFields(fields, handles, Arrays.asList(MonitorField.NAME, MonitorField.VALUE));
        if (extraColumns.isEmpty()) {
            updatableFields = Collections.singletonList(MonitorField.VALUE);
        } else {
            updatableFields = new ArrayList<>();
            updatableFields.add(MonitorField.VALUE);
            extraColumns.stream().filter(MonitorField::isUpdatable).forEach(f -> updatableFields.add(f));
            ((ArrayList)updatableFields).trimToSize();
        }

        // Sectioned map of channels:
        
        sections = new LinkedHashMap<>();
        List<Map.Entry<String[], DisplayChannel>> defSection = new ArrayList<>();
        sections.put("", defSection);

        for (Map.Entry<String, DisplayChannel> e : channels) {
            String[] ss = e.getKey().split("/+");
            if (ss.length == 2) {
                defSection.add(new AbstractMap.SimpleImmutableEntry<>(ss, e.getValue()));
            } else if (ss.length == 3) {
                String section = ss[0];
                List<Map.Entry<String[], DisplayChannel>> cc = sections.get(section);
                if (cc == null) {
                    cc = new ArrayList<>();
                    sections.put(section, cc);
                }
                cc.add(new AbstractMap.SimpleImmutableEntry<>(Arrays.copyOfRange(ss, 1, 3), e.getValue()));
            }
        }
        
        // Build list of rows:
        
        ArrayList<String> columns = new ArrayList<>();
        cells = new ArrayList<>();
        ArrayList<DefaultMonitorCell> columnNamesRow = new ArrayList<>();
        columnNamesRow.add(DefaultMonitorCell.EMPTY);
        cells.add(columnNamesRow);
        int rowOffset = 1;
        
        for (Map.Entry<String, List<Map.Entry<String[], DisplayChannel>>> es : sections.entrySet()) {
            String section = es.getKey();
            if (!section.isEmpty()) { // section title row
                ArrayList<DefaultMonitorCell> sectionTitleRow = new ArrayList<>();
                sectionTitleRow.add(new SectionTitleCell(section));
                cells.add(sectionTitleRow);
                rowOffset++;
            }
            ArrayList<String> rows = new ArrayList<>();
            for (Map.Entry<String[], DisplayChannel> e : es.getValue()) {
                String rowName = e.getKey()[0];
                String columnName = e.getKey()[1];
                int rowIndex = rows.indexOf(rowName);
                ArrayList<DefaultMonitorCell> row;
                if (rowIndex == -1) {
                    rows.add(rowName);
                    rowIndex = rows.size() - 1 + rowOffset;
                    row = new ArrayList<>();
                    row.add(new DefaultMonitorCell(null, null, new FormattedValue(rowName, SwingConstants.LEFT)));
                    cells.add(row);
                } else {
                    rowIndex += rowOffset;
                    row = cells.get(rowIndex);
                }
                int columnIndex = columns.indexOf(columnName);
                if (columnIndex == -1) {
                    columns.add(columnName);
                    columnNamesRow.add(new DefaultMonitorCell(null, null, new FormattedValue(columnName, SwingConstants.CENTER)));
                    columnIndex = columns.size();
                } else {
                    columnIndex++;
                }
                while (row.size() <= columnIndex) {
                    row.add(DefaultMonitorCell.EMPTY);
                }
                ValueCell cell = new ValueCell(e.getValue(), rowIndex);
                e.getValue().setTarget(cell);
                format.format(cell);
                row.set(columnIndex, cell);
            }
            rowOffset += rows.size();
        }
        
        cells.trimToSize();
        nRows = cells.size();
        nColumns = 1 + columns.size() + extraColumns.size();
        firstExtraColumn = columns.size();
        
        ArrayList<DefaultMonitorCell> titleRow = cells.get(0);
        for (MonitorField f : extraColumns) {
            titleRow.add(new DefaultMonitorCell(null, null, new FormattedValue(f.getTitle(), SwingConstants.CENTER)));
        }
        
        for (int i=1; i<nRows; i++) {
            ArrayList<DefaultMonitorCell> row = cells.get(i);
            ArrayList<DisplayChannel> items = new ArrayList<>(row.size());
            for (DefaultMonitorCell cell : row) {
                items.addAll(cell.getChannels());
            }
            items.trimToSize();
            while (row.size() < columns.size()+1) {
                row.add(DefaultMonitorCell.EMPTY);
            }
            for (MonitorField f : extraColumns) {
                if (items.isEmpty()) {
                    row.add(DefaultMonitorCell.EMPTY);
                } else {
                    DefaultMonitorCell cell = new DefaultMonitorCell(items, f);
                    format.format(cell);
                    row.add(cell);
                }
            }
            row.trimToSize();
        }
    }
    
    @Override
    public void destroy() {
        sections.values().forEach(items -> items.forEach(e -> e.getValue().setTarget(null)));
    }

// -- Overriding MonitorTable : ------------------------------------------------
    
    @Override
    public boolean showHeader() {
        return false;
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class SectionTitleCell extends DefaultMonitorCell {
        
        final String title;
        
        SectionTitleCell(String title) {
            super(null, null, new FormattedValue("<html><b> "+ title, SwingConstants.LEFT));
            this.title = title;
        }
        
        String getTitle() {
            return title;
        }        
    }
    
    private class ValueCell extends DefaultMonitorCell implements Updatable {
        
        final int row;
        
        ValueCell(DisplayChannel channelHandle, int row) {
            super(channelHandle, MonitorField.VALUE);
            this.row = row;
        }

        @Override
        public void update(DisplayChannel channelHandle, List<MonitorField> fields) {
            for (MonitorField f : fields) {
                if (MonitorField.VALUE.equals(f)) {
                    format.format(this);
                } else {
                    int column = extraColumns.indexOf(f);
                    if (column != -1) {
                        format.format(cells.get(row).get(column + firstExtraColumn), channelHandle);
                    }
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            format.format(this);
            for (int column = 0; column < extraColumns.size(); column++) {
                MonitorField f = extraColumns.get(row);
                if (f.isUpdatable()) {
                    format.format(cells.get(row).get(column + firstExtraColumn), channelHandle);
                }
            }
            fireTableRowsUpdated(row, row);
            fireChangeEvent(CELLS_EVENT);
        }

        @Override
        public List<MonitorField> getAffectedFields(List<String> attributes) {
            return MonitorField.getAffectedFields(attributes, updatableFields);
        }
        
    }
    
}
