package org.lsst.ccs.gconsole.jas3;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GraphicsConfiguration;
import java.awt.Point;
import java.awt.Window;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Consumer;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JInternalFrame;
import org.freehep.application.mdi.DockPageManager;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.lsst.ccs.gconsole.base.panel.PanelEvent;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelListener;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.base.panel.PanelType;

/**
 * Jas3-specific implementation of {@link PanelManager}.
 * 
 * @author onoprien
 */
public final class JasPanelManager implements PanelManager, PageListener {
    
// -- Fields : -----------------------------------------------------------------
   
    static private final String CONTEXT_KEY = "_context_";
    
    private final Studio app;
    private final ArrayList<JComponent> panels = new ArrayList<>();
    private final CopyOnWriteArrayList<FilteredListener> listeners = new CopyOnWriteArrayList<>();
    private final Object listenersLock = new Object();
    private final Queue<PanelEvent> eventQueue = new LinkedList<>();
    
//    private CommandProcessor commander;
    
    
// -- Life cycle : -------------------------------------------------------------
    
    public JasPanelManager(Studio studio) {
        app = studio;
    }


// -- Opening and closing pages : ----------------------------------------------

    @Override
    public void open(Component panel, Map<?,?> properties) {
        
        if (! (panel instanceof JComponent)) {
            throw new IllegalArgumentException("Cannot manage panels that do not subclass JComponent");
        }
        JComponent view = (JComponent) panel;
        if (getContext(view) != null) {
            throw new IllegalArgumentException("The panel is already being managed");
        }
        
        EnumMap<Panel,Object> standardProperties = new EnumMap<>(Panel.class);
        HashMap<String,Object> extraProperties = null;
        if (properties != null) {
            for (Map.Entry<?,?> e : properties.entrySet()) {
                Object key = e.getKey();
                try {
                    Panel pKey = Panel.getKey(key);
                    standardProperties.put(pKey, e.getValue());
                } catch (IllegalArgumentException x) {
                    if (extraProperties == null) extraProperties = new HashMap<>();
                    extraProperties.put(key.toString(), e.getValue());
                }
            }
        }
        for (Panel key : Panel.values()) {
            if (standardProperties.get(key) == null) {
                Object valueInPanel = view.getClientProperty(key);
                if (valueInPanel != null && key.isValueValid(valueInPanel)) {
                    standardProperties.put(key, valueInPanel);
                }
            }
        }
        
        PanelType type = (PanelType) standardProperties.get(Panel.TYPE);
        if (type == null) {
            type = PanelType.DATA;
            standardProperties.put(Panel.TYPE, PanelType.DATA);
        }
        PageManager pageMan = getPageManager(type);
        PageContext context = pageMan.getContext(view);
        String title = (String) standardProperties.get(Panel.TITLE);
        Icon icon = (Icon) standardProperties.get(Panel.ICON);
        String group = (String) standardProperties.get(Panel.GROUP);
        
        if (context == null) {
            Boolean selected = (Boolean) standardProperties.getOrDefault(Panel.SELECTED, true);
            
            if ((Boolean)standardProperties.getOrDefault(Panel.DOCKED, true) || !(pageMan instanceof DockPageManager)) {
                context = pageMan.openPage(view, title, icon, group, selected);
            } else {
                Dimension size = (Dimension) standardProperties.get(Panel.SIZE);
                Point location = (Point) standardProperties.get(Panel.LOCATION);
                String device = (String) standardProperties.get(Panel.DEVICE);
                Boolean maximized = (Boolean) standardProperties.getOrDefault(Panel.MAXIMIZED, false);
                context = ((DockPageManager)pageMan).openPage(view, title, icon, group, selected, size, location, device, maximized);
            }
        } else {
            if (title != null) context.setTitle(title);
            if (icon != null) context.setIcon(icon);
            if (group != null) context.setType(group);
        }
        
        EnumSet<Panel> toSave = EnumSet.of(Panel.TYPE, Panel.ON_SAVE_AS, Panel.ON_CLOSE);
        for (Panel key : toSave) {
            Object value = standardProperties.get(key);
            if (value != null) {
                context.putProperty(key, value);
            }
        }
        if (extraProperties != null) {
            for (Map.Entry<String,Object> e : extraProperties.entrySet()) {
                context.putProperty(e.getKey(), e.getValue());
            }
        }
        view.putClientProperty(CONTEXT_KEY, context);
        panels.add(view);
        fireEvent(view, Panel.OPEN, false, true);
        context.addPageListener(this);
        view.validate();
    }

    @Override
    public void close(Component component) {
        PageContext context = getContext(component);
        if (context != null) {
            context.close();
        }
    }
    

// -- Getters : ----------------------------------------------------------------

    @Override
    public List<JComponent> getPanels() {
        return Collections.unmodifiableList(panels);
    }

    /**
     * Returns a property of the specified graphical console panel.
     * 
     * @param component Graphical console panel or one of its descendants.
     * @param key Property key.
     * @return The current value of the specified property, or {@code null} is the property is not set.
     * @throws IllegalArgumentException if the specified component is not a descendant of a panel managed by this {@code PanelManager}.
     */
    @Override
    public Object get(Component component, Object key) {
        PageContext context = getContext(component);
        if (context == null) throw new IllegalArgumentException("Not a managed component");
        JComponent panel = (JComponent) context.getPage();
        PageManager pm;
        Component c;
        try {
            Panel pKey = key instanceof Panel ? (Panel) key : Panel.valueOf(key.toString());
            switch (pKey) {
                //case TYPE:
                case GROUP:
                   return context.type();
                case TITLE:
                    return context.getTitle();
                case ICON:
                    return context.getIcon();
                case OPEN:
                    return true;
                case SELECTED:
                    pm = getPageManager((PanelType)context.getProperty(Panel.TYPE));
                    try {
                        return pm.getSelectedPage() == context;
                    } catch (Exception x) {
                        return false;
                    }
                case ICONIZED:
                    c = panel.getParent();
                    while (c != null) {
                        if (c instanceof Frame) {
                            return (((Frame)c).getExtendedState() & Frame.ICONIFIED) > 0;
                        } else if (c instanceof JInternalFrame) {
                            return ((JInternalFrame)c).isIcon();
                        }
                        c = c.getParent();
                    }
                    return false;
                case DOCKED:
                    pm = getPageManager((PanelType) context.getProperty(Panel.TYPE));
                    return pm instanceof DockPageManager ? ((DockPageManager)pm).isDocked(panel) : true;
                case SIZE:
                    return component.getSize();
                case LOCATION:
                    c = panel;
                    while (c != null) {
                        if (c == app) {
                            return null;
                        } else if (c instanceof Window || c instanceof JInternalFrame) {
                            return c.getLocation();
                        }
                        c = c.getParent();
                    }
                    return null;
                case DEVICE:
                    GraphicsConfiguration gc = panel.getGraphicsConfiguration();
                    return gc == null ? null : gc.getDevice().getIDstring();
                case MAXIMIZED:
                    c = panel.getParent();
                    while (c != null) {
                        if (c == app) {
                            return false;
                        } else if (c instanceof Frame) {
                            return (~((Frame)c).getExtendedState() & Frame.MAXIMIZED_BOTH) == 0;
                        } else if (c instanceof JInternalFrame) {
                            return ((JInternalFrame)c).isMaximum();
                        }
                        c = c.getParent();
                    }
                    return false;
                //case LAST_DESELECTED:
                default:
                    return context.getProperty(pKey);
            }
        } catch (IllegalArgumentException x) {
            return context.getProperty(key.toString());
        }
    }

    @Override
    public boolean isInState(Component panel, Map<?, ?> properties) {
        for (Map.Entry<?, ?> e : properties.entrySet()) {
            if (! Objects.equals(e.getValue(), get(panel, e.getKey()))) {
                return false;
            }
        }
        return true;
    }
    
    @Override
    public JComponent getPanel(Component component) {
        PageContext context = getContext(component);
        return context == null ? null : (JComponent) context.getPage();
    }
    
    
// -- Setters : ----------------------------------------------------------------

    /**
     * Sets a property on the graphical console panel that contains the specified component.
     * 
     * @param component Graphical console panel or one of its descendants.
     * @param key Property key.
     * @param value The new property value.
     * @throws IllegalArgumentException if the specified component is not a descendant of a panel managed by this {@code PanelManager}.
     */
    @Override
    public void set(Component component, Object key, Object value) {
        PageContext context = getContext(component);
        if (context == null) throw new IllegalArgumentException("Not a managed component");
        JComponent panel = (JComponent) context.getPage();
        try {
            Panel pKey = key instanceof Panel ? (Panel) key : Panel.valueOf(key.toString());
            pKey.checkValue(value);
            switch (pKey) {
                //case TYPE:
                case GROUP:
                    context.setType(value.toString());
                    break;
                case TITLE:
                    context.setTitle(value.toString());
                    break;
                case ICON:
                    context.setIcon((Icon)value);
                    break;
                //case OPEN:
                case SELECTED:
                    if ((Boolean)value) {
                        context.requestShow();
                    } else {
                        // FIXME
                    }
                    break;
                //case ICONIZED:
                //case DOCKED:
                //case SIZE:
                //case LOCATION:
                //case DEVICE:
                //case MAXIMIZED:
                case LAST_DESELECTED:
                    context.putProperty(Panel.LAST_DESELECTED, (Long)value);
                    break;
                default:
                    throw new IllegalArgumentException("Immutable property");
            }
        } catch (IllegalArgumentException x) {
            Object oldValue = context.putProperty(key, value);
            if (!Objects.equals(oldValue, value)) {
                fireEvent(panel, key, oldValue, value);
            }
        }
    }
    
    
// -- Responding to PageEvents fired by Jas3 : ---------------------------------

    @Override
    public void pageChanged(PageEvent pe) {
        PageContext context = pe.getPageContext();
        if (context == null) return;
        JComponent panel = (JComponent) context.getPage();
        if (panel == null || context != panel.getClientProperty(CONTEXT_KEY)) return;
        switch (pe.getID()) {
            case PageEvent.PAGEOPENED:
                break;
            case PageEvent.PAGECLOSED:
                context.removePageListener(this);
                Iterator<JComponent> it = panels.iterator();
                while (it.hasNext()) {
                    if (it.next() == panel) {
                        it.remove();
                        break;
                    }
                }
                Object onClose = get(panel, Panel.ON_CLOSE);
                if (onClose != null) {
                    try {
                        ((Consumer<JComponent>)onClose).accept(panel);
                    } catch (ClassCastException x) {}
                }
                fireEvent(panel, Panel.OPEN, true, false);
                removeListeners(panel);
                panel.putClientProperty(CONTEXT_KEY, null);
                break;
            case PageEvent.PAGEICONIZED:
                fireEvent(panel, Panel.ICONIZED, false, true);
                break;
            case PageEvent.PAGEDEICONIZED:
                fireEvent(panel, Panel.ICONIZED, true, false);
                break;
            case PageEvent.PAGESELECTED:
//                if (commander != null) {
//                    app.getCommandTargetManager().remove(commander);
//                }
//                commander = createCommander(panel);
//                if (commander != null) {
//                    app.getCommandTargetManager().add(commander);
//                }
                fireEvent(panel, Panel.SELECTED, false, true);
                String group = (String) get(panel, Panel.GROUP);
                if (group != null) {
                    List<? extends Component> groupPanels = getPanels(Panel.GROUP, group);
                    if (groupPanels.size() > 1) {
                        EnumSet<PanelType> alreadySelected = EnumSet.noneOf(PanelType.class);
                        List<? extends Component> selectedPanels = getPanels(Panel.SELECTED, true);
                        for (Component c : selectedPanels) {
                            String g = (String) get(c, Panel.GROUP);
                            if (group.equals(g)) {
                                alreadySelected.add((PanelType)get(c, Panel.TYPE));
                            }
                        }
                        EnumMap<PanelType,Component> toSelect = new EnumMap<>(PanelType.class);
                        for (Component p : groupPanels) {
                            if (p != panel) {
                                PanelType type = (PanelType) get(p, Panel.TYPE);
                                if (!alreadySelected.contains(type)) {
                                    Component pp = toSelect.get(type);
                                    if (pp != null) {
                                        long pTime = (long) getOrDefault(p, Panel.LAST_DESELECTED, 0L);
                                        long ppTime = (long) getOrDefault(pp, Panel.LAST_DESELECTED, 0L);
                                        if (pTime > ppTime) pp = null;
                                    }
                                    if (pp == null) {
                                        toSelect.put(type, p);
                                    }
                                }
                            }
                        }
                        toSelect.values().forEach(p -> set(p, Panel.SELECTED, true));
                    }
                }
                break;
            case PageEvent.PAGEDESELECTED:
//                if (commander != null) {
//                    app.getCommandTargetManager().remove(commander);
//                    commander = null;
//                }
                context.putProperty(Panel.LAST_DESELECTED, System.currentTimeMillis());
                fireEvent(panel, Panel.SELECTED, true, false);
                break;
        }
    }


// -- Event and listener handling : --------------------------------------------

    @Override
    public void addListener(PanelListener listener) {
        synchronized (listenersLock) {
            listeners.removeIf(fl -> fl.listener == listener && fl.panel != null);
            if (listeners.stream().noneMatch(fl -> fl.listener == listener)) {
                listeners.add(new FilteredListener(listener));
            }
        }
    }

    @Override
    public void addListener(PanelListener listener, Component component) {
        JComponent panel = getPanel(component);
        if (panel == null) throw new IllegalArgumentException("Attempt to manage listeners of a non-existing panel");
        synchronized (listenersLock) {
            if (listeners.stream().noneMatch(fl -> fl.listener == listener &&(fl.panel == null || fl.panel == panel))) {
                listeners.add(new FilteredListener(listener, panel));
            }
        }
    }

    @Override
    public void removeListener(PanelListener listener) {
        synchronized (listenersLock) {
            listeners.removeIf(fl -> fl.listener == listener);
        }
    }

    @Override
    public void removeListener(PanelListener listener, Component component) {
        JComponent panel = getPanel(component);
        if (panel == null) throw new IllegalArgumentException("Attempt to manage listeners of a non-existing panel");
        synchronized (listenersLock) {
            listeners.removeIf(fl -> fl.listener == listener && fl.panel == panel);
        }
    }
    
    private void removeListeners(JComponent panel) {
        synchronized (listenersLock) {
            listeners.removeIf(fl -> fl.panel == panel);
        }
    }
    
    private void fireEvent(JComponent panel, Object key, Object oldValue, Object newValue) {
        PanelEvent event = new PanelEvent(panel, key, oldValue, newValue);
        if (eventQueue.isEmpty()) {
            while (event != null) {
                for (FilteredListener fl : listeners) {
                    if (fl.panel == null || fl.panel == panel) {
                        fl.listener.process(event);
                    }
                }
                event = eventQueue.poll();
            }
        } else {
            eventQueue.add(event);
        }
    }
    
    private static class FilteredListener {
        final PanelListener listener;
        final JComponent panel;
        FilteredListener(PanelListener listener) {
            this(listener, null);
        }
        FilteredListener(PanelListener listener, JComponent panel) {
            this.listener = listener;
            this.panel = panel;
        }
    }

// -- Jas3-specific public methods : -------------------------------------------
    
    public PageManager getPageManager(PanelType type) {
        switch (type) {
            case DATA: return app.getPageManager();
            case CONTROL: return app.getControlManager();
            case CONSOLE: return app.getConsoleManager();
            default: throw new IllegalArgumentException("Cannot find page manager for panel of type "+ type);
        }
    }
    
    public PageContext findPage(PanelType type, String title) {
        PageManager pm = getPageManager(type);
        for (PageContext context : pm.pages()) {
            if (title.equals(context.getTitle())) {
                return context;
            }
        }
        return null;
    }
    
    /**
     * Returns {@code PageContext} instance whose view is the specified component or one of its ancestors.
     * Note that this method returns non-null value only for components that are currently
     * managed by this panel manager (meaning that they or their ancestors have been
     * passed to the {@link #open open(...)} method, and have not been closed after that.
     * 
     * @param component The graphical component whose context needs to be found.
     * @return The {@code PageContext} that handles the specified panel, or {@code null} if there is none.
     */
    public PageContext getContext(Component component) {
        while (component != null) {
            Object o = component instanceof JComponent ? ((JComponent)component).getClientProperty(CONTEXT_KEY) : null;
            if (o != null && o instanceof PageContext) {
                return (PageContext) o;
            } else {
                component = component.getParent();
            }
        }
        return null;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private CommandProcessor createCommander(JComponent comp) {
        Consumer<JComponent> action = (Consumer<JComponent>) get(comp, Panel.ON_SAVE_AS);
        if (action == null) return null;
        return new Commander(comp, action);
    }
    
    private class Commander extends CommandProcessor {
        
        private final JComponent panel;
        private final Consumer<JComponent> action;
        
        Commander(JComponent component, Consumer<JComponent> action) {
            panel = component;
            this.action = action;
        }
        
        public void onSaveAs() {
            action.accept(panel);
        }
    }
    
}
