package org.lsst.ccs.gconsole.services.optpage;

import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentStatusAggregator;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.messaging.AgentPresenceListener;

/**
 * Manages a set of pages created for a single {@code OptionalPage} descriptor.
 * Listens to {@link AgentStatusAggregator} and calls {@link OptionalPage} methods on the AWT Event Dispatch Thread.
 *
 * @author onoprien
 */
class OptionalPageHandler {

// -- Fields : -----------------------------------------------------------------
    
    private final OptionalPage descriptor;    
    private final HashMap<String,Page> path2page = new HashMap<>();
    private boolean stopped = true;
    private HashMap<String,DataPanelDescriptor> state;
    
    private final AgentPresenceListener listener;

// -- Life cycle : -------------------------------------------------------------
    
    OptionalPageHandler(OptionalPage descriptor) {
        this.descriptor = descriptor;
        listener  = new AgentPresenceListener() {
            @Override
            public void disconnecting(AgentInfo agent) {
                SwingUtilities.invokeLater(() -> onDisconnect(agent));
            }

            @Override
            public void connected(AgentInfo... agents) {
                SwingUtilities.invokeLater(() -> {
                    for (AgentInfo agent : agents) {
                        onConnect(agent);
                    }
                });
            }
        };
    }
    
    void start() {
        stopped = false;
        onConnect(null);
        Console.getConsole().getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(listener);
    }
    
    void stop() {
        Console.getConsole().getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(listener);
        new ArrayList<>(path2page.values()).forEach(page -> page.uninstall());
        stopped = true;
    }
    
    
// -- Getters/setters : --------------------------------------------------------
    
    OptionalPage getDescriptor() {
        return descriptor;
    }
    
//    void setState(HashMap<String,Boolean> state) {
//        this.state = state;
//    }
    
    
// -- Responding to agent connects/disconnects : -------------------------------
    
    private void onConnect(AgentInfo agent) {
        if (stopped) return;
        String path = getPath(agent);
        if (path == null) return;
        Page page = path2page.get(path);
        if (page == null) {
            page = new Page(path);
            page.install();
        }
        page.addAgent(agent);
    }

    private void onDisconnect(AgentInfo agent) {
        if (stopped) return;
        for (Page page : path2page.values()) {
            if (page.removeAgent(agent)) break;
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    public HashMap<String,DataPanelDescriptor> save() {
        HashMap<String,DataPanelDescriptor> out = new HashMap<>();
        path2page.forEach((path, page) -> {
            out.put(path, page.save());
        });
        return out;
    }
    
    public void restore(HashMap<String,DataPanelDescriptor> state) {
        this.state = state;
        path2page.forEach((path, page) -> {
            DataPanelDescriptor open = state.get(path);
            if (open != null) {
                page.setOpen(open.isOpen());
            }
        });
    }
    
    
// -- Other ops : --------------------------------------------------------------
    
    void setOpen(AgentInfo agent, boolean isOpen) {
        Page page = getPage(agent);
        if (page != null) {
            page.setOpen(isOpen);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    /**
     * Temporary - remove once OptionalPage.getPath(...) is removed.
     * 
     * @param agent
     * @return 
     */
    private String getPath(AgentInfo agent) {
        try {
            String name = descriptor.getPage(agent);
            if (name != null) {
                if (name.trim().isEmpty()) {
                    name = agent.getName();
                }
                String path = descriptor.getPath();
                if (path != null) {
                    name = path + "/" + name;
                }
            }
            return name;
        } catch (RuntimeException x) {
            return null;
        }
    }
    
    private Page getPage(AgentInfo agent) {
        if (stopped) return null;
        String path = getPath(agent);
        if (path == null) return null;
        Page page = path2page.get(path);
        return page;
    }
    
    
// -- Handle for a specific page : ---------------------------------------------
    
    private class Page extends AbstractAction {
        
        private final String path;
        JComponent component;
        final ArrayList<AgentInfo> agents = new ArrayList<>(1);
        
        Page(String path) {
            super(path.substring(path.lastIndexOf("/")+1));
            this.path = path;
        }
        
        /**
         * Called when this page is first created.
         * Initializes itself. Adds itself to the menu, to the list of pages.
         */
        void install() {
            DataPanelDescriptor d = state == null ? null : state.get(path);
            boolean open = d == null ? descriptor.isAutoOpen() : d.isOpen();
            putValue(Action.SELECTED_KEY, open);
            path2page.put(path, this);
            Console.getConsole().addMenu(this, null, getMenuPath());
        }
        
        /**
         * Called when this page is removed.
         */
        void uninstall() {
            Console.getConsole().removeMenu(path.split("/"));
            if (component != null) {
                Console.getConsole().getPanelManager().close(component);
            }
            path2page.remove(path);
        }
        
        void addAgent(AgentInfo agent) {
            agents.add(agent);
            if (isOpen()) {
                open(descriptor.open(agent, component));
            }
        }
        
        boolean removeAgent(AgentInfo agent) {
            Iterator<AgentInfo> it = agents.iterator();
            while (it.hasNext()) {
                AgentInfo a = it.next();
                if (a != null && a.getName().equals(agent.getName())) {
                    it.remove();
                    boolean close = descriptor.agentDisconnected(a);
                    if (close || (!isOpen() && agents.isEmpty())) {
                        uninstall();
                    }
                    return true;
                }
            }
            return false;
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (isOpen()) {
                component = null;
                for (AgentInfo agent : agents) {
                    open(descriptor.open(agent, component));
                }
            } else {
                if (component != null) {
                    Console.getConsole().getPanelManager().close(component);
                }
            }
        }
        
        boolean isOpen() {
            return (Boolean) getValue(Action.SELECTED_KEY);
        }
        
        void setOpen(boolean open) {
            boolean wasOpen = isOpen();
            if (open != wasOpen) {
                putValue(Action.SELECTED_KEY, open);
                actionPerformed(null);
            }
            if (open && wasOpen && component != null) {
                Console.getConsole().getPanelManager().set(component, Panel.SELECTED, true);
            }
        }
        
        DataPanelDescriptor save() {
            boolean open = (Boolean) getValue(Action.SELECTED_KEY);
            if (component != null && open) {
                return DataPanelDescriptor.get(component);
            } else {
                DataPanelDescriptor d = new DataPanelDescriptor();
                d.setOpen(false);
                return d;
            }
        }
        
        void restore(DataPanelDescriptor d) {
            
        }
        
        /**
         * Called to make this page visible.
         */
        void open(JComponent component) {
            if (component == null) return;
            if (this.component != component) {
                if (this.component != null) {
                    PanelManager pm = Console.getConsole().getPanelManager();
                    pm.set(this.component, Panel.ON_CLOSE, null);
                    pm.close(this.component);
                }
                this.component = component;
                String title = component.getName();
                if (title == null) title = getValue(NAME).toString();
                HashMap<Object, Object> par = new HashMap<>();
                if (state != null) {
                    DataPanelDescriptor d = state.get(path);
                    if (d != null) {
                        Map<String, Serializable> data = d.getData();
                        if (data != null) {
                            par.putAll(data);
                        }
                    }
                }
                par.put(Panel.TITLE, title);
                Consumer<JComponent> onClose = c -> close();
                par.put(Panel.ON_CLOSE, onClose);
                Console.getConsole().getPanelManager().open(component, par);
            }
            descriptor.opened(path);
        }
        
        /**
         * Called when this page is closed.
         */
        void close() {
            putValue(Action.SELECTED_KEY, false);
            descriptor.closed(path);
            if (agents.isEmpty()) {
                uninstall();
            }
        }
        
        private String[] getMenuPath() {
            return path.substring(0, path.lastIndexOf("/")).split("/");
        }

    }
    
}
