package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Point;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.dnd.DropTargetEvent;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.IOException;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.TransferHandler;
import javax.swing.event.ChangeListener;
import javax.swing.tree.TreePath;
import org.freehep.jas.flavors.ObjectFlavor;
import org.freehep.jas.services.PlotRegion;
import org.freehep.jas.services.PlotRegionDropHandler;
import org.freehep.jas.services.Plotter;
import org.freehep.swing.popup.HasPopupItems;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.gconsole.util.tree.SModel;
import org.lsst.ccs.gconsole.util.tree.STree;
import org.lsst.ccs.gconsole.util.tree.Sort;

/**
 * Control panel that displays a tree of trending channels and plot descriptors.
 * All public methods of this class are called on EDT.
 * <p>
 * <i>Note on drag-and-drop support. With the current implementation of {@code org.freehep.jas.plugin.plotter.DefaultRegion},
 * we have to use a somewhat weird hybrid way to support DND. The {@code DropHandler} to be
 * used on the region side gets embedded into the {@code Transferable}, along with the payload and 
 * a dummy plotter whose only purpose is to prevent {@code DefaultRegion} from rejecting the drop.
 * The only method of the embedded {@code DropHandler} that ever gets called is {@code drop(DropTargetDropEvent)},
 * and the user action is always {@code ACTION_LINK}. This makes it impossible to let the user
 * choose, by standard SWING methods, what happens on every particular drop - the operation is set
 * through preferences. If we ever need to extend DND functionality offered by the control panel, 
 * we should consider rewriting the {@code DefaultRegion} and using the standard SWING DND support.
 * Doing this without breaking other Jas3-based applications might be difficult however.</i> 
 *
 * @author onoprien
 */
public final class ControlPanel extends JPanel implements Persistable, HasPopupItems {

// -- Fields : -----------------------------------------------------------------
    
    static private final String DEFAULT_NAME = "Trending";
    
    private final LsstTrendingPlugin plugin;
    private final Descriptor descriptor;
    private final STree<Object> tree;
            
    private AgentChannelsFilter filter;
    
    private final ChangeListener trendingSourceListener = e -> updateFromTrendingSource();
    
    private final Action actShow, actPlot, actOverlay, actNewPlot, actNewPage, actRefresh;
    
    private Object currentSelection;

// -- Life cycle : -------------------------------------------------------------
    
    public ControlPanel(LsstTrendingPlugin plugin, Descriptor desc) {
        super(new BorderLayout());
        this.plugin = plugin;
        descriptor = desc == null ? new Descriptor() : desc;
        String name = desc == null ? DEFAULT_NAME : desc.getName();
        setName(name == null ? DEFAULT_NAME : name);
        
        SModel tModel = new SModel();
        SModel.Descriptor d = descriptor.getTreeModel();
        if (d != null) tModel.restore(d);
        tree = new STree<>(tModel);
        add(tree, BorderLayout.CENTER);
        
        actShow = new AbstractAction("Show") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.plot((Trend.Descriptor) currentSelection, TrendPlotter.Option.EXIST, TrendPlotter.Option.NEWPAGE);
                }
            }            
        };
        actPlot = new AbstractAction("Plot") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.plot((Trend.Descriptor) currentSelection);
                }
            }
        };
        actPlot.putValue("SHORT_DESCRIPTION", "Plot in the currently selected region.");
        
        actOverlay = new AbstractAction("Overlay") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.plot((Trend.Descriptor) currentSelection, TrendPlotter.Option.OVERLAY);
                }
            }
        };
        actOverlay.putValue("SHORT_DESCRIPTION", "Overlay on the currently selected plot.");
        
        actNewPlot = new AbstractAction("New Plot") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.plot((Trend.Descriptor) currentSelection, TrendPlotter.Option.NEWPLOT);
                }
            }
        };
        actNewPlot.putValue("SHORT_DESCRIPTION", "Plot in a new region.");
        
        actNewPage = new AbstractAction("New Page") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.plot((Trend.Descriptor) currentSelection, TrendPlotter.Option.NEWPAGE);
                }
            }
        };
        actNewPage.putValue("SHORT_DESCRIPTION", "Plot on a new page.");
        
        actRefresh = new AbstractAction("Refresh") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if (currentSelection instanceof Trend.Descriptor) {
                    plugin.refresh((Trend.Descriptor) currentSelection);
                }
            }
        };
        actRefresh.putValue("SHORT_DESCRIPTION", "Refresh plots for this channel.");

        MouseListener ml = new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                TreePath path = tree.getPathForLocation(e.getX(), e.getY());
                if (path != null) {
                    if (e.getClickCount() == 2) {
                        switch (plugin.getPreferences().getDoubleClick()) {
                            case -1:
                                actShow.actionPerformed(null); break;
                            case -2:
                                actPlot.actionPerformed(null); break;
                            case -3:
                                actNewPlot.actionPerformed(null); break;
                            case -4:
                                actNewPage.actionPerformed(null); break;
                            default:
                                actShow.actionPerformed(null);
                        }
                    }
                }
            }
        };
        tree.addMouseListener(ml);
        
        tree.addTreeSelectionListener(e -> {
            currentSelection = tree.getSelectedUserObject();
        });
        
        tree.setDragEnabled(true);
        tree.setTransferHandler(new TreeTransferHandler());
    }
    
    /** Called by the plugin once this panel has been opened. */
    public void start() {
        
        // Create filter if descriptor has one
        
        if (filter == null) {
            Persistable.Descriptor filterDesc = descriptor.getFilter();
            if (filterDesc != null) {
                PersistenceService serv = plugin.getConsole().getSingleton(PersistenceService.class);
                filter = (AgentChannelsFilter) serv.make(filterDesc);
            }
        }
        
        // Update from trending source and start listening for changes:
        
        plugin.getTrendingSource().addListener(trendingSourceListener);
        updateFromTrendingSource();
    }
    
    /** Called by the plugin once this panel has been closed. */
    public void stop() {
        plugin.getTrendingSource().removeListener(trendingSourceListener);
    }
    
    
// -- Updating : ---------------------------------------------------------------
    
    private void updateFromTrendingSource() {
        RestSource source = plugin.getTrendingSource();
        List<String> channels = source.getChannels();
        LinkedHashMap<String, Object> path2desc = new LinkedHashMap<>(channels.size() * 2);
        if (filter == null) {
            channels.forEach(path -> path2desc.put(path, new Trend.Descriptor(path)));
        } else {
            channels.forEach(path -> {
                List<String> displayPaths = AgentChannelsFilter.getDisplayPath(filter, path);
                displayPaths.forEach(dp -> path2desc.put(path, new Trend.Descriptor(path, dp)));
            });
        }
        if (tree.getModel().getRoot() != null && tree.getModel().getRoot().getChildCount() > 0) {
            descriptor.setTree(tree.save());
        }
        tree.getModel().update(path2desc);
        tree.restore(descriptor.getTree());
    }


// -- Pop-up menu : ------------------------------------------------------------
    
    @Override
    public JPopupMenu modifyPopupMenu(JPopupMenu menu, Component component, Point point) {
        if (tree.getModel() == null) return menu;

        if (filter == null) {
            JMenuItem it = new JMenuItem("Filter...");
            it.addActionListener(e -> {
                PersistenceService serv = plugin.getConsole().getSingleton(PersistenceService.class);
                AgentChannelsFilter f = (AgentChannelsFilter) serv.make(null, "Filter trending channels", tree, "AgentChannelsFilter");
                if (f != null) {
                    filter = f;
                    updateFromTrendingSource();
                }
            });
            menu.insert(it, 0);
        } else {
            JMenu m = new JMenu("Filter");
            JMenuItem it = new JMenuItem("New...");
            it.addActionListener(e -> {
                PersistenceService serv = plugin.getConsole().getSingleton(PersistenceService.class);
                AgentChannelsFilter f = (AgentChannelsFilter) serv.make(null, "Filter trending channels", tree, "AgentChannelsFilter");
                if (f != null) {
                    filter = f;
                    updateFromTrendingSource();
                }
            });
            m.add(it);
            if (filter instanceof Persistable) {
                it = new JMenuItem("Edit...");
                it.addActionListener(e -> {
                    PersistenceService serv = plugin.getConsole().getSingleton(PersistenceService.class);
                    AgentChannelsFilter f = (AgentChannelsFilter) serv.edit((Persistable) filter, "Filter trending channels", tree);
                    if (f != null) {
                        filter = f;
                        updateFromTrendingSource();
                    }
                });
                m.add(it);
            }
            it = new JMenuItem("Remove");
            it.addActionListener(e -> {
                filter = null;
                updateFromTrendingSource();
            });
            m.add(it);
            menu.insert(m, 0);
        }
        
        JMenu m = new JMenu("Sort");
        for (Sort sort : Sort.values()) {
            SModel model = tree.getModel();
            JCheckBoxMenuItem it = new JCheckBoxMenuItem(sort.toString(), sort.equals(model.save().getSort()));
            it.addActionListener(e -> {
                descriptor.setTree(tree.save());
                model.sort(sort);
                tree.restore(descriptor.getTree());
            });
            m.add(it);
        }
        menu.insert(m, 0);
        
        TreePath tp = tree.getPathForLocation(point.x, point.y);
        if (tp != null) {
            tree.setSelectionPath(tp);
            if (currentSelection instanceof Trend.Descriptor) {
                menu.insert(new JSeparator(), 0);
                menu.insert(actRefresh, 0);
                menu.insert(actNewPage, 0);
                menu.insert(actNewPlot, 0);
                menu.insert(actOverlay, 0);
                menu.insert(actPlot, 0);
                menu.insert(actShow, 0);
            }
        }
        
        return menu;
    }
    
    
// -- Drag and drop support : --------------------------------------------------
    
    /**
     * {@code TransferHandler} for the tree of plottable channels.
     */
    private class TreeTransferHandler extends TransferHandler {

        @Override
        protected Transferable createTransferable(JComponent c) {
            Trans trans = new Trans();
            trans.addDataForClass(PlotRegionDropHandler.class, new ChannelDropHandler(plugin));
            trans.addDataForClass(Plotter.class, DummyPlotter.INSTANCE);
            Object payload = tree.getSelectedUserObject();
            if (payload instanceof Trend.Descriptor) {
                trans.addDataForClass(Trend.Descriptor.class, payload);
            }
            return trans;
        }

        @Override
        public int getSourceActions(JComponent c) {
            return TransferHandler.LINK;
        }

    }
    
    /**
     * {@code Transferable} created by the tree of plottable channels.
     */
    static private class Trans implements Transferable {
        
        private final LinkedHashMap<DataFlavor, Object> data = new LinkedHashMap<>(4);

        @Override
        public DataFlavor[] getTransferDataFlavors() {
            return (new ArrayList<>(data.keySet())).toArray(new DataFlavor[0]);
        }

        @Override
        public boolean isDataFlavorSupported(DataFlavor flavor) {
            return data.keySet().contains(flavor);
        }

        @Override
        public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
            if (data.containsKey(flavor)) {
                return data.get(flavor);
            } else {
                throw new UnsupportedFlavorException(flavor);
            }
        }
    
        public void addDataForClass(Class clazz, Object payload) {
            data.put(new ObjectFlavor(clazz), payload);
        }
        
    }
    
    /**
     * AWT {@code DropHandler} to be embedded into the {@code Transferable} created by
     * the tree of plottable channels and used by the {@code DefaultRegion}.
     */
    static private class ChannelDropHandler implements PlotRegionDropHandler {

        private final LsstTrendingPlugin plugin;
        private PlotRegion region;
        
        public ChannelDropHandler(LsstTrendingPlugin plugin) {
            this.plugin = plugin;
        }

        @Override
        public void setPlotRegion(PlotRegion region) {
            this.region = region;
        }

        @Override
        public void dragEnter(DropTargetDragEvent dtde) {
        }

        @Override
        public void dragOver(DropTargetDragEvent dtde) {
        }

        @Override
        public void dropActionChanged(DropTargetDragEvent dtde) {
        }

        @Override
        public void dragExit(DropTargetEvent dte) {
        }

        @Override
        public void drop(DropTargetDropEvent dtde) {
            if (region == null) {
                return;
            }
            Transferable trans = dtde.getTransferable();
            Trend.Descriptor channel = null;
            for (DataFlavor flavor : trans.getTransferDataFlavors()) {
                Class k = flavor.getRepresentationClass();
                if (Trend.Descriptor.class.isAssignableFrom(k)) {
                    try {
                        channel = (Trend.Descriptor) trans.getTransferData(flavor);
                        break;
                    } catch (IOException | UnsupportedFlavorException x) {
                    }
                }
            }
            if (channel != null) {
                if (plugin.getPreferences().getDnd() == 1) {
                    plugin.plot(channel, region, TrendPlotter.Option.OVERLAY);
                } else {
                    plugin.plot(channel, region);
                }
            } else {
                dtde.rejectDrop();
            }
            dtde.dropComplete(true);
        }

    }

    /**
     * Dummy plotter for hoodwinking {@code DefaultRegion} into accepting drop.
     */
    static class DummyPlotter implements Plotter {
        static final DummyPlotter INSTANCE = new DummyPlotter();
        @Override
        public void plot(Object data, int mode) {
        }
        @Override
        public void plot(Object data, int mode, Object style, String options) {
        }
        @Override
        public void remove(Object data) {
        }
        @Override
        public void clear() {
        }
        @Override
        public Component viewable() {
            return null;
        }
        @Override
        public List getData() {
            return null;
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends Persistable.Descriptor {

        private Persistable.Descriptor filter;
        private STree.Descriptor tree;
        private SModel.Descriptor treeModel;
        
        public Persistable.Descriptor getFilter() {
            return filter;
        }

        public void setFilter(Persistable.Descriptor filter) {
            this.filter = filter;
        }

        public STree.Descriptor getTree() {
            return tree;
        }

        public void setTree(STree.Descriptor tree) {
            this.tree = tree;
        }

        public SModel.Descriptor getTreeModel() {
            return treeModel;
        }

        public void setTreeModel(SModel.Descriptor treeModel) {
            this.treeModel = treeModel;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        if (filter instanceof Persistable) {
            Persistable.Descriptor desc = ((Persistable)filter).save();
            if (desc != null) descriptor.setFilter(desc);
        }
        descriptor.setTree(tree.save());
        descriptor.setTreeModel(tree.getModel().save());
        return descriptor;
    }

}
