package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.*;

/**
 * Data model for a graphical component that displays a single monitored value.
 * There is a single {@link MonitorField} instance associated with this model, 
 * describing the type of the displayed value, and a list of contributing {@code ChannelHandle} objects.
 * The model also contains the current {@link FormattedValue}.
 * 
 * @author onoprien
 */
public class MonitorCell {

// -- Fields : -----------------------------------------------------------------

    static public MonitorCell EMPTY = new MonitorCell(Collections.emptyList(), MonitorField.NULL, FormattedValue.EMPTY);
    
    protected List<ChannelHandle> items;
    protected MonitorField field;
    
    private FormattedValue formattedValue;


// -- Life cycle : -------------------------------------------------------------

    /**
     * Constructs a cell with no associated channels, {@code NULL} field, and no formatted current value.
     */
    public MonitorCell() {
        this(Collections.emptyList(), MonitorField.NULL);
    }

    /**
     * Constructs a cell affected by a single data channel, with no current formatted value.
     * @param item Handle of the channel to be displayed, or {@code null} if cell is unaffected by any channels.
     * @param field Field to be displayed.
     */
    public MonitorCell(ChannelHandle item, MonitorField field) {
        this(item == null ? Collections.emptyList() : Collections.singletonList(item), field);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param items Handles of the channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     */
    public MonitorCell(List<? extends ChannelHandle> items, MonitorField field) {
        this(items, field, null);
    }

    /**
     * Constructs a cell affected by zero or more data channels, with no current formatted value.
     * @param items Handles of the channels contributing to this cell. The list will be owned by this cell.
     * @param field Field to be displayed.
     */
    public MonitorCell(List<? extends ChannelHandle> items, MonitorField field, FormattedValue data) {
        this.items = items == null ? Collections.emptyList() : (List<ChannelHandle>) items;
        this.field = field == null ? MonitorField.NULL : field;
        this.formattedValue = data;
    }
    
    /**
     * Constructs a cell whose content is specified by a format string and a list of channel paths.
     * 
     * @param format Format string. See {@link java.util.Formatter} for syntax description.
     * @param paths Data paths in {@code channel_path[?monitor_field]} format.
     */
//    MonitorCell(String format, String... paths) {
//        
//    } 
    
    
// -- Getters and setters : ----------------------------------------------------

    /**
     * Returns the list of handles for data channels that affect this cell.
     * @return List of contributing channels.
     */
    public List<ChannelHandle> getChannels() {
        return items;
    }

    /**
     * Returns the descriptor of the monitored field this cell displays.
     * @return Field descriptor.
     */
    public MonitorField getField() {
        return field;
    }

    /**
     * Sets the current formatted value to be displayed by this cell.
     * @param formattedValue New formatted value.
     */
    public void setFormattedValue(FormattedValue formattedValue) {
        this.formattedValue = formattedValue;
    }

    /**
     * Returns the current formatted value to be displayed by this cell.
     * @return Current formatted value.
     */
    public FormattedValue getFormattedValue() {
        return formattedValue;
    }

}
