package org.lsst.ccs.gconsole.agent.filter;

import java.util.*;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.annotations.ConsoleLookup;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;

/**
 * Generic configurable channel filter.
 *
 * @author onoprien
 */
public class GenericFilterPersistable extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final List<String> agents;
    private final List<String> channels;
    private final String delimeter;
    private final boolean agentTabs;
    private final List<String> fields;
    
    private HashMap<String,String> display2origin;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs a filter.
     * 
     * @param name Name of the filter.
     * @param agents Comma-separated list of subsystems accepted by this filter.
     * @param channels Comma-separated list of channels accepted by this filter (may contain templates and selectors).
     * @param delimeter Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).
     * @param separatePagesForAgents If {@code true}, the subsystem name is used as a page name.
     * @param fields Comma-separated list of fields to display.
     */
    @Create(category = "AgentChannelsFilter",
            name = "Configurable generic filter",
            path = "Built-In/Generic",
            description = "Channels filter that leaves paths unchanged (except for possibly replacing delimeters with slashes and inserting doupbe slashes to separate pages), but allows selecting desired subsystems, channels, and monitor fields.")
    public GenericFilterPersistable(
            @Par(def = Par.NULL, desc = "Filter name. Optional.") String name,
            @Par(def = Par.NULL, desc = "List of subsystems accepted by this filter. If not specified, all subsystems are accepted.") List<String> agents,
            @Par(def = Par.NULL, desc = "List of channels accepted by this filter (may contain templates and selectors). If not specified, all subsystems are accepted.") List<String> channels,
            @Par(def = Par.NULL, desc = "Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).") String delimeter,
            @Par(def = "false", desc = "If true, the first slash in the path is replaced by a double slash. This causes some monitoring views display each subsystem data in a separate tab.") boolean separatePagesForAgents,
            @Par(def = Par.NULL, desc = "List of fields to display in monitor views. If not specified, the default set of fields is displayed. The default depends on a specific view. Some views may ignore the list of fields suggested by the filter. Available standard fields: VALUE, UNITS, LOW_ALARM, LOW_WARN, ALERT_LOW, HIGH_ALARM, HIGH_WARN, ALERT_HIGH, DESCR.") List<String> fields)
    {
        
        this.name = name;
        this.agents = agents == null ? null : new ArrayList<>(agents);
        this.channels = channels == null ? null : new ArrayList<>(channels);
        
        if (delimeter == null || delimeter.isEmpty()) {
            this.delimeter = null;
        } else {
            String s = delimeter.trim();
            if (s.length() > 2 && s.startsWith("\"") && s.endsWith("\"")) {
                this.delimeter = s.substring(1, s.length()-1);
            } else {
                this.delimeter = Pattern.quote(delimeter);
            }
            display2origin = new HashMap<>();
        }
        
        this.agentTabs = separatePagesForAgents;
        this.fields = fields == null ? null : new ArrayList<>(fields);
    }
    
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public List<String> getDisplayPath(String originPath) {
        String displayPath;
        if (delimeter == null) {
            if (agentTabs) {
                displayPath = originPath.replaceFirst("/", "//");
            } else {
                displayPath = originPath;
            }
        } else {
            int i = originPath.indexOf("/");
            String agent = originPath.substring(0, i);
            String localPath = originPath.substring(i+1);
            localPath = localPath.replaceAll(delimeter, "/");
            if (agentTabs) {
                displayPath = agent +"//"+ localPath;
            } else {
                displayPath = agent +"/"+ localPath;
            }
            display2origin.putIfAbsent(displayPath, originPath);
        }
        return Collections.singletonList(displayPath);
    }

    @Override
    public String getOriginPath(String displayPath) {
        if (delimeter == null) {
            if (agentTabs) {
                return displayPath.replaceFirst("//", "/");
            } else {
                return displayPath;
            }
        } else {
            return display2origin.get(displayPath);
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return channels;
    }

    @Override
    public List<String> getAgents() {
        return agents;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getFields(boolean compact) {
        return compact ? null : fields;
    }

    
// -- Specific filters : -------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "Monitor channels",
            path = "Built-In/Monitor",
            description = "Channels filter that leaves paths unchanged (except for possibly replacing delimeters with slashes and inserting doupbe slashes to separate pages), but accepts only monitored channels.")
    static public AgentChannelsFilter filterMonitorOnly(
            @Par(def = "Monitored channels", desc = "Filter name.") String name,
            @Par(def = Par.NULL, desc = "List of subsystems accepted by this filter. If not specified, all subsystems are accepted.") List<String> agents,
            @Par(def = Par.NULL, desc = "Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).") String delimeter,
            @Par(def = "false", desc = "If true, the first slash in the path is replaced by a double slash. This causes some monitoring views display each subsystem data in a separate tab.") boolean separatePagesForAgents,
            @Par(def = Par.NULL, desc = "List of fields to display in monitor views. If not specified, the default set of fields is displayed. The default depends on a specific view. Some views may ignore the list of fields suggested by the filter.") List<String> fields)
    {
        return new GenericFilterPersistable(name, agents, Collections.singletonList(AgentChannel.Key.FORMAT), delimeter, separatePagesForAgents, fields);
    }
    
    @Create(category = "AgentChannelsFilter",
            name = "Subsystem Selector",
            path = "Built-In/Subsystem Selector",
            description = "Selects channels from the specified subsystem only. Leaves paths unchanged.")
    static public AgentChannelsFilter agentSelector(
            @Par(desc = "Subsystem name.") String subsystem)
    {
        return new GenericFilterPersistable(subsystem +" m-r", Collections.singletonList(subsystem), null, null, false, null);
    }
    
    
}
