package org.lsst.ccs.gconsole.agent.command;

import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.command.TokenizedCommand;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.CommandRejectedException;

/**
 * A {@code CommandTask} sends a command to a target subsystem and handles the responses. 
 * Methods are provided to check if the command processing is complete, to wait for its completion, and to retrieve the result.
 *
 * @author onoprien
 */
public class CommandTask {

// -- Fields : -----------------------------------------------------------------
    
    private final CommandSender sender;
        
    private final CommandHandle commandHandle;
    private long timeout;
    private final String command;
    private final Object[] args;
    
    private final ReentrantLock lock = new ReentrantLock();
    private final Condition condition = lock.newCondition();

    private ScheduledFuture<?> timeoutTask;
    private boolean done;
    private boolean canceled;
    private Object result;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Creates a {@code CommandTask}.
     * 
     * @param sender CommandSender that created this task.
     * @param handle Object to be notified of responses to the command.
     * @param timeout Timeout. Zero means the command will never time out. Negative
     *                timeout means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command in subsystem/path/method format.
     * @param args Command arguments.
     */
    public CommandTask(CommandSender sender, CommandHandle handle, Duration timeout, String command, Object... args) {
        this.sender = sender;
        this.commandHandle = handle;
        this.timeout = timeout.toMillis();
        this.command = command;
        this.args = args;
    }
    
    /**
     * Triggers sending the command to the target subsystem.
     */
    public void send() {
        sender.workerExecutor.execute(() -> {
            lock.lock();
            try {
                if (done) return;
            } finally {
                lock.unlock();
            }
            try {
                int i = command.lastIndexOf("/");
                if (i == -1) {
                    throw new IllegalArgumentException("Illegal command: " + command + ". Use \"agent[/component]/method\" format.");
                }
                String destination = command.substring(0, i);
                String com = command.substring(i + 1);
                CommandRequest request;
                if (areAllArgumentsStrings()) {
                    StringBuilder sb = new StringBuilder(com);
                    for (Object arg : args) {
                        sb.append(" ").append(arg);
                    }
                    TokenizedCommand tc = new TokenizedCommand(sb.toString());
                    request = new CommandRequest(destination, tc);
                } else {
                    request = new CommandRequest(destination, com, args);
                }
                sender.messenger.sendCommandRequest(request, new Originator());
                lock.lock();
                try {
                    if (!done && timeout != 0L) {
                        timeoutTask = sender.timer.schedule(CommandTask.this::timeOut, Math.abs(timeout), TimeUnit.MILLISECONDS);
                    }
                } finally {
                    lock.unlock();
                }
            } catch (Throwable x) {
                lock.lock();
                try {
                    done = true;
                    result = new IllegalStateException(x);
                    if (commandHandle != null) {
                        sender.callbackExecutor.execute(() -> {
                            commandHandle.onSendingFailure(x, CommandTask.this);
                        });
                    }
                    condition.signalAll();
                } finally {
                    lock.unlock();
                }
            }
        });
    }
    
// -- Future : -----------------------------------------------------------------
    
    /**
     * Waits if necessary for the command execution to complete, and then returns its result.
     * 
     * @return The command result.
     * @throws IllegalStateException If the command cannot be sent.
     * @throws InterruptedException If the current thread was interrupted while waiting.
     * @throws CommandRejectedException If the command was rejected by the target subsystem.
     * @throws CancellationException If the command was canceled.
     * @throws TimeoutException If the command timed out.
     * @throws ExecutionException If an exception was thrown by the target subsystem while executing the command,
     *                            or the result of the execution is an instance of {@code Throwable}.
     */
    public Object get() throws IllegalStateException, InterruptedException, CommandRejectedException, CancellationException, TimeoutException, ExecutionException {
        getResult();
        if (result instanceof Throwable) {
            if (result instanceof IllegalStateException) {
                throw (IllegalStateException)result;
            } else if (result instanceof InterruptedException) {
                throw (InterruptedException)result;
            } else if (result instanceof CommandRejectedException) {
                throw (CommandRejectedException)result;
            } else if (result instanceof CancellationException) {
                throw (CancellationException)result;
            } else if (result instanceof TimeoutException) {
                throw (TimeoutException)result;
            } else if (result instanceof ExecutionException) {
                throw (ExecutionException)result;
            } else {
                throw new ExecutionException((Throwable)result);
            }
        }
        return result;
    }
    
    /**
     * Waits if necessary for the command execution to complete, and then returns its result or exception.
     * Unlike the {@code get()} method, this method does not throw command-related exceptions. Instead, 
     * the outcome of this command task is reported through the return value.
     * 
     * @return The command result, or the exception that would be thrown by the {@link get()} method.
     * @throws InterruptedException If the current thread was interrupted while waiting.
     */
    public Object getResult() throws InterruptedException {
        lock.lock();
        try {
            while (!done) {
                condition.await();
            }
        } finally {
            lock.unlock();
        }
        return result;
    }
    
    /**
     * Cancels this command task.
     * Once the command is canceled, {@link CommandHandle} methods will not be called, and the {@code get()} method will throw {@code CancellationException}.
     * The target subsystem will not be notified of cancellation.
     * 
     * @return {@code False} if the command task could not be canceled, typically because it has already completed normally; {@code true} otherwise.
     */
    public boolean cancel() {
        lock.lock();
        try {
            if (done) return false;
            canceled = true;
            done = true;
            if (timeoutTask != null) timeoutTask.cancel(false);
            result = new CancellationException();
            if (commandHandle != null) {
                sender.callbackExecutor.execute(() -> commandHandle.onCancel((CancellationException) result, CommandTask.this));
            }
            condition.signalAll();
        } finally {
            lock.unlock();
        }
        return true;
    }
    
    /**
     * Returns {@code true} if this command task was canceled before it completed normally.
     * 
     * @return True if the command was canceled.
     */
    public boolean isCancelled() {
        lock.lock();
        try {
            return canceled;
        } finally {
            lock.unlock();
        }
    }
    
    /**
     * Returns true if this command task has completed.
     * Completion may be due to normal completion, an exception, or cancellation.
     * 
     * @return {@code True} if this task completed.
     */
    public boolean isDone() {
        lock.lock();
        try {
            return done;
        } finally {
            lock.unlock();
        }
    }
    
    
// -- Other getters : ----------------------------------------------------------
    
    /**
     * Returns the command as a string in subsystem/path/method format.
     * @return Command.
     */
    public String getCommand() {
        return command;
    }
    
    /**
     * Returns the command arguments.
     * @return Command arguments.
     */
    public Object[] getArguments() {
        return args;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void timeOut() {
        lock.lock();
        try {
            if (done) return;
            done = true;
            timeoutTask = null;
            result = new TimeoutException("Timed out after "+ Math.abs(timeout)/1000L +" seconds.");
            if (commandHandle != null) {
                sender.callbackExecutor.execute(() -> commandHandle.onTimeout((TimeoutException) result, CommandTask.this));
            }
            condition.signalAll();
        } finally {
            lock.unlock();
        }
    }
    
    private boolean areAllArgumentsStrings() {
        for (Object arg : args) {
            if (!(arg instanceof String)) return false;
        }
        return true;
    }
    
// -- Auxiliary classes : ------------------------------------------------------
    
    private class Originator implements CommandOriginator {

        @Override
        public void processNack(CommandNack nack) {
            lock.lock();
            try {
                if (done) return;
                done = true;
                if (timeoutTask != null) timeoutTask.cancel(false);
                result = new CommandRejectedException(nack);
                if (commandHandle != null) {
                    sender.callbackExecutor.execute(() -> commandHandle.onNack(nack, CommandTask.this));
                }
                condition.signalAll();
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void processAck(CommandAck ack) {
            lock.lock();
            try {
                if (done) return;
                if (ack != null && timeout > 0L) {
                    Duration customTimeout = ack.getTimeout();
                    if (customTimeout != null && !(customTimeout.isZero() || customTimeout.isNegative())) {
                        if (timeoutTask != null) timeoutTask.cancel(false);
                        timeout = customTimeout.toMillis();
                        timeout = timeout + timeout/10L + 1000L; // treat custom timeout as an estimate, use longer actual timeout
                        timeoutTask = sender.timer.schedule(CommandTask.this::timeOut, timeout, TimeUnit.MILLISECONDS);
                    }
                }
                if (commandHandle != null) {
                    sender.callbackExecutor.execute(() -> commandHandle.onAck(ack, CommandTask.this));
                }
            } finally {
                lock.unlock();
            }
        }

        @Override
        public void processResult(CommandResult result) {
            lock.lock();
            try {
                if (done) return;
                done = true;
                if (timeoutTask != null) timeoutTask.cancel(false);
                CommandTask.this.result = result.getResult();
                if (commandHandle != null) {
                    if (CommandTask.this.result instanceof Throwable) {
                        sender.callbackExecutor.execute(() -> commandHandle.onExecutionFailure((Throwable)CommandTask.this.result, CommandTask.this));
                    } else {
                        sender.callbackExecutor.execute(() -> commandHandle.onSuccess(CommandTask.this.result, CommandTask.this));
                    }
                }
                condition.signalAll();
            } finally {
                lock.unlock();
            }
        }
        
    }
    

}
