package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Color;
import java.awt.Dimension;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.Properties;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.ButtonModel;
import javax.swing.InputVerifier;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.event.CaretListener;
import org.freehep.application.Application;
import org.freehep.application.PropertyUtilities;
import org.freehep.application.studio.Studio;
import org.freehep.jas.services.PreferencesTopic;
import org.lsst.ccs.gconsole.base.Console;

/**
 * Plugin manager preferences panel.
 *
 * @author onoprien
 * @version $Id: $
 */
public class TrendingPreferences implements PreferencesTopic {

// -- Private parts : ----------------------------------------------------------

    private final LsstTrendingPlugin plugin;
    private final Console console;
    private final Studio app;

    private final String keyPrefix = "org.lsst.ccs.gui.trending.";
  
    private final String autoStartKey = "autoStart";
    private final String autoSaveKey = "autoSave";
    private final String useRawDataKey = "rawData";
    private final String nBinsKey = "bins";
    private final String refreshMinKey = "refreshMin";
    private final String refreshMaxKey = "refreshMax";
    private final String warnOnAlarmKey = "warnOnAlarm";
    private final String drawMetaKey = "drawMeta";
    private final String doubleClickKey = "doubleClick";
    private final String dndKey = "dnd";
    private final String ignoreMetaRangeKey = "ignoreMetaRange";
  
    /** property - start trending on application startup */
    private boolean autoStart;
    /** property - save/restore configuration on startup/shutdown */
    private boolean autoSave;
    /** property - use raw data if true, stat data if false */
    private boolean useRawData;
    /** property - number of bins to fetch from database and plot */
    private int nBins;
    /** property - minimum refresh interval, in seconds */
    private int refreshMin;
    /** property - maximum refresh interval, in seconds */
    private int refreshMax;
    /** property - warn if data outside alarm band */
    private boolean warnOnAlarm;
    /** property - metadata to draw */
    private EnumSet<Trend.Meta> drawMeta;
    /** property - double-click action */
    private int doubleClick;
    /** property - drag-and-drop action */
    private int dnd;
    /** property - ignore metadata when computing plot Y-axis range */
    private boolean ignoreMetaRange;
    

// -- Construction, reading and saving preferences : ---------------------------

    /** 
     * Constructs {@code TrendingPreferences} and initializes property values from saved user preferences.
     * @param plugin The trending plugin instance.
     */
    public TrendingPreferences(LsstTrendingPlugin plugin) {
        this.plugin = plugin;
        console = plugin.getConsole();
        app = (Studio) Application.getApplication();
        restore();
    }

    /** Initializes current settings from saved user preferences. */
    private void restore() {
        
        autoStart = (Boolean) console.addProperty(keyPrefix + autoStartKey, false);
        autoSave = (Boolean) console.addProperty(keyPrefix + autoSaveKey, false);
        useRawData = (Boolean) console.addProperty(keyPrefix + useRawDataKey, false);
        nBins = (Integer) console.addProperty(keyPrefix + nBinsKey, 50);
        refreshMin = (Integer) console.addProperty(keyPrefix + refreshMinKey, 10);
        refreshMax = (Integer) console.addProperty(keyPrefix + refreshMaxKey, 300);
        warnOnAlarm = (Boolean) console.addProperty(keyPrefix + warnOnAlarmKey, false);
        
        String dm = (String) console.addProperty(keyPrefix + drawMetaKey, "");
        String[] ss = dm.split("\\+");
        drawMeta = EnumSet.noneOf(Trend.Meta.class);
        for (String s : ss) {
            try {
                drawMeta.add(Trend.Meta.valueOf(s));
            } catch (IllegalArgumentException x) {
            }
        }
        
        doubleClick = (Integer) console.addProperty(keyPrefix + doubleClickKey, -1);
        dnd = (Integer) console.addProperty(keyPrefix + dndKey, 0);
        ignoreMetaRange = (Boolean) console.addProperty(keyPrefix + ignoreMetaRangeKey, false);
    }
    
    /** Saves user preferences. */
    private void save() {
        
        console.setProperty(keyPrefix + autoStartKey, autoStart);
        console.setProperty(keyPrefix + autoSaveKey, autoSave);
        console.setProperty(keyPrefix + useRawDataKey, useRawData);
        console.setProperty(keyPrefix + nBinsKey, nBins);
        console.setProperty(keyPrefix + refreshMinKey, refreshMin);
        console.setProperty(keyPrefix + refreshMaxKey, refreshMax);
        console.setProperty(keyPrefix + warnOnAlarmKey, warnOnAlarm);

        if (drawMeta.isEmpty()) {
            console.setProperty(keyPrefix + drawMetaKey, "");
        } else {
            StringBuilder sb = new StringBuilder();
            drawMeta.forEach(m -> sb.append(m).append("+"));
            console.setProperty(keyPrefix + drawMetaKey, sb.substring(0, sb.length() - 1));
        }
        
        console.setProperty(keyPrefix + doubleClickKey, doubleClick);
        console.setProperty(keyPrefix + dndKey, dnd);
        console.setProperty(keyPrefix + ignoreMetaRangeKey, ignoreMetaRange);
    }

// -- Getters : ----------------------------------------------------------------

    public boolean isAutoStart() {
        return autoStart;
    }

    public boolean isAutoSave() {
        return autoSave;
    }
    
    public boolean isUseRawData() {
        return useRawData;
    }

    public int getNBins() {
        return nBins;
    }
    
    public int getRefreshMin() {
        return refreshMin;
    }

   public int getRefreshMax() {
        return refreshMax;
    }

    public boolean isWarnOnAlarm() {
        return warnOnAlarm;
    }

    public EnumSet<Trend.Meta> getDrawMeta() {
        return drawMeta;
    }
    
    public int getDoubleClick() {
        return doubleClick;
    }
    
    public int getDnd() {
        return dnd;
    }

    public boolean isIgnoreMetaRange() {
        return ignoreMetaRange;
    }

    
// -- Implement PreferencesTopic : ---------------------------------------------

    @Override
    public String[] path() {
        return new String[] {"LSST","Trending"};
    }

    @Override
    public JComponent component() {
        return new GUI();
    }
    
    /** Reads GUI and updates current settings. */
    @Override
    public boolean apply(JComponent gui) {
        try {
            boolean out = ((GUI)gui).get();
            if (out) {
                save();
            }
            return out;
        } catch (ClassCastException x) {
            return false;
        }
    }
    

// -- GUI : --------------------------------------------------------------------
    
    
    private class GUI extends JPanel {
    
        private final int HSPACE = 10;
        private final int VSPACE = 5;
        
        JCheckBox _autoStartBox;
        JCheckBox _autoSaveBox;

        JRadioButton _useRawDataRadio;
        JRadioButton _useStatDataRadio;
        ButtonGroup _dataButtonGroup;
        JCheckBox _nBinsBox;
        JTextField _nBinsField;
        
        String[] _doubleClickOptions = new String[] {"Show", "Plot", "New Plot", "New Page"};
        JRadioButton[] _doubleClickRadio;
        ButtonGroup _doubleClickGroup;
        String[] _dndOptions = new String[] {"Replace", "Overlay"};
        JRadioButton[] _dndRadio;
        ButtonGroup _dndGroup;
        JCheckBox _warnOnAlarmBox;
        
        EnumMap<Trend.Meta,JCheckBox> _drawMetaBox = new EnumMap<>(Trend.Meta.class);
        JCheckBox selectedMetaBox;
        
        JCheckBox _ignoreMetaRangeBox;
        
        JRadioButton _refreshFixedRadio;
        JRadioButton _refreshAdaptiveRadio;
        ButtonGroup _refreshGroup;
        JTextField _refreshFixedField;
        JTextField _refreshMinField;
        JTextField _refreshMaxField;
        
        InputVerifier standardVerifier = new InputVerifier() {
            public boolean verify(JComponent input) {
                return input.getForeground().equals(Color.BLACK);
            }
        };

        GUI() {

            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            
            // Startup :

            Box panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Startup"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            Box row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(_autoStartBox = new JCheckBox("Start trending on startup."));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(_autoSaveBox = new JCheckBox("Restore configuration on startup."));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));

            // Plotting:

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Plotting"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Double-click action: "));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            _doubleClickRadio = new JRadioButton[_doubleClickOptions.length];
            _doubleClickGroup = new ButtonGroup();
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
           for (int i = 0; i<_doubleClickOptions.length; i++) {
                _doubleClickRadio[i] = new JRadioButton(_doubleClickOptions[i]);
                _doubleClickGroup.add(_doubleClickRadio[i]);
                row.add(_doubleClickRadio[i]);
            }
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            panel.add(Box.createRigidArea(new Dimension(0,VSPACE)));

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Drag-And-Drop action: "));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            _dndRadio = new JRadioButton[_dndOptions.length];
            _dndGroup = new ButtonGroup();
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
           for (int i = 0; i<_dndOptions.length; i++) {
                _dndRadio[i] = new JRadioButton(_dndOptions[i]);
                _dndGroup.add(_dndRadio[i]);
                row.add(_dndRadio[i]);
            }
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            panel.add(Box.createRigidArea(new Dimension(0,VSPACE)));

            Box metaPanel = Box.createVerticalBox();
            metaPanel.setBorder(BorderFactory.createTitledBorder("Show metadata:"));
            metaPanel.setAlignmentX(LEFT_ALIGNMENT);
            ButtonGroup bg = new ButtonGroup();
            bg.add(new JCheckBox());
            for (Trend.Meta m : Trend.Meta.values()) {
                JCheckBox cb = new JCheckBox(m.toString());
                if (m.isOnPoint()) {
                    bg.add(cb);
                    cb.addActionListener(e -> {
                        if (cb == selectedMetaBox) {
                            selectedMetaBox = null;
                            bg.clearSelection();
                        } else {
                            selectedMetaBox = cb;
                        }
                    });
                }
                _drawMetaBox.put(m, cb);
                row = Box.createHorizontalBox();
                row.setAlignmentX(LEFT_ALIGNMENT);
                row.add(cb);
                row.add(Box.createHorizontalGlue());
                metaPanel.add(row);
            }
            panel.add(metaPanel);
            
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _ignoreMetaRangeBox = new JCheckBox("Ignore metadata wnen computing axis range");
            row.add(_ignoreMetaRangeBox);
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _warnOnAlarmBox = new JCheckBox("Scan for plots with alarms");
            row.add(_warnOnAlarmBox);
            row.add(Box.createHorizontalGlue());
//            panel.add(row);
            
            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));

            // Data type and handling:

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Data"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("Data type: "));
            _dataButtonGroup = new ButtonGroup();
            _useRawDataRadio = new JRadioButton("Raw");
            _dataButtonGroup.add(_useRawDataRadio);
            row.add(_useRawDataRadio);
            _useStatDataRadio = new JRadioButton("Statistical");
            _dataButtonGroup.add(_useStatDataRadio);
            row.add(_useStatDataRadio);
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _nBinsBox = new JCheckBox("Reduce to");
            _nBinsBox.addActionListener(e -> {
                if (_nBinsBox.isSelected()) {
                    _nBinsField.setEnabled(true);
                    _nBinsField.setText(Integer.toString(nBins));
                } else {
                    _nBinsField.setEnabled(false);
                    _nBinsField.setText("");
                }
            });
            
            row.add(_nBinsBox);
            _nBinsField = new JTextField(3);
            _nBinsField.setMaximumSize(new Dimension(_nBinsField.getPreferredSize().width, _nBinsField.getPreferredSize().height));
            _nBinsField.addCaretListener(e -> {
                String text = _nBinsField.getText().trim();
                boolean valid = true;
                if (!text.isEmpty()) {
                    try {
                        int value = Integer.parseInt(text);
                        valid = value > 0;
                    } catch (NumberFormatException x) {
                        valid = false;
                    }
                }
                _nBinsField.setForeground(valid ? Color.BLACK : Color.RED);
            });
            _nBinsField.setInputVerifier(standardVerifier);
            row.add(_nBinsField);
            row.add(new JLabel(" bins if possible"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);

            add(panel);
            add(Box.createRigidArea(new Dimension(0, VSPACE)));
            
            // Auto-refresh

            panel = Box.createVerticalBox();
            panel.setBorder(BorderFactory.createTitledBorder("Auto Refresh Interval"));
            panel.setAlignmentX(LEFT_ALIGNMENT);

            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _refreshGroup = new ButtonGroup();
            _refreshFixedRadio = new JRadioButton("Fixed: ");
            _refreshGroup.add(_refreshFixedRadio);
            _refreshFixedRadio.addActionListener(e -> {
                if (_refreshFixedRadio.isSelected()) {
                    _refreshFixedField.setEnabled(true);
                    _refreshMinField.setEnabled(false);
                    _refreshMaxField.setEnabled(false);
                }
            });
            row.add(_refreshFixedRadio);
            _refreshFixedField = new JTextField(10);
            _refreshFixedField.setMaximumSize(new Dimension(_refreshFixedField.getPreferredSize().width, _refreshFixedField.getPreferredSize().height));
            _refreshFixedField.addCaretListener(e -> {
                String text = _refreshFixedField.getText().trim();
                boolean valid;
                try {
                    int value = Integer.parseInt(text);
                    valid = value > 0;
                } catch (NumberFormatException x) {
                    valid = false;
                }
                _refreshFixedField.setForeground(valid ? Color.BLACK : Color.RED);
            });
            _refreshFixedField.setInputVerifier(standardVerifier);
            row.add(_refreshFixedField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            _refreshAdaptiveRadio = new JRadioButton("Adaptive: ");
            _refreshGroup.add(_refreshAdaptiveRadio);
            _refreshAdaptiveRadio.addActionListener(e -> {
                if (_refreshAdaptiveRadio.isSelected()) {
                    _refreshFixedField.setEnabled(false);
                    _refreshMinField.setEnabled(true);
                    _refreshMaxField.setEnabled(true);
                }
            });
            row.add(_refreshAdaptiveRadio);
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("     Minimum "));
            _refreshMinField = new JTextField(6);
            _refreshMinField.setMaximumSize(new Dimension(_refreshMinField.getPreferredSize().width, _refreshMinField.getPreferredSize().height));
            row.add(_refreshMinField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
 
            row = Box.createHorizontalBox();
            row.setAlignmentX(LEFT_ALIGNMENT);
            row.add(new JLabel("     Maximum "));
            _refreshMaxField = new JTextField(6);
            _refreshMaxField.setMaximumSize(new Dimension(_refreshMaxField.getPreferredSize().width, _refreshMaxField.getPreferredSize().height));
            row.add(_refreshMaxField);
            row.add(new JLabel(" seconds"));
            row.add(Box.createHorizontalGlue());
            panel.add(row);
            
            CaretListener minMaxListener = e -> {
                int min = getInt(_refreshMinField);
                if (min < 0) {
                    _refreshMinField.setForeground(min == -2 ? Color.RED : Color.BLACK);
                    int max = getInt(_refreshMaxField);
                    _refreshMaxField.setForeground(max == -2 ? Color.RED : Color.BLACK);
                } else {
                    int max = getInt(_refreshMaxField);
                    if (max < 0) {
                        _refreshMinField.setForeground(Color.BLACK);
                        _refreshMaxField.setForeground(max == -2 ? Color.RED : Color.BLACK);
                    } else {
                        Color color = min < max ? Color.BLACK : Color.RED;
                        _refreshMinField.setForeground(color);
                        _refreshMaxField.setForeground(color);
                    }
                }
            };
            _refreshMinField.addCaretListener(minMaxListener);
            _refreshMaxField.addCaretListener(minMaxListener);
 
            add(panel);
            add(Box.createVerticalGlue());

            set();
        }
        
        final boolean get() {
            
            boolean autoStart = _autoStartBox.isSelected();
            boolean autoSave = _autoSaveBox.isSelected();
            boolean useRawData = _useRawDataRadio.isSelected();
            int nBins = 0;
            if (_nBinsBox.isSelected()) {
                try {
                    nBins = Integer.parseInt(_nBinsField.getText());
                    if (nBins < 1) {
                        return false;
                    }
                } catch (NumberFormatException x) {
                    return false;
                }
            }
            
            int doubleClick = -1;
            ButtonModel bm = _doubleClickGroup.getSelection();
            for (int i = 0; i<_doubleClickRadio.length; i++) {
                if (_doubleClickRadio[i].getModel().equals(bm)) {
                    doubleClick = -1 - i;
                }
            }
            int dnd = 0;
            bm = _dndGroup.getSelection();
            for (int i = 0; i<_dndRadio.length; i++) {
                if (_dndRadio[i].getModel().equals(bm)) {
                    dnd = i;
                }
            }
            
            boolean warnOnAlarm = _warnOnAlarmBox.isSelected();
            
            EnumSet<Trend.Meta> drawMeta = EnumSet.noneOf(Trend.Meta.class);
            for (Trend.Meta m : Trend.Meta.values()) {
                if (_drawMetaBox.get(m).isSelected()) {
                    drawMeta.add(m);
                }
            }
            
            int refreshMin;
            int refreshMax;
            if (_refreshFixedRadio.isSelected()) {
                try {
                    refreshMin = Integer.parseInt(_refreshFixedField.getText());
                    if (refreshMin < 1) {
                        return false;
                    }
                    refreshMax = refreshMin;
                } catch (NumberFormatException x) {
                    return false;
                }
            } else {
                refreshMin = getInt(_refreshMinField);
                if (refreshMin == -2) {
                    return false;
                } else if (refreshMin == -1) {
                    refreshMin = 0;
                }
                refreshMax = getInt(_refreshMaxField);
                if (refreshMax == -2) {
                    return false;
                } else if (refreshMax == -1) {
                    refreshMax = Integer.MAX_VALUE;
                }
                if (refreshMin >= refreshMax) {
                    return false;
                }
            }
            
            boolean ignoreMetaRange = _ignoreMetaRangeBox.isSelected();
            
            TrendingPreferences.this.autoStart = autoStart;
            TrendingPreferences.this.autoSave = autoSave;
            TrendingPreferences.this.doubleClick = doubleClick;
            TrendingPreferences.this.dnd = dnd;
            TrendingPreferences.this.useRawData = useRawData;
            TrendingPreferences.this.nBins = nBins;
            TrendingPreferences.this.refreshMin = refreshMin;
            TrendingPreferences.this.refreshMax = refreshMax;
            TrendingPreferences.this.warnOnAlarm = warnOnAlarm;
            TrendingPreferences.this.drawMeta = drawMeta;
            TrendingPreferences.this.ignoreMetaRange = ignoreMetaRange;
            return true;
        }

        final void set() {
            
            _autoStartBox.setSelected(autoStart);
            _autoSaveBox.setSelected(autoSave);
            _useRawDataRadio.setSelected(useRawData);
            _useStatDataRadio.setSelected(!useRawData);
            _nBinsBox.setSelected(nBins > 0);
            if (nBins > 0) {
                _nBinsField.setText(Integer.toString(nBins));
            }
            
            try {
                _doubleClickRadio[-doubleClick-1].setSelected(true);
            } catch (IndexOutOfBoundsException x) {
                _doubleClickRadio[0].setSelected(true);
            }
            try {
                _dndRadio[dnd].setSelected(true);
            } catch (IndexOutOfBoundsException x) {
                _dndRadio[0].setSelected(true);
            }
            
            _warnOnAlarmBox.setSelected(warnOnAlarm);
            
            for (Trend.Meta m : Trend.Meta.values()) {
                if (drawMeta.contains(m)) {
                    JCheckBox cb = _drawMetaBox.get(m);
                    cb.setSelected(true);
                    selectedMetaBox = cb;
                }
            }
            
            boolean fixedRefreshRate = refreshMin == refreshMax;
            if (fixedRefreshRate) {
                _refreshFixedRadio.setSelected(true);
                _refreshFixedField.setText(Integer.toString(refreshMin));
                _refreshFixedField.setEnabled(true);
                _refreshMinField.setEnabled(false);
                _refreshMaxField.setEnabled(false);
            } else {
                _refreshAdaptiveRadio.setSelected(true);
                _refreshMinField.setText(refreshMin == 0 ? "" : Integer.toString(refreshMin));
                _refreshMaxField.setText(refreshMax == Integer.MAX_VALUE ? "" : Integer.toString(refreshMax));
                _refreshFixedField.setEnabled(false);
                _refreshMinField.setEnabled(true);
                _refreshMaxField.setEnabled(true);
            }
            
            _ignoreMetaRangeBox.setSelected(ignoreMetaRange);
            
        }
        
        private int getInt(JTextField field) {
            String text = field.getText().trim();
            if (text.isEmpty()) {
                return -1;
            } else {
                try {
                    int out = Integer.parseInt(text);
                    return out < 0 ? -2 : out;
                } catch (NumberFormatException x) {
                    return -2;
                }
            }
        }
    }

}
