package org.lsst.ccs.gconsole.plugins.monitor;

import java.util.List;

/**
 * Implemented by software components whose states should be updated in response to changes in data channels.
 * The component state might depend on one or more data channels.
 *
 * @author onoprien
 */
public interface Updatable {

    /**
     * Updates this component.
     * The default implementation forwards the call to {@link #update(ChannelHandle)}.
     * 
     * @param channelHandle Handle of the data channel that might have changed.
     * @param fields List of fields that might need updating.
     */
    void update(ChannelHandle channelHandle, List<MonitorField> fields);

    /**
     * Updates this component.
     * The implementation should assume all fields might have changed.
     * 
     * @param channelHandle Handle of the data channel that might have changed.
     */
    void update(ChannelHandle channelHandle);
    
    /**
     * Translates a list of attributes whose values might have changed into the list of fields that might have been affected.
     * The default implementation forwards the call to {@link MonitorField#getAffectedFields(Collection)}.
     * 
     * @param attributes List of attributes whose values might have changed.
     * @return List of fields that might have been affected by changes.
     */
    default List<MonitorField> getAffectedFields(List<String> attributes) {
        return MonitorField.getAffectedFields(attributes);
    }

}
