package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Color;
import java.beans.Transient;
import java.io.Serializable;
import java.util.*;
import java.util.function.Function;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.plugins.trending.timeselection.TimeWindow;
import org.lsst.ccs.gconsole.services.persist.Savable;

/**
 * Data object that can be plotted by {@link TrendPlotter}.
 * This class is thread-safe, all methods can be called on any thread.
 * If the Trend instance is associated with a plotter, the plotter will be automatically
 * notified of any relevant changes in the  trend.
 *
 * @author onoprien
 */
public class Trend implements Savable {
   
    /**
     * Enumeration of metadata types that might be plotted for a trending channel.
     * On-point metadata is plotted as vertical bars associated with every data point. 
     * Only one on-point metadata type can be displayed at a time.
     * Off-point metadata is plotted as lines. 
     */
    public enum Meta {
        /** Min/max range of data represented by a single point on the plot. On-point. */
        RANGE(null, 
              in -> {
                  int n = in[0].length;
                  double[] errMin = new double[n];
                  double[] errMax = new double[n];
                  for (int i = 0; i < n; i++) {
                      errMin[i] = Math.max(in[0][i] - in[1][i], 0.);
                      errMax[i] = Math.max(in[2][i] - in[0][i], 0.);
                  }
                  return new double[][]{errMin, errMax};
              },
              "min", "max"),
        /** RMS of data represented by a single point on the plot. On-point. */
        RMS(null, 
            in -> new double[][] {in[1], in[1]},
            "rms"), 
        /** Alarm triggering level. Off-point. */
        ALARM(Color.RED, null, "alarmLow", "alarmHigh"),
        /** Warning triggering level. Off-point. */
        WARNING(Color.GREEN, null, "warningLow", "warningHigh");
        
        static private final EnumSet<Meta> onPointSet = EnumSet.of(RANGE, RMS);
        static private final EnumSet<Meta> offPointSet = EnumSet.of(ALARM, WARNING);
        private final Color color;
        private final List<String> keys;
        private final Function<double[][],double[][]> computeBars;
        
        Meta(Color color, Function<double[][],double[][]> computeBars, String... keys) {
            this.color = color;
            this.computeBars = computeBars;
            this.keys = Collections.unmodifiableList(Arrays.asList(keys));
        }
        
        static public EnumSet<Meta> getOnPointSet() {
            return onPointSet;
        }
        static public EnumSet<Meta> getOffPointSet() {
            return offPointSet;
        }
            
        public boolean isOnPoint() {
            return onPointSet.contains(this);
        }
        public Color getColor() {
            return color;
        }
        public int size() {
            return keys.size();
        }
        public List<String> getKeys() {
            return keys;
        }
        /**
         * Computes data for displaying error bars.
         * @param in double[size()+1][n] array. The first index: 0 for value, then for all keys.
         * @return double[2][n] array. The first index is 0 for lower bar.
         */
        public double[][] getBars(double[][] in) {
            if (in == null || computeBars == null) return new double[2][];
            for (double[] data : in) {
                if (data == null) return new double[2][];
            }
            return computeBars.apply(in);
        }
    }

// -- Fields : -----------------------------------------------------------------
    
    private final Descriptor descriptor;
    
    private TrendPlotter plotter;
    private TrendData data;
    private TimeWindow timeWindow;
    private long timestamp;
    private boolean loading;

// -- Life cycle : -------------------------------------------------------------
    
    public Trend(Descriptor descriptor) {
        this.descriptor = descriptor;
    }

// -- Getters and setters : ----------------------------------------------------
    
    synchronized public Descriptor getDescriptor() {
        return descriptor;
    }

    synchronized public TrendPlotter getPlotter() {
        return plotter;
    }

    synchronized void setPlotter(TrendPlotter plotter) {
        this.plotter = plotter;
    }

    synchronized public TrendData getData() {
        return data;
    }

    synchronized public void setData(TrendData data) {
        this.data = data;
        timestamp = System.currentTimeMillis();
        loading = false;
        if (plotter != null) {
            TrendPlotter p = plotter;
            SwingUtilities.invokeLater(() -> {
                p.onDataChange(this);
            });
        }
    }

    synchronized public TimeWindow getTimeWindow() {
        return timeWindow;
    }

    synchronized public void setTimeWindow(TimeWindow timeWindow) {
        if (plotter != null && this.timeWindow != null && !this.timeWindow.equals(timeWindow)) {
            TrendPlotter p = plotter;
            SwingUtilities.invokeLater(() -> {
                p.onTimeWindowChange(this);
            });
        }
        this.timeWindow = timeWindow;
        descriptor.timeWindow = timeWindow.toCompressedString();
    }

    synchronized public long getTimestamp() {
        return timestamp;
    }

    synchronized public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }
    
    synchronized public long[] getTimeRange() {
        if (data == null) {
            long now = System.currentTimeMillis();
            return new long[] {timeWindow.getLowerEdge(now), timeWindow.getUpperEdge(now)};
        } else {
            return data.getTimeRange();
        }
    }
    
    synchronized public boolean setLoading(boolean value) {
        boolean out = value != loading;
        if (!loading && value) {
            timestamp = System.currentTimeMillis();
            loading = true;
            if (plotter != null) {
                TrendPlotter p = plotter;
                SwingUtilities.invokeLater(() -> {
                    p.onRefreshRequest(this);
                });
            }
        }
        loading = value;
        return out;
    }
    
    synchronized public boolean isLoading() {
        return loading;
    }

    
// -- Descriptor : -------------------------------------------------------------
    
    @Override
    synchronized public Descriptor save() {
        return descriptor;
    }
    
    static public class Descriptor implements Serializable {

        private String path;
        private String displayPath;
        private String timeWindow;
        
        public Descriptor() {
        }
        
        public Descriptor(String path) {
            this.path = path;
        }
        
        public Descriptor(String path, String displayPath) {
            this.path = path;
            this.displayPath = displayPath;
        }
        
        public Descriptor(Descriptor other) {
            this.path = other.path;
            this.displayPath = other.displayPath;
            this.timeWindow = other.timeWindow;
        }

        public String getTimeWindow() {
            return timeWindow;
        }

        public void setTimeWindow(String timeWindow) {
            this.timeWindow = timeWindow;
        }

        public String getDisplayPath() {
            return displayPath == null ? path : displayPath;
        }

        public void setDisplayPath(String displayPath) {
            this.displayPath = displayPath;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }
        
        // Extra getters :
        
        @Transient
        public String getTitle() {
            String s = getDisplayPath();
//            return s.substring(s.lastIndexOf("/")+1);
            return s;
        }
        
        // Override Object :

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("[TD: ");
            if (displayPath != null) {
                sb.append(displayPath).append(" --> ");
            }
            sb.append(path);
            if (timeWindow != null) {
                sb.append(" (").append(timeWindow).append(")");
            }
            sb.append("]");
            return sb.toString();
        }

    }
    
}
