package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.LayoutManager2;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.*;
import javax.imageio.ImageIO;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;

/**
 * Image-based {@link MonitorView}.
 * <p>
 * The background image fills layer zero. 
 * Its size determines the preferred size of the view. Minimum and maximum sizes
 * supplied determine minimum and maximum sizes of the panel. When the panel is resized,
 * the background image is scaled.
 * <p>
 * The {@code addImage(...)}, {@code addComponent(...)}, {@code addView(...)} methods
 * can be used to add extra images, components, and monitoring data views to other layers.
 * The requirement that components in the same layer do not overlap adds an extra constraint to 
 * the minimum size of the panel. There are two versions of these methods,
 * for images and components that should and should not scale when the page is resized.
 * <p>
 * All content should be added before the {@link #getPanel} method is called.
 *
 * @author onoprien
 */
public class ImageView implements MonitorView {

// -- Fields : -----------------------------------------------------------------
    
    static public final float MAX_SCALE = 2.f;
    static public final float MIN_SCALE = .5f;
    static private final Dimension BIG = new Dimension(Integer.MAX_VALUE - 10, Integer.MAX_VALUE - 10);
    
    private String name;
    private final ArrayList<MListener> mListeners = new ArrayList<>(0);
    
    private BufferedImage back;
    private float minScale, maxScale;
    private final ArrayList<Item> items = new ArrayList<>();
    
    private JScrollPane scrollPane; // graphic component for this view
//    private JPanel viewPanel; // panel displayed by scrollPane; tracks viewport size; its layout maneger positions everything inside.
    private JLayeredPane layers; // contains backgound image panel and all items
    private ImagePanel backPanel; // backgound image panel
    
    
// -- Life cycle : -------------------------------------------------------------
    
    public ImageView(String name, BufferedImage image) {
        this(name, image, MIN_SCALE, MAX_SCALE);
    }
    
    public ImageView(String name, BufferedImage image, double minScale, double maxScale) {
        this.name = name;
        back = image;
        this.minScale = (float) minScale;
        this.maxScale = (float) maxScale;
    }
    
    public ImageView(String name, BufferedImage image, Dimension minSize, Dimension maxSize) {
        this.name = name;
        back = image;
        float w = image.getWidth();
        float h = image.getHeight();
        minScale = Math.max(minSize.width/w, minSize.height/h);
        maxScale = Math.min(maxSize.width/w, maxSize.height/h);
    }
        
    public void add(Item item) {
        if (isInitialized()) throw new IllegalStateException("Items cannot be added once the image view has been displayed");
        items.add(item);
    }
    
    public void addMouseListener(MListener listener) {
        mListeners.add(listener);
        mListeners.trimToSize();
    }
    
    private void initialize() {
        Layout layout = new Layout();
        JPanel viewPanel = new JPanel(layout);
        layers = new JLayeredPane();
        float originalWidth = back.getWidth();
        float originalHeight = back.getHeight();        
        for (Item item : items) {
            item.norm(originalWidth, originalHeight);
            JComponent component = item.getComponent();
            if (component != null) {
                layers.add(component, Integer.valueOf(item.getLayer()));
                if (component instanceof CellTableView) {
                    CellTableView ctw = (CellTableView) component;
                    ctw.setResizeListener(layout);
                }
            }
        }
        backPanel = new ImagePanel(back);
        backPanel.setOpaque(true);
        backPanel.addMouseListener(new MouseListener() {
            @Override
            public void mouseClicked(MouseEvent e) {
                mListeners.forEach(listener -> listener.mouseClicked(translateEvent(e)));
            }
            @Override
            public void mousePressed(MouseEvent e) {
                mListeners.forEach(listener -> listener.mousePressed(translateEvent(e)));
            }
            @Override
            public void mouseReleased(MouseEvent e) {
                mListeners.forEach(listener -> listener.mouseReleased(translateEvent(e)));
            }
            @Override
            public void mouseEntered(MouseEvent e) {
                mListeners.forEach(listener -> listener.mouseEntered(translateEvent(e)));
            }
            @Override
            public void mouseExited(MouseEvent e) {
                mListeners.forEach(listener -> listener.mouseExited(translateEvent(e)));
            }
            private MEvent translateEvent(MouseEvent e) {
                double x = e.getX() / (double) backPanel.getWidth();
                double y = e.getY() / (double) backPanel.getHeight();
                return new MEvent(e, x, y);
            }
        });
//        backPanel.setBackground(Color.WHITE);
        layers.add(backPanel, Integer.valueOf(0));
        backPanel.setLocation(0, 0);
        viewPanel.add(layers);
        scrollPane = new JScrollPane(viewPanel);
    }
    
    private boolean isInitialized() {
        return scrollPane != null;
    }

    
// -- Implementing MonitorView : -----------------------------------------------

    @Override
    public String getName() {
        return name;
    }
    
    @Override
    public void setName(String name) {
        this.name = name;
    }

    @Override
    public JComponent getPanel() {
        if (!isInitialized()) {
            initialize();
        }
        return scrollPane;
    }

    /**
     * Returns {@code null} since there is no filter associated with an image-based view.
     * @return Null.
     */
    @Override
    public AgentChannelsFilter getFilter() {
        return null;
    }

    /**
     * Does nothing since there is no filter associated with an image-based view.
     * @param filter Filter to ignore.
     */
    @Override
    public void setFilter(AgentChannelsFilter filter) {
    }

    @Override
    public void install() {
        items.forEach(item -> {
            Object content = item.getContent();
            if (content instanceof MonitorView) {
                ((MonitorView)content).install();
            }
        });
    }

    @Override
    public void uninstall() {
        items.forEach(item -> {
            Object content = item.getContent();
            if (content instanceof MonitorView) {
                ((MonitorView)content).uninstall();
            }
        });
    }

    @Override
    public Descriptor save() {
        return MonitorView.super.save();
    }
    
    @Override
    public void restore(Descriptor descriptor) {
        MonitorView.super.restore(descriptor);
    }
    
    
// -- Graphical component class : ----------------------------------------------
    
    /**
     * Matches the size of the scroll pane viewport.
     */
    private final class Panel extends JPanel {
        
        Panel() {
            super(new Layout());
            setName("Root");
        }

        @Override
        public void validate() {
//            System.out.println("Panel.validate() "+ isValid());
//            invalidate();
            super.validate();
        }
    }

    /**
     * Layout manager associated with "Panel viewPanel".
     * Handles placement of "JLayeredPane layers" inside "Panel viewPanel" and 
     * placement of items inside "JLayeredPane layers".
     */
    private class Layout implements LayoutManager2, ChangeListener {
    
        private Dimension minBackSize, maxBackSize;
        private final float originalWidth, originalHeight;
        
        private Dimension viewSize;  // cached viewport size
        private float scale; // scale of background image
        private int backX, backY, backWidth, backHeight; // position and size of "JLayeredPane layers" inside viewport
        
        private int w,h;
        private Item item;
        private Dimension pref, min, max;
        
        private Item scaleChangeItem;
        private int scaleModified, scaleModifiedThisPass;
        
        private boolean layoutValid = false;
        
        Layout() {
            originalWidth = back.getWidth();
            originalHeight = back.getHeight();
            minBackSize = new Dimension(Math.round(originalWidth * minScale), Math.round(originalHeight * minScale));
            maxBackSize = new Dimension(Math.round(originalWidth * maxScale), Math.round(originalHeight * maxScale));
            viewSize = new Dimension(0, 0);
        }
        
        @Override
        public void layoutContainer(Container parent) {

            // if viewport size has not changed, or is zero, return
            
            w = parent.getWidth();
            h = parent.getHeight();
            if ((w == viewSize.width && h == viewSize.height && layoutValid) || w == 0 || h == 0) {
                return;
            }
            viewSize.width = w;
            viewSize.height = h;

            // compute scale based on viewport size

            if (viewSize.width <= minBackSize.width || viewSize.height <= minBackSize.height) {
                scale = minScale;
                w = minBackSize.width;
                h = minBackSize.height;
            } else if (viewSize.width >= maxBackSize.width && viewSize.height >= maxBackSize.height) {
                scale = maxScale;
                w = maxBackSize.width;
                h = maxBackSize.height;
            } else {
                scale = Math.min(viewSize.width / originalWidth, viewSize.height / originalHeight);
                w = Math.round(originalWidth * scale);
                h = Math.round(originalHeight * scale);
            }

            // if scale has not changed, position layered pane and return
            
            if (w == backWidth && h == backHeight && layoutValid) {
                backX = Math.max((viewSize.width - w) / 2, 0);
                backY = Math.max((viewSize.height - h) / 2, 0);
                layers.setLocation(backX, backY);
//                System.out.println("Layout: no need to lay out items, same back size: "+ backWidth +" "+ backHeight);
                return;
            }
            
            backWidth = Math.round(originalWidth * scale);
            backHeight = Math.round(originalHeight * scale);
            
            // position items - might take several passes and result in changes to scale, backWidth, backHeight
            
            scaleModified = 0;
            scaleChangeItem = null;
            do {
                scaleModifiedThisPass = 0;
                try {
                    Iterator<Item> it = items.iterator();
                    while (it.hasNext()) {
                        boolean done = scaleChangeItem != null && scaleChangeItem == item;
                        if (done) scaleChangeItem = null;
                        item = it.next();
                        layoutItem();
                        if (done) break;
                    }
                } catch (IllegalStateException x) { // no way to satisfy all constraints, disable max size checks
                    int big = Integer.MAX_VALUE - 10;
                    if (maxBackSize.width >= big) throw new RuntimeException("Failure of ImageView layout: "+ getName());
//                    System.out.println("Removing max");
                    maxBackSize = BIG;
                    item.getComponent().setMaximumSize(maxBackSize);
                }
                if (scaleModified == 0) scaleModified = scaleModifiedThisPass;
            } while (scaleModifiedThisPass != 0);
            
            // adjust min and max if scale changed while laying out items
            
            if (scaleModified > 0) {
//                System.out.println("Scale increased: "+ backWidth +" "+ backHeight);
                minBackSize.width = backWidth;
                minBackSize.height = backHeight;
                minScale = scale;
                parent.getParent().validate();
            } else if (scaleModified < 0) {
//                System.out.println("Scale decreased: "+ backWidth +" "+ backHeight);
                maxBackSize.width = backWidth;
                maxBackSize.height = backHeight;
                maxScale = scale;
                parent.getParent().validate();
            }
            
            // position layered pane
            
            backX = Math.max((viewSize.width - backWidth)/2, 0);
            backY = Math.max((viewSize.height - backHeight)/2, 0);
            backPanel.setSize(backWidth, backHeight);
            layers.setBounds(backX, backY, backWidth, backHeight);
            layoutValid = true;
        }
        
        private void layoutItem() {
//            System.out.println("layout item: "+ item.getContent());
            
            JComponent component = item.getComponent();
            if (component == null) return;
            pref = component.getPreferredSize();
            min = component.getMinimumSize();
            max = component.getMaximumSize();
            
            if (item.getWidth() > 0.f) { // external size set
                w = Math.round(backWidth * item.getWidth());
                if (w < min.width) {
                    w = min.width;
                    checkWidth(1);
                } else if (w > max.width) {
                    w = max.width;
                    checkWidth(-1);
                }
            } else {
                w = pref.width;
                int wMin = Math.round(backWidth * item.getMinWidth());
                if (w < wMin) {
                    w = wMin;
                    checkWidth(-1);
                } else {
                    int wMax = Math.round(backWidth * item.getMaxWidth());
                    if (w > wMax) {
                        w = wMax;
                        checkWidth(1);
                    }
                }
            }
            
            if (item.getHeight() > 0.f) { // external size set
                h = Math.round(backHeight * item.getHeight());
                if (h < min.height) {
                    h = min.height;
                    checkHeight(1);
                } else if (w > max.height) {
                    h = max.height;
                    checkHeight(-1);
                }
            } else {
                h = pref.height;
                int hMin = Math.round(backHeight * item.getMinHeight());
                if (h < hMin) {
                    h = hMin;
                    checkHeight(-1);
                } else {
                    int hMax = Math.round(backHeight * item.getMaxHeight());
                    if (h > hMax) {
                        h = hMax;
                        checkHeight(1);
                    }
                }
            }
            
            if (item != scaleChangeItem) {
                float align = item.getAlignmentX() < 0.f ? component.getAlignmentX() : item.getAlignmentX();
                int x = Math.round(backWidth * item.getX() - w * align);
                align = item.getAlignmentY() < 0.f ? component.getAlignmentY() : item.getAlignmentY();
                int y = Math.round(backHeight * item.getY() - h * align);
//                System.out.println("Setting item bounds: " + x + " " + y + " " + w + " " + h);
                component.setBounds(x, y, w, h);
            }
            
        }
            
        private void checkWidth(int direction) {
            if (direction == -scaleModifiedThisPass) throw new IllegalStateException();
            if (direction > 0) {
                int wMax = Math.round(backWidth * item.getMaxWidth());
                if (wMax < min.width) {
                    backWidth = Math.round(min.width / item.getMaxWidth() + .5f);
                    scaleChangeItem = item;
                }
            } else {
                int wMin = Math.round(backWidth * item.getMinWidth());
                if (wMin > max.width) {
                    backWidth = Math.round(max.width / item.getMinWidth() - .5f);
                    scaleChangeItem = item;
                }
            }
            if (scaleChangeItem == item) {
                scaleModifiedThisPass = direction;
                scale = backWidth / (float) originalWidth;
                backHeight = Math.round(originalHeight * scale);
            }
        }
            
        private void checkHeight(int direction) {
            if (direction == -scaleModifiedThisPass) throw new IllegalStateException();
            if (direction > 0) {
                int hMax = Math.round(backHeight * item.getMaxHeight());
                if (hMax < min.height) {
                    backHeight = Math.round(min.height / item.getMaxHeight() + .5f);
                    scaleChangeItem = item;
                }
            } else {
                int hMin = Math.round(backHeight * item.getMinHeight());
                if (hMin > max.height) {
                    backHeight = Math.round(max.height / item.getMinHeight() - .5f);
                    scaleChangeItem = item;
                }
            }
            if (scaleChangeItem == item) {
                scaleModifiedThisPass = direction;
                scale = backHeight / (float) originalHeight;
                backWidth = Math.round(originalWidth * scale);
            }
        }

        @Override
        public void addLayoutComponent(Component comp, Object constraints) {
        }

        @Override
        public void addLayoutComponent(String name, Component comp) {
        }

        @Override
        public void removeLayoutComponent(Component comp) {
        }

        @Override
        public Dimension preferredLayoutSize(Container parent) {
//            System.out.println("Layout: preferredLayoutSize: "+ minBackSize);
            return minBackSize;
        }

        @Override
        public Dimension minimumLayoutSize(Container parent) {
//            System.out.println("Layout: minimumLayoutSize "+ minBackSize);
            return minBackSize;
        }

        @Override
        public Dimension maximumLayoutSize(Container target) {
//            System.out.println("Layout: maximumLayoutSize = BIG, should not be called");
            return BIG;
        }

        @Override
        public float getLayoutAlignmentX(Container target) {
            return JComponent.LEFT_ALIGNMENT;
        }

        @Override
        public float getLayoutAlignmentY(Container target) {
            return JComponent.TOP_ALIGNMENT;
        }

        @Override
        public void invalidateLayout(Container target) {
//            layoutValid = false;
//            System.out.println("Invalidating layout");
        }

        @Override
        public void stateChanged(ChangeEvent e) {
            layoutValid = false;
        }
                        
    }
    
    
// -- Helper classes : ---------------------------------------------------------
    
    static final class ImagePanel extends JPanel {
        
        private final BufferedImage image;
        
        private Image imageScaled;
        private int xSize, ySize;
        
        ImagePanel(BufferedImage image) {
            this.image = image;
            setAlignmentX(CENTER_ALIGNMENT);
            setAlignmentY(CENTER_ALIGNMENT);
            setPreferredSize(new Dimension(image.getWidth(), image.getHeight()));
            setOpaque(false);
        }

        @Override
        protected void paintComponent(Graphics g) {
            super.paintComponent(g);
            int widthCurrent = getWidth();
            int heightCurrent = getHeight();
            if (widthCurrent == xSize && heightCurrent == ySize) {
//                System.out.println("Painting cached image: "+ xSize +" "+ ySize);
                g.drawImage(imageScaled, 0, 0, null);
            } else {
                xSize = widthCurrent;
                ySize = heightCurrent;
                imageScaled = scale(image, xSize, ySize);
//                System.out.println("Painting re-scaled image: "+ xSize +" "+ ySize);
                g.drawImage(imageScaled, 0, 0, null);
            }
        }
        
        private Image scale(BufferedImage source, int width, int height) {
            return source.getScaledInstance(width, height, Image.SCALE_SMOOTH);
//            BufferedImage tmp = new BufferedImage(width, width, source.getType());
//            Graphics2D g2 = tmp.createGraphics();
////            g2.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
//            g2.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
//            g2.drawImage(source, 0, 0, width, height, null);
//            g2.dispose();
//            return tmp;
        }
        
    }
    
    static public class Item {
        
        Object content;
        float x, y;
        float width = -1;
        float height = -1;
        float maxWidth = -1;
        float maxHeight = -1;
        float minWidth = -1;
        float minHeight = -1;
        float alignmentX = -1;
        float alignmentY = -1;
        int layer = 1;
        
        public Item() {
        }
        
        public Item(Object content) {
            this.content = content;
        }
        
        void norm(float backWidth, float backHeight) {
            
            if (content instanceof BufferedImage) {
                BufferedImage image = (BufferedImage) content;
                content = new ImagePanel(image);
            } else if (content instanceof String) {
                content = new CellView((String)content);
            }
            
            if (x < 0.f || y < 0.f || x > backWidth || y > backHeight) throw new IllegalArgumentException();
            if (x > 1.f) x /= backWidth;
            if (y > 1.f) y /= backHeight;
                
            if (width > backWidth || height > backHeight) throw new IllegalArgumentException();
            if (width > 1.f) width /= backWidth;
            if (height > 1.f) height /= backHeight;
                
            if (maxWidth > backWidth || maxHeight > backHeight) throw new IllegalArgumentException();
            if (maxWidth > 1.f) {
                maxWidth /= backWidth;
            } else if (maxWidth < 0.f) {
                maxWidth = 10.f;
            }
            if (maxHeight > 1.f) {
                maxHeight /= backHeight;
            } else if (maxHeight < 0.f) {
                maxHeight = 10.f;
            }
                
            if (minWidth > backWidth || height > backHeight) throw new IllegalArgumentException();
            if (minWidth > 1.f) minWidth /= backWidth;
            if (minHeight > 1.f) minHeight /= backHeight;
            
            if (layer < 1) throw new IllegalArgumentException();
        }

        public Object getContent() {
            return content;
        }

        public void setContent(Object content) {
            this.content = content;
        }

        public float getX() {
            return x;
        }

        public void setX(double x) {
            this.x = (float) x;
        }

        public float getY() {
            return y;
        }

        public void setY(double y) {
            this.y = (float) y;
        }

        public float getWidth() {
            return width;
        }

        public void setWidth(double width) {
            this.width = (float) width;
        }

        public float getHeight() {
            return height;
        }

        public void setHeight(double height) {
            this.height = (float) height;
        }

        public float getMinWidth() {
            return minWidth;
        }

        public void setMinWidth(double minWidth) {
            this.minWidth = (float) minWidth;
        }

        public float getMinHeight() {
            return minHeight;
        }

        public void setMinHeight(double minHeight) {
            this.minHeight = (float) minHeight;
        }

        public float getMaxWidth() {
            return maxWidth;
        }

        public void setMaxWidth(double maxWidth) {
            this.maxWidth = (float) maxWidth;
        }

        public float getMaxHeight() {
            return maxHeight;
        }

        public void setMaxHeight(double maxHeight) {
            this.maxHeight = (float) maxHeight;
        }

        public float getAlignmentX() {
            return alignmentX;
        }

        public void setAlignmentX(double alignmentX) {
            this.alignmentX = (float) alignmentX;
        }

        public float getAlignmentY() {
            return alignmentY;
        }

        public void setAlignmentY(double alignmentY) {
            this.alignmentY = (float) alignmentY;
        }

        public int getLayer() {
            return layer;
        }

        public void setLayer(int layer) {
            this.layer = layer;
        }
        
        
        public void setPosition(double x, double y) {
            setX(x);
            setY(y);
        }
        
        public void setSize(double width, double height) {
            setWidth(width);
            setHeight(height);
        }
        
        public void setMinSize(double width, double height) {
            setMinWidth(width);
            setMinHeight(height);
        }
        
        public void setMaxSize(double width, double height) {
            setMaxWidth(width);
            setMaxHeight(height);
        }
        
        public void setAlignment(double alignmentX, double alignmentY) {
            setAlignmentX(alignmentX);
            setAlignmentY(alignmentY);
        }
        
        
        JComponent getComponent() {
            if (content instanceof MonitorView) {
                return ((MonitorView)content).getPanel();
            } else if (content instanceof JComponent) {
                return (JComponent) content;
            } else {
                return null;
            }
        }
        
    }
    
    public interface MListener {

        default public void mouseExited(MEvent e) {}

        default public void mouseEntered(MEvent e) {}

        default public void mouseReleased(MEvent e) {}

        default public void mousePressed(MEvent e) {}

        default public void mouseClicked(MEvent e) {}
        
    }
    
    static public class MEvent {
        
        private final MouseEvent event;
        private final double x;
        private final double y;
        
        public MEvent(MouseEvent event, double x, double y) {
            this.event = event;
            this.x = x;
            this.y = y;
        }

        public MouseEvent getEvent() {
            return event;
        }

        public double getX() {
            return x;
        }

        public double getY() {
            return y;
        }        
    }
    
    
// -- Testing : ----------------------------------------------------------------

    public static void main(String[] args) {
//        SwingUtilities.invokeLater(() -> createAndShowGUI());
        SwingUtilities.invokeLater(() -> measure());
    }

    private static void createAndShowGUI() {
        
        System.out.println("Starting... ");
        JFrame f = new JFrame("ImageView test");
        f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        
        try {
            BufferedImage back = ImageIO.read(ImageView.class.getResource("/org/lsst/ccs/gconsole/plugins/demo/monitor/image/grid.png"));
            ImageView imageView = new ImageView("ImageView test", back, .5, 1.2);
            Item item;
            
            DefaultChannelHandle handle = new DefaultChannelHandle("demo-subsystem-with-monitoring/main/DemoDevice1/DemoChannel0", null, null);
            FormattedValue fv = new FormattedValue();
            fv.horizontalAlignment = SwingUtilities.CENTER;
            MonitorField field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            MonitorCell c0 = new MonitorCell(handle, field);
            
            handle = new DefaultChannelHandle("demo-subsystem-with-monitoring/main/DemoDevice1/DemoChannel1", null, null);
            fv = new FormattedValue();
            fv.horizontalAlignment = SwingUtilities.CENTER;
            field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            MonitorCell c1 = new MonitorCell(handle, field);
            
            handle = new DefaultChannelHandle("demo-subsystem-with-monitoring/main/DemoDevice1/DemoChannel2", null, null);
            fv = new FormattedValue();
            fv.horizontalAlignment = SwingUtilities.CENTER;
            field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            MonitorCell c2 = new MonitorCell(handle, field);
            
            handle = new DefaultChannelHandle("demo-subsystem-with-monitoring/main/DemoDevice1/DemoChannel3", null, null);
            fv = new FormattedValue();
            fv.horizontalAlignment = SwingUtilities.CENTER;
            field = new MonitorField(MonitorField.VALUE.getKey(), null, fv, MonitorField.VALUE.getKeys());
            MonitorCell c3 = new MonitorCell(handle, field);

            CellTableView t = new CellTableView(2, 2, c0, c1, c2, c3);
            item = new ImageView.Item(t);
            item.setPosition(.5, .5);
            item.setAlignment(JComponent.LEFT_ALIGNMENT, JComponent.TOP_ALIGNMENT);
            item.setSize(.25, .25);
            item.setMaxSize(.375, .375);
            imageView.add(item);

//            BufferedImage demo = ImageIO.read(ImageView.class.getResource("/org/lsst/ccs/gconsole/plugins/demo/monitor/image/demo.png"));
//            item = new Item(demo);
//            item.setPosition(0, 0);
//            item.setAlignment(JComponent.LEFT_ALIGNMENT, JComponent.TOP_ALIGNMENT);
//            item.setSize(1.,1.);
//            item.setLayer(2);
//            imageView.add(item);
//
//            BufferedImage cir = ImageIO.read(ImageView.class.getResource("/org/lsst/ccs/gconsole/plugins/demo/monitor/image/circle.png"));
//            item = new Item(cir);
//            item.setPosition(40, 399);
//            item.setAlignment(JComponent.CENTER_ALIGNMENT, JComponent.CENTER_ALIGNMENT);
//            item.setSize(25,25);
//            item.setLayer(3);
//            imageView.add(item);

            f.add(imageView.getPanel());
        
            JTextField jfield = new JTextField();
            jfield.addActionListener(e -> {
                String[] ss = jfield.getText().trim().split(" ");
                int row = Integer.parseInt(ss[0]);
                int col = Integer.parseInt(ss[1]);
                t.test(row, col, ss[2]);
            });
            f.add(jfield, BorderLayout.SOUTH);
        
            f.pack();
            f.setVisible(true);
            
        } catch (IOException e) {
            System.out.println("Failed: "+ e);
        }

    }
    
    private static void measure() {
        
        JFrame f = new JFrame("ImageView test");
        f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        
        try {
            BufferedImage back = ImageIO.read(ImageView.class.getResource("/org/lsst/ccs/gconsole/util/refrig/vacuum.png"));
            ImageView imageView = new ImageView("ImageView ruler", back, 1., 1.);
            f.add(imageView.getPanel());
            f.pack();
            
            double[] prev = {0.,0.};

            imageView.addMouseListener(new MListener() {
                @Override
                public void mousePressed(MEvent e) {
                    System.out.println(String.format("%6.4f  %6.4f,  shift: %6.4f  %6.4f,   shift*2: %6.4f  %6.4f", e.getX(), e.getY(), e.getX()-prev[0], e.getY()-prev[1], 2*(e.getX()-prev[0]), 2*(e.getY()-prev[1])));
                    prev[0] = e.getX();
                    prev[1] = e.getY();
                }
            });
            
            f.setVisible(true);
        } catch (IOException e) {
            System.out.println("Failed: "+ e);
        }
        
    }

    
}
