package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.RowFilter;
import javax.swing.RowSorter;
import javax.swing.SortOrder;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableModel;
import javax.swing.table.TableRowSorter;
import org.lsst.ccs.bus.data.AgentGroup;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentLock;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import static org.lsst.ccs.gconsole.plugins.commandbrowser.Browser.ICON_UNLOCKED;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Root panel of the dictionary tool, complete with subsystem selection panel.
 *
 * @author onoprien
 */
public final class BrowserFull implements Browser {

// -- Fields : -----------------------------------------------------------------
    
    private final Logger logger = Logger.getLogger("org.lsst.ccs.plugin.jas3.dictionary");

    private JPanel browserPanel;
    private AgentTableModel agentTableModel;
    private JTable agentTable;    
    private AgentPanel agentPanel;
    private JLabel filterLabel;
   
    private final ArrayList<AgentHandle> agents = new ArrayList<>();
    private final Filter filter = new Filter();
    private Descriptor descriptor;
    
    private AgentStatusListener connectionsListener;
    private ChangeListener agentHandleListener;
    

// -- Life cycle : -------------------------------------------------------------
    
    public BrowserFull() {
        descriptor = new Descriptor();
    }
    
    @Override
    public JComponent getPanel() {
        if (browserPanel == null) {
            
            browserPanel = new JPanel(new BorderLayout());
            
            // Agent dictionary panel (right):
            
            agentPanel = new AgentPanel();
            browserPanel.add(agentPanel, BorderLayout.CENTER);

            // Subsystem selection panel (left):
            
            JPanel leftPane = new JPanel(new BorderLayout());
            leftPane.setPreferredSize(new Dimension(300, 800));
            browserPanel.add(leftPane, BorderLayout.WEST);

            agentTableModel = new AgentTableModel();
            agentTable = new JTable(agentTableModel);
            agentTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            agentTable.setFillsViewportHeight(true);
            agentTable.setDefaultRenderer(String.class, new AgentNameRenderer());
            agentTable.setDefaultRenderer(AgentLock.class, new LockRenderer());
            TableCellRenderer headerRenderer = agentTable.getTableHeader().getDefaultRenderer();
            for (int i = 0; i < 3; i++) {
                TableColumn column = agentTable.getColumnModel().getColumn(i);
                int w = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0).getPreferredSize().width;
                column.setMinWidth(Math.round(w*1.5f));
                column.setPreferredWidth(i == 0 ? 10*w : Math.round(w*1.5f));
            }
            agentTable.setRowHeight(Math.max(agentTable.getRowHeight(), 27));
            
            TableRowSorter<TableModel> sorter = new TableRowSorter<>(agentTableModel);
            List<RowSorter.SortKey> sortKeys = new ArrayList<>();
            sortKeys.add(new RowSorter.SortKey(0, SortOrder.ASCENDING));
            sortKeys.add(new RowSorter.SortKey(1, SortOrder.ASCENDING));
            sorter.setSortKeys(sortKeys);
            sorter.setRowFilter(filter);
            agentTable.setRowSorter(sorter);
            
            agentTable.getSelectionModel().addListSelectionListener(e -> {
                if (!e.getValueIsAdjusting()) {
                    AgentHandle agent = getSelectedAgent();
                    agentPanel.setAgent(agent);
                }
            });
            
            agentTable.addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    if (e.getClickCount() == 2) {
                        Point point = e.getPoint();
                        int column = agentTable.columnAtPoint(point);
                        if (column == 1) {
                            int row = agentTable.convertRowIndexToModel(agentTable.rowAtPoint(point));
                            if (row >= 0) {
                                AgentHandle agent = agents.get(row);
                                AgentLock lock = agent.getLock();
                                if (lock == null) {
                                    agent.setLock(true);
                                } else if (lock.getOwnerAgentName().equals(Console.getConsole().getAgentInfo().getName())) {
                                    agent.setLock(false);
                                }
                            }
                        }
                    }
                }
                
            });
            
            JScrollPane agentsView = new JScrollPane(agentTable);
            leftPane.add(agentsView, BorderLayout.CENTER);

            // Filter panel
            
            Box filterPanel = Box.createHorizontalBox();
            leftPane.add(filterPanel, BorderLayout.SOUTH);
            JButton filterButton = new JButton("Filter...");
            filterButton.addActionListener(e -> FilterDialog.show(filter, browserPanel));
            filterPanel.add(filterButton);
            filterPanel.add(Box.createRigidArea(Const.HDIM));
            filterLabel = new JLabel(filter.getDescription());
            filterPanel.add(filterLabel);
            
            // Listener for agent lock status changes:
            
            agentHandleListener = e -> {
                AgentHandle agent = (AgentHandle) e.getSource();
                for (int i=0; i<agents.size(); i++) {
                    if (agent.equals(agents.get(i))) {
                        if (agent.isOnline() || agent.isLockedByMe()) {
                            agentTableModel.fireTableRowsUpdated(i, i);
                        } else {
                            agents.remove(i);
                            agentTableModel.fireTableRowsDeleted(i, i);
                            agent.shutdown();
                        }
                    }
                }
            };
            
            // Set state based on descriptor to the extent possible without seeing connected agents:
            
            restore(descriptor);
            
            // Start listening for agent connections:
            
            connectionsListener = new AgentStatusListener() {
                @Override
                public void connect(AgentStatusEvent event) {
                    SwingUtilities.invokeLater(() -> {
                        AgentHandle agent = findAgent(event.getSource().getName());
                        if (agent == null) {
                            agent = new AgentHandle(event.getSource());
                            agent.init();
                            agents.add(agent);
                            agentTableModel.fireTableRowsInserted(agents.size() - 1, agents.size() - 1);
                            agent.addListener(agentHandleListener);
                            if (agent.getName().equals(descriptor.getAgent())) {
                                int i = agentTable.convertRowIndexToView(agents.size() - 1);
                                agentTable.getSelectionModel().setSelectionInterval(i, i);
                            }
                        } else {
                            agent.updateOnline(event.getSource());
                        }
                    });
                }
                @Override
                public void disconnect(AgentStatusEvent event) {
                    String agentName = event.getSource().getName();
                    SwingUtilities.invokeLater(() -> {
                        for (int i=0; i<agents.size(); i++) {
                            AgentHandle agent = agents.get(i);
                            if (agent.getName().equals(agentName)) {
                                agent.updateOnline(null);
                            }
                        }
                    });
                }
            };
            Console.getConsole().getStatusAggregator().addListener(connectionsListener, null, Collections.emptyList());
            
        }
        return browserPanel;
    }

    @Override
    public void shutdown() {
        agents.forEach(a -> a.shutdown());
        Console.getConsole().getStatusAggregator().removeListener(connectionsListener);
        browserPanel = null;
    }

    @Override
    public String getName() {
        return descriptor.getName();
    }

// -- Saving/restoring : -------------------------------------------------------

    @Override
    public void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            this.descriptor = (Descriptor) descriptor;
            filter.restore();
            if (browserPanel != null) {
                String selectedAgentName = this.descriptor.getAgent();
                if (selectedAgentName == null) {
                    agentTable.getSelectionModel().clearSelection();
                } else {
                    for (int i = agents.size()-1; i>=0; i--) {
                        if (selectedAgentName.equals(agents.get(i).getName())) {
                            i = agentTable.convertColumnIndexToView(i);
                            agentTable.getSelectionModel().setSelectionInterval(i, i);
                            break;
                        }
                    }

                }
                AgentPanel.Descriptor apDesc = this.descriptor.getAgentPanel();
                if (apDesc != null) agentPanel.restore(apDesc);
            }
        } else {
            logger.debug("Incorrect descriptor type for BrowserFull command browser: "+ descriptor);
        }
    }

    @Override
    public Descriptor save() {
        if (agentPanel == null) {
            descriptor.setAgentPanel(null);
        } else {
            descriptor.setAgentPanel(agentPanel.save());
        }
        if (browserPanel != null) {
            descriptor.setPage(DataPanelDescriptor.get(browserPanel));
        }
        return descriptor;
    }
    
    public static class Descriptor extends Browser.Descriptor {

        private String[] filterTypes;
        private ArrayList<String> filterNames;
        private ArrayList<String> filterGroups;
        private HashMap<String, String> filterProperties;

        public String[] getFilterTypes() {
            return filterTypes;
        }

        public void setFilterTypes(String[] filterTypes) {
            this.filterTypes = filterTypes;
        }

        public ArrayList<String> getFilterNames() {
            return filterNames;
        }

        public void setFilterNames(ArrayList<String> filterNames) {
            this.filterNames = filterNames;
        }

        public ArrayList<String> getFilterGroups() {
            return filterGroups;
        }

        public void setFilterGroups(ArrayList<String> filterGroups) {
            this.filterGroups = filterGroups;
        }

        public HashMap<String, String> getFilterProperties() {
            return filterProperties;
        }

        public void setFilterProperties(HashMap<String, String> filterProperties) {
            this.filterProperties = filterProperties;
        }
        
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private AgentHandle getSelectedAgent() {
        if (agentTable == null) return null;
        int i = agentTable.getSelectedRow();
        if (i != -1) i = agentTable.convertRowIndexToModel(i);
        return i < 0 ? null : agents.get(i);
    }
    
    private AgentHandle findAgent(String name) {
        for (AgentHandle agent : agents) {
            if (agent.getName().equals(name)) return agent;
        }
        return null;
    }
    
    
// -- Local classes : ----------------------------------------------------------
    
    private class AgentTableModel extends AbstractTableModel {

        @Override
        public int getRowCount() {
            return agents.size();
        }

        @Override
        public int getColumnCount() {
            return 3;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            AgentHandle a = agents.get(rowIndex);
            switch (columnIndex) {
                case 0:
                    return a.getName();
                case 1:
                    return a.getLock();
                case 2:
                    return a.getLevel();
            }
            throw new IllegalArgumentException();
        }

        @Override
        public boolean isCellEditable(int rowIndex, int columnIndex) {
            return columnIndex == 2 && !agents.get(rowIndex).isAdjusting();
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            switch (columnIndex) {
                case 0:
                    return String.class;
                case 1:
                    return AgentLock.class;
                case 2:
                    return Integer.class;
            }
            throw new IllegalArgumentException();
        }

        @Override
        public String getColumnName(int columnIndex) {
            switch (columnIndex) {
                case 0:
                    return "Subsystem";
                case 1:
                    return "Lock";
                case 2:
                    return "Level";
            }
            throw new IllegalArgumentException();
        }

        @Override
        public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
            if (columnIndex == 2) {
                try {
                    int level = Integer.parseInt(aValue.toString());
                    getSelectedAgent().setLevel(level);
                } catch (NumberFormatException | NullPointerException x) {
                }
            }
        }
        
    }

    private class AgentNameRenderer extends DefaultTableCellRenderer {
        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            setToolTipText(value.toString());
            AgentHandle agent = agents.get(table.convertRowIndexToModel(row));
            setEnabled(agent.isOnline());
            return this;
        }
    }
    
    static private class LockRenderer extends DefaultTableCellRenderer {
        
        private final String consoleName = Console.getConsole().getName();

        @Override
        protected void setValue(Object value) {
            super.setValue(value);
            setText("");
            if (value == null) {
                setIcon(ICON_UNLOCKED);
                setToolTipText("double-click to lock");
            } else {
                AgentLock lock = (AgentLock) value;
                if (lock.getOwnerAgentName().equals(consoleName)) {
                    setIcon(ICON_LOCKED);
                    setToolTipText("double-click to unlock");
                } else {
                    setIcon(ICON_UNAVAILABLE);
                    setToolTipText("locked by: "+ lock.getOwnerAgentName());
                }
            }
        }
        
    }
    
    class Filter extends RowFilter<Object,Integer> {
        
        EnumSet<AgentInfo.AgentType> types;
        ArrayList<String> names;
        ArrayList<String> groups;
        HashMap<String, String> properties;

        @Override
        public boolean include(RowFilter.Entry<? extends Object, ? extends Integer> entry) {
            int i = entry.getIdentifier();
            AgentInfo info = agents.get(i).getInfo();
            String name = agents.get(i).getName();

            if (types != null && info != null) {
                if (!types.contains(info.getType())) {
                    return false;
                }
            }

            if (names != null) {
                boolean fail = true;
                for (String fragment : names) {
                    if (name.contains(fragment)) {
                        fail = false;
                        break;
                    }
                }
                if (fail) {
                    return false;
                }
            }

            if (groups != null && info != null) {
                String group = info.getAgentProperty(AgentGroup.AGENT_GROUP_PROPERTY, "");
                if (!groups.contains(group)) {
                    return false;
                }
            }

            if (properties != null && info != null) {
                boolean fail = true;
                for (Map.Entry<String, String> e : properties.entrySet()) {
                    String value = info.getAgentProperty(e.getKey());
                    if (Objects.equals(value, e.getValue()) || (value != null && "".equals(e.getValue()))) {
                        fail = false;
                        break;
                    }
                }
                if (fail) {
                    return false;
                }
            }

            return true;
        }
        
        void apply() {
            agentTableModel.fireTableDataChanged();
            filterLabel.setText(getDescription());
        }
    
        void save() {
            if (types == null) {
                descriptor.setFilterTypes(null);
            } else {
                String[] tt = new String[this.types.size()];
                int i = 0;
                for (AgentInfo.AgentType type : this.types) {
                    tt[i++] = type.name();
                }
                descriptor.setFilterTypes(tt);
            }

            descriptor.setFilterNames(names == null ? null : new ArrayList<>(names));
            descriptor.setFilterGroups(groups == null ? null : new ArrayList<>(groups));
            descriptor.setFilterProperties(properties == null ? null : new HashMap<>(properties));
        }

        void restore() {
            if (descriptor.getFilterTypes() == null) {
                types = null;
            } else {
                types = EnumSet.noneOf(AgentInfo.AgentType.class);
                for (String s : descriptor.getFilterTypes()) {
                    try {
                        types.add(AgentInfo.AgentType.valueOf(s));
                    } catch (IllegalArgumentException x) {
                    }
                }
            }
            names = descriptor.getFilterNames() == null ? null : new ArrayList<>(descriptor.getFilterNames());
            groups = descriptor.getFilterGroups() == null ? null : new ArrayList<>(descriptor.getFilterGroups());
            properties = descriptor.getFilterProperties() == null ? null : new HashMap<>(descriptor.getFilterProperties());
        }
        
        String getDescription() {
            if (types == null) {
                if (names == null && groups == null && properties == null) {
                    return "All";
                }
            } else {
                if (names == null && groups == null && properties == null && types.size() < 3) {
                    return String.join(" & ", types.stream().map(t -> t.displayName()).collect(Collectors.toList()));
                }
            }
            return "Custom";
        }
    }
    
}
