package org.lsst.ccs.gconsole.services.command;

import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.agent.command.CommandSender;
import org.lsst.ccs.gconsole.agent.command.CommandHandle;
import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.function.Consumer;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.ConsoleService;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Provides services related to sending commands to remote subsystems.
 *
 * @author onoprien
 */
@Plugin(name = "Command Service Plugin",
        id="command-service",
        description = "Command service utility.")
public final class CommandService extends ConsolePlugin implements ConsoleService {

// -- Fields : -----------------------------------------------------------------
    
    private final CommandSender defaultSender;
    private final DefaultCommandHandle defaultHandle;

// -- Life cycle : -------------------------------------------------------------
    
    public CommandService() {
        defaultHandle = new DefaultCommandHandle();
        defaultSender = getSender();
        defaultSender.setCommandHandle(defaultHandle);
    }

// -- Sending commands : -------------------------------------------------------
    /**
     * Creates a new customizable instance of {@link CommandSender} configured for use in the graphical console environment.
     * For any commands sent through the {@code CommandSender} created by this method,
     * {@link CommandHandle} methods will be called on the AWT Event Dispatching Thread.
     * The sender created by this method can be further customized.
     * 
     * @return New instance of {@code CommandSender}.
     */
    public CommandSender getSender() {
        AgentMessagingLayer messenger = Console.getConsole().getMessagingAccess();
        Executor callbackExecutor = run -> SwingUtilities.invokeLater(run);
        Executor workerExecutor = run -> (new SwingWorker() {
            @Override
            protected Object doInBackground() throws Exception {
                run.run();
                return null;
            }
        }).execute();
        Scheduler timer = Console.getConsole().getScheduler();
        CommandSender sender = new CommandSender(messenger, callbackExecutor, workerExecutor, timer);
        sender.setCommandHandle(defaultHandle);
        return sender;
    }
    
    /**
     * Sends a command to a remote subsystem.
     * <p>
     * This method returns immediately, without waiting for the command to be processed
     * by the messaging service or the target subsystem. Synchronous command execution can
     * be done by calling {@link CommandTask#get get()} or {@link CommandTask#getResult() getResult()}
     * on the {@link CommandTask} returned by this method.
     * 
     * @param handle Handle for processing the command responses. May be {@code null} if no processing is required.
     *               All callbacks are executed on the AWT Event Dispatching Thread.
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command in subsystem/path format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(CommandHandle handle, Duration timeout, String command, Object... args) {
        return defaultSender.execute(handle, timeout, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout.
     * <p>
     * This method returns immediately, without waiting for the command to be processed
     * by the messaging service or the target subsystem. Synchronous command execution can
     * be done by calling {@link CommandTask#get get()} or {@link CommandTask#getResult() getResult()}
     * on the {@link CommandTask} returned by this method.
     * 
     * @param handle Handle for processing the command responses.
     *               Cannot be {@code null}. If no processing is required, use {@code CommandHandle.NONE}.
     *               All callbacks are executed on the AWT Event Dispatching Thread.
     * @param command Command in subsystem/path format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(CommandHandle handle, String command, Object... args) {
        return defaultSender.execute(handle, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem, and displays a message if the command fails.
     * <p>
     * This method returns immediately, without waiting for the command to be processed
     * by the messaging service or the target subsystem. Synchronous command execution can
     * be done by calling {@link CommandTask#get get()} or {@link CommandTask#getResult() getResult()}
     * on the {@link CommandTask} returned by this method.
     * 
     * @param timeout Timeout. Cannot be {@code null}.
     *                Zero duration means the command will never time out.
     *                Negative duration means its absolute value will be used and the custom timeout
     *                suggested by the target subsystem through an ACK will be ignored.
     * @param command Command in subsystem/path format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(Duration timeout, String command, Object... args) {
        return defaultSender.execute(timeout, command, args);
    }
    
    /**
     * Sends a command to a remote subsystem with the default timeout, and displays a message if the command fails.
     * The default timeout is 10 seconds, unless the target subsystem returns a custom timeout with ACK.
     * <p>
     * This method returns immediately, without waiting for the command to be processed
     * by the messaging service or the target subsystem. Synchronous command execution can
     * be done by calling {@link CommandTask#get get()} or {@link CommandTask#getResult() getResult()}
     * on the {@link CommandTask} returned by this method.
     * 
     * @param command Command in subsystem/path format.
     * @param args Command arguments.
     * @return CommandTask created to process the command. Can be used to wait for the response or cancel the command processing.
     */
    public CommandTask execute(String command, Object... args) {
        return defaultSender.execute(command, args);
    }

    
// -- Watching command state : -------------------------------------------------
    
    /**
     * <b>Not yet implemented.</b>
     * Registers a component that should be notified whenever the availability of the specified command changes.
     * Commands might be available or unavailable depending on whether the target subsystem is connected to the buses,
     * and whether the lock level held by the console is sufficient to execute the command.
     * All notifications are delivered on the AWT Event Dispatching Thread.
     * 
     * @param watcher Component to be notified.
     * @param command Command in subsystem/path format.
     * @param numberOfArguments Number of arguments accepted by the command.
     */
    public void watch(Consumer<Boolean> watcher, String command, int numberOfArguments) {
        throw new UnsupportedOperationException("Not yet implemented");
    }

}
