package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeListener;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.RowSorter;
import javax.swing.SwingUtilities;
import javax.swing.event.ChangeListener;
import javax.swing.event.TableModelEvent;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableModel;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.MutableAgentChannel;

/**
 * Monitoring data table to be displayed as a part of an {@link ImageView}.
 * <p>
 * The implementation provided by this class is optimized for handling small tables
 * (typically 1-4 cells) displayed on top of graphical representation of equipment
 * by {@link ImageView}. If this class is ever used for large tables, it will have to
 * be subclassed and at least the {@code update(...)} method will have to be overridden.
 *
 * @author onoprien
 */
public class CellTableView extends JTable implements MonitorView, AgentChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final int nRows, nColumns;
    private final ArrayList<MonitorCell> cells;
    
    private final Dimension maxSize = new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE);
    private int[] minRowHeights, minColumnWidths;
    private int minRowHeight, minColumnWidth;
    private int height;
//    private boolean rowHeightChanged = true;
    private boolean resizeAndRepaintEnabled = true;
    
    private ChangeListener resizeListener;
    
    private MonitorFormat formatter = MonitorFormat.DEFAULT;

// -- Life cycle : -------------------------------------------------------------
    
    public CellTableView(int rows, int columns, MonitorCell... content) {
        if (rows*columns != content.length) throw new IllegalArgumentException("Incorrect number of cell descriptors");
        nRows = rows;
        nColumns = columns;
        cells = new ArrayList<>(rows*columns);
        cells.addAll(Arrays.asList(content));
        init();
    }
    
    public void setResizeListener(ChangeListener listener) {
        this.resizeListener = listener;
    }

// -- Implementing MonitorView : -----------------------------------------------
    
    @Override
    public String getName() {
        return "";
    }
    
    @Override
    public void setName(String name) {
    }

    @Override
    public JComponent getPanel() {
        return this;
    }

    @Override
    public AgentChannelsFilter getFilter() {
        return this;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
    }

    @Override
    public Descriptor save() {
        return MonitorView.super.save();
    }

    @Override
    public void restore(Descriptor descriptor) {
        MonitorView.super.restore(descriptor);
    }
    
    
// -- Implementing AgentStatusListener : ---------------------------------------

    @Override
    public void connect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void configure(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void disconnect(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        SwingUtilities.invokeLater(() -> {
            update(event);
        });
    }
    
    /**
     * Updates the table.
     * The implementation is efficient for small tables used on image views.
     */
    private void update(AgentStatusEvent event) {
        for (AgentChannel channel : event.getAddedChannels()) {
            String path = channel.getPath();
            for (MonitorCell cell : cells) {
                for (ChannelHandle handle : cell.getChannels()) {
                    if (path.equals(handle.getPath())) {
                        handle.setChannel(channel);
                        formatter.format(cell);
                        break;
                    }
                }
            }
        }
        for (AgentChannel channel : event.getRemovedChannels()) {
            String path = channel.getPath();
            for (MonitorCell cell : cells) {
                for (ChannelHandle handle : cell.getChannels()) {
                    if (path.equals(handle.getPath())) {
                        AgentChannel ch = handle.getChannel();
                        if (ch != null) {
                            if (channel instanceof MutableAgentChannel) {
                                ((MutableAgentChannel) channel).set(AgentChannel.Key.STATE, DataProviderState.OFF_LINE);
                                handle.update(null);
                            } else {
                                handle.setChannel(null);
                            }
                            formatter.format(cell);
                        }
                        break;
                    }
                }
            }
        }
        Set<AgentChannel> set = event.getStatusChanges().keySet();
        for (AgentChannel channel : set) {
            String path = channel.getPath();
            for (MonitorCell cell : cells) {
                for (ChannelHandle handle : cell.getChannels()) {
                    if (path.equals(handle.getPath())) {
                        handle.setChannel(channel);
                        formatter.format(cell);
                        break;
                    }
                }
            }
        }
        ((Model)getModel()).fireTableDataChanged();
    }
    
    
// -- Implementing Filter : ----------------------------------------------------
    
    @Override
    public List<String> getOriginChannels() {
        return cells.stream().flatMap(cell -> cell.getChannels().stream())
                             .map(handle -> handle.getPath())
                             .collect(Collectors.toList());
    }

    @Override
    public List<String> getDisplayChannels() {
        return getOriginChannels();
    }

    @Override
    public String getOriginPath(String displayPath) {
        return displayPath.contains("/") ? displayPath : null;
    }

    @Override
    public List<String> getDisplayPath(String originPath) {
        return Collections.singletonList(originPath);
    }

    @Override
    public List<String> getFields(boolean compact) {
        return Collections.singletonList(AgentChannel.Key.VALUE);
    }
    
    
// -- Customizing JTable : -----------------------------------------------------
    
    private void init() {
        setOpaque(true);
        setBorder(BorderFactory.createLineBorder(Color.BLACK));
        setShowGrid(true);
        setRowSelectionAllowed(false);
        setColumnSelectionAllowed(false);
        setFillsViewportHeight(true);
//        MonitorTableCellRenderer renderer = new MonitorTableCellRenderer();
//        renderer.setEnsureWidth(true);
//        renderer.setEnsureHeight(true);
        Renderer renderer = new Renderer();
        setDefaultRenderer(Object.class, renderer);
        setModel(new Model());
        minRowHeight = getRowHeight();
        minColumnWidth = 0;
        Enumeration<TableColumn> en = getColumnModel().getColumns();
        while (en.hasMoreElements()) {
            TableColumn tc = en.nextElement();
            minColumnWidth = Math.max(minColumnWidth, tc.getMinWidth());
            tc.setPreferredWidth(minColumnWidth);
        }
    }
    
    public void setFormat(MonitorFormat format) {
        formatter = format;
    }
    
    /**
     * Sets {@code equalRows} flag.
     * By default, rows of this table have the same height. That means every row is sized
     * to accommodate the highest content. If this flag is set to {@code false}, 
     * each row will be sized individually. In either case, any extra height assigned to
     * this table by the layout manager will be split equally between all rows.
     * 
     * @param value Flag value.
     */
    public void setEqualRows(boolean value) {
        if (value && minRowHeights != null) {
            minRowHeight = 0;
            for (int h : minRowHeights) {
                minRowHeight = Math.max(minRowHeight, h);
            }
            minRowHeights = null;
        } else if (!value && minRowHeights == null) {
            minRowHeights = new int[nRows];
            Arrays.fill(minRowHeights, minRowHeight);
            minRowHeight *= nRows;
        }
//        rowHeightChanged = true;
    }
    
    public void setEqualColumns(boolean value) {
        if (value && minColumnWidths != null) {
            minColumnWidth = 0;
            for (int h : minColumnWidths) {
                minColumnWidth = Math.max(minColumnWidth, h);
            }
            minColumnWidths = null;
        } else if (!value && minColumnWidths == null) {
            minColumnWidths = new int[nColumns];
            Arrays.fill(minColumnWidths, minColumnWidth);
            minColumnWidth = 0;
        }
    }
    
    public boolean isEqualRows() {
        return minRowHeights == null;
    }
    
    public boolean isEqualColumns() {
        return minColumnWidths == null;
    }

    @Override
    public Dimension getPreferredSize() {
        Dimension d = super.getPreferredSize();
        if (isEqualRows()) {
            d.height = minRowHeight * nRows;
        } else {
            d.height = 0;
            for (int i=0; i<nRows; i++) {
                d.height += minRowHeights[i];
            }
        }
//        System.out.println("Table pref size: "+ d);
        return d;
    }

    @Override
    public Dimension getMinimumSize() {
//        System.out.print("Min: ");
        return getPreferredSize();
    }

    @Override
    public Dimension getMaximumSize() {
        return maxSize;
    }

    @Override
    public void setBounds(int x, int y, int width, int height) {
//        System.out.println("Table setBounds "+ x +" "+ y +" "+ width +" "+ height);
        if (/*rowHeightChanged || */height != this.height) {
//            System.out.println("Adjusting row heights...");
            resizeAndRepaintEnabled = false;
            if (isEqualRows()) {
                
                int totalHeight = minRowHeight * nRows;
                if (totalHeight < height) {
                    int h = height / nRows;
                    height = h * nRows;
                    setRowHeight(h);
                } else {
                    super.setRowHeight(minRowHeight);
                }
                
//                int totalHeight = minRowHeight * nRows;
//                if (totalHeight < height) {
//                    int h = height / nRows;
//                    int extra = height % nRows;
//                    if (extra == 0) {
//                        setRowHeight(h);
//                        System.out.println("All rows "+ h);
//                    } else {
//                        for (int row = 0; row < nRows; row++) {
//                            super.setRowHeight(row, row < extra ? h+1 : h);
//                            System.out.println("Row "+ row +" set "+ (row < extra ? h+1 : h) +" get "+ getRowHeight(row));
//                        }
//                    }
//                } else {
//                    super.setRowHeight(minRowHeight);
//                    System.out.println("Setting to "+ minRowHeight);
//                }
                
                
            } else {
                int excessHeight = height - minRowHeight;
                if (excessHeight > 0) {
                    int h = excessHeight / nRows;
                    int extra = excessHeight % nRows;
                    for (int row = 0; row < nRows; row++) {
                        super.setRowHeight(row, row < extra ? minRowHeights[row] + h + 1 : minRowHeights[row] + h);
                    }
                } else {
                    for (int row = 0; row < nRows; row++) {
                        super.setRowHeight(row, minRowHeights[row]);
                    }
                }
            }            
            this.height = height;
            resizeAndRepaintEnabled = true;
//            rowHeightChanged = false;
        }
        super.setBounds(x, y, width, height);
//        System.out.println(height +" "+ getHeight() +" "+ getRowHeight() +"["+ getRowHeight(0) +","+ getRowHeight(1) +"]");
    }
    
    @Override
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        super.addPropertyChangeListener(listener);
    }

    @Override
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        super.addPropertyChangeListener(propertyName, listener);
    }
    
    @Override
    protected void resizeAndRepaint() {
//        System.out.println("resizeAndRepaint: "+ resizeAndRepaintEnabled);
        if (resizeAndRepaintEnabled) {
            if (resizeListener != null) {
                resizeListener.stateChanged(null);
            }
            super.resizeAndRepaint();
        }
    }
    
    
// -- Auxiliary classes : ------------------------------------------------------
    
    private class Model extends AbstractTableModel {

        @Override
        public int getRowCount() {
            return nRows;
        }

        @Override
        public int getColumnCount() {
            return nColumns;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            return cells.get(rowIndex * nColumns + columnIndex).getFormattedValue();
        }
        
    } 
    
    private class Renderer extends MonitorTableCellRenderer {

        @Override
        public Component getTableCellRendererComponent(JTable table, Object value, boolean isSelected, boolean hasFocus, int row, int column) {
            super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
            boolean resized = false;
            
            int w = getMinimumSize().width + 2;
            TableColumnModel tcm = table.getColumnModel();
            TableColumn c = tcm.getColumn(column);
            if (w > c.getMinWidth()) {
                resizeAndRepaintEnabled = false;
                if (isEqualColumns()) {
                    for (int i=0; i<nColumns; i++) {
                        tcm.getColumn(i).setMinWidth(w);
                    }
                } else {
                    c.setMinWidth(w);
                }
                resizeAndRepaintEnabled = true;
                resized = true;
            }

            int h = getMinimumSize().height;
            if (isEqualRows()) {
                if (h > minRowHeight) {
                    minRowHeight = h;
                    resized = true;
                }
            } else {
                if (h > minRowHeights[row]) {
                    minRowHeight = minRowHeight - minRowHeights[row] + h;
                    minRowHeights[row] = h;
                    resized = true;
                }
            }
            
            if (resized) resizeAndRepaint();
            return this;
        }
        
    }
    
    
// -- Testing : ----------------------------------------------------------------
    
    public void test(int rowIndex, int columnIndex, String text) {
        FormattedValue fv = new FormattedValue(text, SwingUtilities.CENTER);
        cells.get(rowIndex * nColumns + columnIndex).setFormattedValue(fv);
        ((Model)getModel()).fireTableRowsUpdated(0, nRows-1);
//        ((Model)getModel()).fireTableDataChanged();
    }
    
//    static public void main(String... args) {
//        System.out.println(String.format("xxx%2sxxx%nyyy", "11"));
//    }
    
}
