package org.lsst.ccs.gconsole.plugins.commandbrowser;

import java.awt.Color;
import java.awt.Component;
import java.util.*;
import java.util.concurrent.ExecutionException;
import javax.swing.JTree;
import javax.swing.SwingWorker;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;
import org.lsst.ccs.command.CommandSet;
import org.lsst.ccs.command.Dictionary;
import org.lsst.ccs.command.DictionaryCommand;
import org.lsst.ccs.utilities.structs.TreeBranch;

/**
 * Represent the dictionary of the selected subsystem as a tree.
 * @author emarin
 */
public final class DictionaryTreePanel extends JTree {
    
    private final AgentPanel.Descriptor descriptor;
    
    private TreeBranch<ComponentDictionary> dictionaryTree;
    private AgentHandle agent;
    
    DictionaryTreePanel(AgentPanel agentPanel) {
        super();
        descriptor = agentPanel.descriptor;
        setModel(new DefaultTreeModel(dictionaryTree));
        getSelectionModel().setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
        setCellRenderer(new Renderer());
    }
    
    void updateData(AgentHandle dict){
        agent = dict;
        if (agent == null) {
            setModel(new DefaultTreeModel(null));
        } else {
            new BuildDictionaryWorker(dict).execute();
        }
    }
    
    public void updateRenderer(){
        if (descriptor.isHideUnavailableCommands()) {
            updateData(agent);
        } else {
            this.revalidate();
            this.repaint();
        }
    }
    
    public void setHideUnavailable(boolean hide) {
        if (hide != descriptor.isHideUnavailableCommands()) {
            descriptor.setHideUnavailableCommands(hide);
            updateData(agent);
        }
    }
    
    private class BuildDictionaryWorker extends SwingWorker<TreeBranch<ComponentDictionary>, String> {
        
        private final AgentHandle agentHandle;
        Map<String, Dictionary> dictionaries;
        Map<String, CommandSet> commandSets;
        
        BuildDictionaryWorker(AgentHandle dict) {
            this.agentHandle = dict;
            dictionaries = new HashMap<>(agentHandle.dictionaries);
            commandSets = agentHandle.commandSets;
        }
        
        @Override
        protected TreeBranch<ComponentDictionary> doInBackground() throws Exception {

            // Extract the command sets for the requested destination
            TreeBranch<ComponentDictionary> dictionaryTree = null;

            String rootName = agentHandle.getName();
            // First node building
            Dictionary rootDictionary = dictionaries.remove(rootName);
            dictionaryTree = new TreeBranch<>(new ComponentDictionary(rootName, rootName, rootDictionary, commandSets.get(rootName)));
            TreeBranch<ComponentDictionary> treeWalker;
            // Tree building
            for (Map.Entry<String, Dictionary> entry : dictionaries.entrySet()) {
                treeWalker = dictionaryTree;
                String[] pathTab = entry.getKey().split("/");
                boolean found;
                for (int i = 1; i < pathTab.length; i++) {
                    found = false;
                    if (treeWalker.getChildren() != null) {
                        Iterator<TreeBranch<ComponentDictionary>> it = treeWalker.getChildIterator();
                        while (it.hasNext() && !found) {
                            TreeBranch<ComponentDictionary> currIt = it.next();
                            if (currIt.getContent().name.equals(pathTab[i])) {
                                treeWalker = currIt;
                                found = true;
                                if (i == pathTab.length - 1) {
                                    // The node exists but its dictionary has not been set
                                    treeWalker.setContent(new ComponentDictionary(entry.getKey(), pathTab[i], entry.getValue(), commandSets.get(entry.getKey())));
                                }
                            }
                        }
                    }
                    if (!found) {
                        treeWalker = new TreeBranch(treeWalker, new ComponentDictionary(entry.getKey(), pathTab[i], entry.getValue(), commandSets.get(entry.getKey())));
                    }
                }
            }
            
            // remove empty branches

            removeLeafsWithEmptyDicts(dictionaryTree);
            
            // remove unavailable branches
            
            if (descriptor.isHideUnavailableCommands()) {
                removeUnavailable(dictionaryTree);
            }
            
            // sort FIXME: a hack for now, if time allows should be remove along with TreeBranch
            
            sort(dictionaryTree);
            
            return dictionaryTree;
        }
        
        private void removeLeafsWithEmptyDicts(TreeBranch<ComponentDictionary> node) {
            Iterator<TreeBranch<ComponentDictionary>> it = node.getChildIterator();
            while(it.hasNext()) {
                TreeBranch<ComponentDictionary> currIt = it.next();
                removeLeafsWithEmptyDicts(currIt);
                if(currIt.getChildCount()==0 && !currIt.getContent().dictionary.iterator().hasNext()) {
                    it.remove();
                }
            }
        }
        
        private void removeUnavailable(TreeBranch<ComponentDictionary> node) {
            Iterator<TreeBranch<ComponentDictionary>> it = node.getChildIterator();
            while(it.hasNext()) {
                TreeBranch<ComponentDictionary> currIt = it.next();
                if(Renderer.greyOut(currIt)) {
                    it.remove();
                }
            }
        }
        
        @Override
        public void done() {
            try {
                dictionaryTree = get();
                if (dictionaryTree != null){
                    setModel(new DefaultTreeModel(dictionaryTree));
                    getSelectionModel().setSelectionPath(new TreePath(getModel().getRoot()));
                }
                // revalidate();
                //repaint();
                
            } catch (InterruptedException | ExecutionException ex) {
                ex.printStackTrace();
            }
        }
        
    }
    
    static private class Renderer extends DefaultTreeCellRenderer{
        
        @Override
        public Component getTreeCellRendererComponent(JTree tree, Object value, boolean selected, boolean expanded, boolean leaf, int row, boolean hasFocus) {
            super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf, row, hasFocus);
            setForeground(greyOut((TreeBranch<ComponentDictionary>)value) ? Color.GRAY : Color.BLACK);
            return this;
        }
        
        static boolean greyOut(TreeBranch<ComponentDictionary> tree) {
            try {
                for (TreeBranch<ComponentDictionary> child : tree.getChildren()) {
                    if (!greyOut(child)) {
                        return false;
                    }
                }
                Dictionary d = tree.getContent().dictionary;
                for (DictionaryCommand dc : d) {
                    if (d.isDictionaryCommandVisible(dc)) {
                        return false;
                    }
                }
            } catch (NullPointerException x) {
            }
            return true;
        }
        
    }
    
    static private class Sorter implements Comparator<TreeBranch<ComponentDictionary>> {

        @Override
        public int compare(TreeBranch<ComponentDictionary> o1, TreeBranch<ComponentDictionary> o2) {
            boolean leaf1 = o1.isLeaf();
            boolean leaf2 = o2.isLeaf();
            if (leaf1 == leaf2) {
                String[] path1 = o1.getContent().path.split("/");
                String[] path2 = o2.getContent().path.split("/");
                int i = 0;
                while (true) {
                    if (i == path1.length) {
                        return (i == path2.length) ? 0 : -1;
                    } else if (i == path2.length) {
                        return 1;
                    } else {
                        int out = path1[i].compareTo(path2[i]);
                        if (out == 0) {
                            i++;
                        } else {
                            return out;
                        }
                    }
                }
            } else {
                return leaf1 ? 1 : -1;
            }
            
        }
        
    }
    
    // FIXME: a hack for now, if time allows should be remove along with TreeBranch
    private void sort(TreeBranch<ComponentDictionary> node) {
        if (!node.isLeaf()) {
            ArrayList<TreeBranch<ComponentDictionary>> children = (ArrayList<TreeBranch<ComponentDictionary>>) node.getChildren();
            children.sort(new Sorter());
            children.forEach(child -> sort(child));
        }
    }
    
}