package org.lsst.ccs.gconsole.agent.filter;

import java.util.*;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;

/**
 * Regular expression based channels filter.
 * The filter selects channels that match a regular expression and leaves the paths unchanged.
 *
 * @author onoprien
 */
public class RegExFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final Pattern pattern;
    private final boolean wholePath;

// -- Life cycle : -------------------------------------------------------------
    
    @Create(category = "AgentChannelsFilter",
            name = "Regular expression filter",
            path = "Built-In/Regular Expression",
            description = "Channels filter that uses a regular expression to select channels and leaves the path unchanged.")
    public RegExFilter(
            @Par(def = Par.NULL, desc = "Filter name. Optional.") String name,
            @Par(def = Par.NULL, desc = "Regular expression to be applied tochanel path.") String regex,
            @Par(def = "true", desc = "Match the whole path.") boolean wholePath)
    {
        this.name = name;
        pattern = Pattern.compile(regex);
        this.wholePath = wholePath;
    }
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getDisplayPath(String originPath) {
        boolean accept;
        if (wholePath) {
            accept = pattern.matcher(originPath).matches();
        } else {
            accept = pattern.matcher(originPath).find();
        }
        return accept ? Collections.singletonList(originPath) : Collections.emptyList();
    }

    @Override
    public String getOriginPath(String displayPath) {
        return displayPath;
    }
    
}
