package org.lsst.ccs.gconsole.services.persist;

import java.io.Serializable;

/**
 * A component that can be persisted between console sessions using {@link PersistenceService}.
 * 
 * The {@code save()} and {@code restore()} methods save/restore the component state to/from a JavaBean.
 * Exactly what is being saved and restored is up to a specific component - in general, one should not
 * expect that saving and then restoring a component will result in an identical object.
 * 
 * The {@code getDescriptor()} method provides access to the descriptor maintained by this component.
 * Implementations are expected to ensure that any properties set on the descriptor returned by this
 * method are reflected by descriptors subsequently returned by {@code save()}, unless modified
 * between the two calls.
 *
 * @author onoprien
 */
public interface Persistable extends Savable {

    /**
     * Overrides {@code Savable.restore(Serializable)} to forward the call to {@link #restore(Descriptor)}
     * if the provided argument value is of type {@link Descriptor}, and do nothing otherwise.
     * 
     * @param descriptor Descriptor.
     */
    @Override
    public default void restore(Serializable descriptor) {
        if (descriptor instanceof Descriptor) {
            restore((Descriptor) descriptor);
        }
    }
    
    /**
     * Restores this component to the state specified by the provided JavaBean.
     * The default implementation does nothing.
     * 
     * @param descriptor JavaBean encapsulating the state.
     */
    default void restore(Descriptor descriptor) {
    }
    
    /**
     * Returns JavaBean describing the current state of this component.
     * The default implementation forwards the call to {@code getDescriptor()}.
     * 
     * @return JavaBean encapsulating the state, or {@code null} if no state information needs to be saved.
     */
    @Override
    default Descriptor save() {
        return getDescriptor();
    }
    
    /**
     * Returns a reference to the descriptor maintained by this component.
     * Implementations are expected to ensure that this method never returns {@code null}, and
     * that any properties set on the descriptor returned by this method are reflected by
     * descriptors subsequently returned by {@code save()}, unless modified between the two calls.
     * 
     * @return JavaBean describing this component.
     */
    Descriptor getDescriptor();
    
    /**
     * Returns the category this {@code Persistable} belongs to.
     * The default implementation extracts the category from the descriptor.
     * @return Category of this {@code Persistable}.
     */
    default String getCategory() {
        Descriptor descriptor = getDescriptor();
        if (descriptor == null) return null;
        String cat = descriptor.getCategory();
        if (cat == null) {
            Creator.Descriptor desc = descriptor.getCreator();
            if (desc != null) {
                cat = desc.getCategory();
            }
        }
        return cat;
    }
    
    /**
     * Returns an {@code Editor} capable of modifying this {@code Persistable}.
     * The default implementation returns {@code null}.
     * 
     * @return {@code Editor} for this {@code Persistable}, or {@code null} is there is none.
     */
    default Editor getEditor() {
        return null;
    }
    
    
// -- Descriptor class : -------------------------------------------------------
    
    /**
     * JavaBean that describes a state of a {@link Persistable} component.
     * Provides a property that describes {@link Creator} invocation, as well as
     * properties used when saving or loading a {@code Persistable} component.
     * All classes implementing {@code Persistable} should use descriptors that extend this class.
     */
    static public class Descriptor implements Serializable {
        
        // -- Properties :

        private Creator.Descriptor creator;
        private String name;
        private String category;
        private String path;
        private String description;
        
        // -- Convenience methods :
        
        public void saveOrigin(Descriptor other) {
            if (other == null) return;
            setCreator(other.getCreator());
            setName(other.getName());
            setCategory(other.getCategory());
            setPath(other.getPath());
            setDescription(other.getDescription());
        }
        
        // -- Property setters/getters :

        public Creator.Descriptor getCreator() {
            return creator;
        }

        public void setCreator(Creator.Descriptor creator) {
            this.creator = creator;
        }

        public String getName() {
            return name;
        }

        public void setName(String name) {
            this.name = name;
        }

        public String getCategory() {
            return category;
        }

        public void setCategory(String category) {
            this.category = category;
        }

        public String getPath() {
            return path;
        }

        public void setPath(String path) {
            this.path = path;
        }

        public String getDescription() {
            return description;
        }

        public void setDescription(String description) {
            this.description = description;
        }
        
    }
    
}
