package org.lsst.ccs.gconsole.services.persist;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Parameter;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.base.Console;
import static org.lsst.ccs.gconsole.base.Const.*;
import org.lsst.ccs.gconsole.util.tree.SModel;
import org.lsst.ccs.gconsole.util.tree.STree;

/**
 * Dialog that allows selecting a class name and instantiating an object.
 *
 * @author onoprien
 */
public final class CreationDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private Persistable out;
    private Creator factory;
    private Creator.Descriptor descriptor;
    
    private final Box rightPanel;
    private final JButton okButton;
    private List<ParPanel> parPanels;
    
    private final Pattern fullToSimpleClassName = Pattern.compile("(\\w+\\.)+");

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor.
     *
     * @param path Path to be selected when the dialog is first displayed, or {@code null} if no {@code Creator} should be selected.
     * @param parameters String representations of the initial values of parameters.
     * @param parentComponent Owner component.
     * @param title Dialog title.
     * @param factories List of {@code Creator} factories the user can select from.
     */
    public CreationDialog(String path, String[] parameters, String title, Component parentComponent, Collection<? extends Creator> factories) {
        super(parentComponent == null ? Console.getConsole().getWindow() : SwingUtilities.getWindowAncestor(parentComponent), title, Dialog.ModalityType.APPLICATION_MODAL);

        Map<String,Creator> path2factory = factories.stream().collect(Collectors.toMap(f -> f.getPath(), f -> f));
        SModel<Creator> m = new SModel<>(path2factory);
        STree<Creator> tree = new STree<>(m);
        tree.expandLevel(3);
        tree.setShowsRootHandles(true);
        JScrollPane treeScrollPane = new JScrollPane(tree);

        rightPanel = Box.createVerticalBox();
        JScrollPane rightScrollPane = new JScrollPane(rightPanel);
 
        JSplitPane splitPane = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, treeScrollPane, rightScrollPane);
        splitPane.setDividerLocation(300);
        Dimension minimumSize = new Dimension(100, 50);
        treeScrollPane.setMinimumSize(minimumSize);
        rightScrollPane.setMinimumSize(minimumSize);
        splitPane.setPreferredSize(new Dimension(850, 500));
        add(splitPane, BorderLayout.CENTER);
        
        Box buttonBox = Box.createHorizontalBox();
        buttonBox.setBorder(BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE));
        buttonBox.add(Box.createHorizontalGlue());
        JButton b = new JButton("Cancel");
        b.addActionListener(e -> {
            out = null;
            factory = null;
            descriptor = null;
            dispose();
        });
        buttonBox.add(b);
        buttonBox.add(Box.createRigidArea(HDIM));
        okButton = new JButton("OK");
        okButton.addActionListener(e -> {
            if (factory != null) {
                int n = parPanels.size();
                Object[] pars = new Object[n];
                boolean parsValid = true;
                for (int i=0; i<n; i++) {
                    try {
                        pars[i] = parPanels.get(i).getValue();
                    } catch (IllegalStateException x) {
                        parsValid = false;
                    }
                }
                if (parsValid) {
                    try {
                        out = factory.make(pars);
                        dispose();
                    } catch (Exception x) {
                        Console.getConsole().error("Unable to create", x);
                    }
                }
            }
        });
        buttonBox.add(okButton);
        buttonBox.add(Box.createRigidArea(HDIM));
        add(buttonBox, BorderLayout.SOUTH);
        
        try {
            Creator f = path2factory.get(path);
            tree.setSelectedPath(path);
            showFactory(f, parameters);
        } catch (NullPointerException x) {
            showFactory(null, null);
        }
                
        tree.addTreeSelectionListener(e -> {
            showFactory(tree.getSelectedUserObject(), null);
        });
    }
    
    
// -- Static methods for displaying dialog and returning created instance : ----
    
    static public Persistable show(String path, String[] parameters, String title, Component parentComponent, Collection<? extends Creator> factories) {
        CreationDialog dialog = new CreationDialog(path, parameters, title, parentComponent, factories);
        dialog.setSize(dialog.getPreferredSize());
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        return dialog.out;
    }
    
    static public Persistable show(Creator.Descriptor desc, String title, Component parentComponent, Collection<? extends Creator> factories) {
        String path;
        String[] parameters;
        if (desc == null) {
            path = null;
            parameters = null;
        } else {
            path = desc.getPath();
            parameters = desc.getParameters();
        }
        return show(path, parameters, title, parentComponent, factories);
    }
    
    static public Persistable show(Persistable.Descriptor descriptor, String title, Component parentComponent, Collection<? extends Creator> factories) {
        if (descriptor == null) {
            return show(null, null, title, parentComponent, factories);
        } else {
            String path = descriptor.getPath();
            if (path == null) {
                return show(descriptor.getCreator(), title, parentComponent, factories);
            } else {
                return show(path, null, title, parentComponent, factories);
            }
        }
    }


// -- Getters : ----------------------------------------------------------------
    
    public Creator.Descriptor getDescriptor() {
        return descriptor;
    }
    

// -- Local methods and classes : ----------------------------------------------
        
    private void showFactory(Creator fact, String[] parameters) {
        this.factory = fact;
        rightPanel.removeAll();
        parPanels = Collections.emptyList();
        if (fact != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("<html>").append(fact.getPath()).append("<br>");
            sb.append("<h3>").append(fact.getName()).append("</h3>");
            String descr = fact.getDescription();
            if (descr != null && !descr.isEmpty()) {
                sb.append(descr).append("<p>");
            }
            sb.append("Enter parameters, if any, the same way you would in a CCS command (lists in square brackets, comma-separated).");
            sb.append("</html>");
            JEditorPane ep = new DescriptionPane(sb.toString());
            ep.setEditable(false);
            rightPanel.add(ep);
            
            Parameter[] pars = fact.getParameters();
            parPanels = new ArrayList<>(pars.length);
            if (pars.length > 0) {
                if (parameters != null && parameters.length != pars.length) {
                    parameters = null;
                }
                Box parBox = Box.createVerticalBox();
                parBox.setBorder(BorderFactory.createTitledBorder("Parameters"));
                parBox.add(Box.createRigidArea(VDIM));
                for (int i=0; i<pars.length; i++) {
                    ParPanel panel = new ParPanel(pars[i], parameters == null ? null : parameters[i]);
                    parBox.add(panel);
                    parPanels.add(panel);
                    parBox.add(Box.createRigidArea(new Dimension(HSPACE, 2*VSPACE)));
                }
                rightPanel.add(Box.createRigidArea(VDIM));
                rightPanel.add(parBox);
                rightPanel.add(Box.createRigidArea(VDIM));
            }
            
            rightPanel.add(Box.createVerticalGlue());
            rightPanel.revalidate();
            okButton.setEnabled(true);
        } else {
            okButton.setEnabled(false);
        }
    }
    
    private final class ParPanel extends JPanel {
        
        private final Parameter par;
        private JComponent selector;
        private String stringValue;
        
        ParPanel(Parameter parameter, String value) {
            
            par = parameter;
            
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setBorder(BorderFactory.createCompoundBorder(
                    BorderFactory.createLineBorder(Color.DARK_GRAY),
                    BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
            
            Box hBox;
            Par an = par.getAnnotation(Par.class);
            if (an != null) {
                hBox = Box.createHorizontalBox();
                JEditorPane descrPanel = new DescriptionPane(an.desc());
                hBox.add(Box.createRigidArea(HDIM));
                hBox.add(descrPanel);
                add(hBox);
                add(Box.createRigidArea(VDIM));
                if (value == null) {
                    String def = an.def();
                    if (!Par.NODEF.equals(def)) value = def;
                }
            }
            
            hBox = Box.createHorizontalBox();
            add(hBox);
            hBox.add(Box.createRigidArea(HDIM));
            String s = par.getParameterizedType().getTypeName() +" "+ par.getName();
            s = fullToSimpleClassName.matcher(s).replaceAll("");
            JLabel label = new JLabel(s);
            Font f = label.getFont();
            label.setFont(f.deriveFont(f.getStyle() | Font.BOLD));
            hBox.add(label);
            hBox.add(Box.createRigidArea(HDIM));
            Class<?> pc = par.getType();
            if (Enum.class.isAssignableFrom(pc)) {
                try {
                    Object[] enumValues = (Object[])pc.getMethod("values").invoke(null);
                    Object[] comboValues = new Object[enumValues.length+1];
                    comboValues[0] = null;
                    System.arraycopy(enumValues, 0, comboValues, 1, enumValues.length);
                    JComboBox comboBox = new JComboBox(comboValues);
                    if (value != null) {
                        try {
                            Object v = Creator.decode(value, par.getParameterizedType());
                            comboBox.setSelectedItem(v);
                        } catch (Exception x) {
                        }
                    }
                    selector = comboBox;
                } catch (IllegalAccessException | IllegalArgumentException | NoSuchMethodException | SecurityException | InvocationTargetException t) {
                    selector = new JLabel("<html><font color=red>Failed to create selector for "+ pc);
                }                
            } else if (boolean.class.equals(pc) || Boolean.class.equals(pc)) {
                JCheckBox checkBox = new JCheckBox();
                if ("true".equalsIgnoreCase(value)) {
                    checkBox.setSelected(true);
                }
                selector = checkBox;
            } else {
                JTextField field = new JTextField();
                field.setColumns(30);
                field.setMaximumSize(field.getPreferredSize());
                if (value != null) field.setText(value);
                selector = field;
            }
            hBox.add(selector);
            hBox.add(Box.createHorizontalGlue());
        }
        
        Object getValue() {
            Object out = null;
            java.lang.reflect.Type type = par.getParameterizedType();
            try {
                if (selector instanceof JTextField) {
                    JTextField field = (JTextField) selector;
                    out = Creator.decode(field.getText(), type);
                } else if (selector instanceof JCheckBox) {
                    JCheckBox b = (JCheckBox) selector;
                    out = b.isSelected();
                } else if (selector instanceof JComboBox) {
                    JComboBox cb = (JComboBox) selector;
                    out = cb.getSelectedItem();
                }
                setBorder(BorderFactory.createCompoundBorder(
                        BorderFactory.createLineBorder(Color.DARK_GRAY),
                        BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
                stringValue = Creator.encode(out, type);
            } catch (Throwable t) {
                setBorder(BorderFactory.createCompoundBorder(
                        BorderFactory.createLineBorder(Color.RED),
                        BorderFactory.createEmptyBorder(VSPACE, HSPACE, VSPACE, HSPACE)));
                stringValue = null;
                throw new IllegalStateException(t);
            }
            return out;
        }
        
        String getStringValue() {
            return stringValue;
        }
        
    }
    
    private final class DescriptionPane extends JEditorPane {
        
        DescriptionPane(String s) {
            super("text/html", s);
            setEditable(false);
            setOpaque(true);
            setBackground(new Color(255,255,255,0));
        }

        @Override
        public Dimension getPreferredSize() {
            Dimension d = new Dimension(super.getPreferredSize());
//            d.width = rightPanel.getWidth() - 2*HSPACE;
            d.width = 200;
            return d;
        }
        
    }

}
