package org.lsst.ccs.gconsole.services.persist;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.*;
import java.util.stream.Collectors;
import javax.swing.JComponent;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.ConsoleService;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;

/**
 * Service that facilitates persisting data between the graphical console sessions.
 *
 * @author onoprien
 */
@Plugin(name = "Persistence Service Plugin",
        id="persistence-service",
        description = "Service that facilitates persisting data between the graphical console sessions.")
public class PersistenceService extends ConsolePlugin implements ConsoleService {

// -- Fields : -----------------------------------------------------------------
    
    private final ArrayList<Creator> factories = new ArrayList<>();

// -- Life cycle : -------------------------------------------------------------
    
    @Override
    public void startService() {
        try {
            ClassLoader loader = getClass().getClassLoader();
            Enumeration<URL> e = loader.getResources(Create.RESOURCE);
            while (e.hasMoreElements()) {
                URL url = e.nextElement();
                try (
                        InputStream ins = url.openStream();
                        BufferedReader in = new BufferedReader(new InputStreamReader(ins));
                    ) 
                {
                    String line;
                    while ((line = in.readLine()) != null) {
                        try {
                            Class<?> clazz = Class.forName(line.trim());
                            for (Method m : clazz.getDeclaredMethods()) {
                                int mod = m.getModifiers();
                                if (Modifier.isPublic(mod) && Modifier.isStatic(mod)) {
                                    Create an = m.getAnnotation(Create.class);
                                    if (an != null) {
                                        factories.add(new CreatorExecutable(m, null));
                                    }
                                }
                            }
                            for (Constructor<?> c : clazz.getConstructors()) {
                                int mod = c.getModifiers();
                                if (Modifier.isPublic(mod)) {
                                    Create an = c.getAnnotation(Create.class);
                                    if (an != null) {
                                        factories.add(new CreatorExecutable(c, null));
                                    }
                                }
                            }
                        } catch (ClassNotFoundException | SecurityException x) {
                        }
                    }
                } catch (IOException x) {
                }
            }
        } catch (IOException x) {
        }
        System.out.println("Factories found: "+ factories.size());
    }

// -- Creating instances : -----------------------------------------------------
    
    /**
     * Creates a {@code Persistable} instance based on the provided descriptor.
     * No user interaction. An instance of {@code Persistable} is constructed by 
     * the {@link Creator} identified in the descriptor, then its {@code restore(...)}
     * method is called.
     * 
     * @param descriptor JavaBean that describes the {@code Persistable} type and state.
     * @return Newly created {@code Persistable}, or {@code null} if it cannot be created.
     */
    public Persistable make(Persistable.Descriptor descriptor) {
        Creator.Descriptor desc = descriptor.getCreator();
        if (desc == null) return null;
        Creator factory = getFactory(desc.getCategory(), desc.getPath());
        if (factory == null) return null;
        Persistable out = null;
        try {
            out = factory.make(desc.getParameters());
        } catch (Exception x) {
        }
        if (out == null) return null;
        Persistable.Descriptor d = out.getDescriptor();
        d.setCategory(descriptor.getCategory());
        d.setPath(descriptor.getPath());
        d.setName(descriptor.getName());
        d.setDescription(descriptor.getDescription());
        out.restore(descriptor);
        return out;
    }
    
    /**
     * Creates a {@code Persistable} instance by letting the user to select one of
     * the available factories in the given category, and using the provided
     * descriptor (if any) as the starting point.
     * 
     * @param descriptor Object state to be used as a starting point.
     * @param title User-interaction dialog title, or {@code null} if the default title should be used.
     * @param parent Graphical component to be use as a parent for user-interaction dialog(s).
     * @param category Object category.
     * @return Newly created {@code Persistable}, or {@code null} if the object creation
     *         was canceled by the user or failed for other reason.
     */
    public Persistable make(Persistable.Descriptor descriptor, String title, JComponent parent, String category) {
        ArrayList<Creator> ff = getSaved(category);
        ff.addAll(getFactories(category));
        return CreationDialog.show(descriptor, title, parent, ff);
    }
    
    public void save(Persistable.Descriptor descriptor) {
        throw new UnsupportedOperationException(); // FIXME
    }
    
    public void saveAs(Persistable.Descriptor descriptor, JComponent parent) {
        throw new UnsupportedOperationException(); // FIXME
    }
    
    
// -- Utilities : --------------------------------------------------------------
    
    /**
     * Returns a factory identified by the provided category and path.
     * 
     * @param category Creator category.
     * @param path Creator path.
     * @return Desired factory, or {@code null} if no such factory is known to this service.
     */
    public Creator getFactory(String category, String path) {
        for (Creator f : factories) {
            if (path.equals(f.getPath())) {
                if (category == null || category.equals(f.getCategory())) {
                    return f;
                }
            }
        }
        return null;
    }
    
    public List<Creator> getFactories(String category) {
        if (category == null) {
            return new ArrayList<>(factories);
        } else {
            return factories.stream().filter(f -> category.equals(f.getCategory())).collect(Collectors.toList());
        }
    }
    
    public List<Persistable.Descriptor> getSavedDescriptors(String category) {
        return Collections.emptyList(); // FIXME
    }
    
    public ArrayList<Creator> getSaved(String category) {
        List<Persistable.Descriptor> dd = getSavedDescriptors(category);
        ArrayList<Creator> out = new ArrayList<>(dd.size());
        for (Persistable.Descriptor d : dd) {
            // FIXME: chack that the referenced factory exists
            out.add(new CreatorDescriptor(d));
        }
        return out;
    }
    
}
