package org.lsst.ccs.gconsole.plugins.trending;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.util.*;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JPopupMenu;
import javax.swing.SwingUtilities;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageManager;
import org.freehep.jas.plugin.tree.DefaultFTreeNodeAdapter;
import org.freehep.jas.plugin.tree.FTree;
import org.freehep.jas.plugin.tree.FTreeNode;
import org.freehep.jas.plugin.tree.FTreeNodeAddedNotification;
import org.freehep.jas.plugin.tree.FTreeNodeRemovedNotification;
import org.freehep.jas.plugin.tree.FTreeNodeTransferable;
import org.freehep.jas.plugin.tree.FTreePath;
import org.freehep.jas.plugin.tree.FTreeProvider;
import org.freehep.jas.services.PlotRegionDropHandler;
import org.freehep.jas.services.Plotter;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelType;
import org.lsst.ccs.gconsole.jas3.JasPanelManager;
import org.openide.util.Lookup;
import org.openide.util.LookupListener;

/**
 * Manages trending sources found in the lookup, opens and maintains channel tree panel.
 *
 * @author onoprien
 */
class TrendingSourcesManager extends DefaultFTreeNodeAdapter implements TrendingSource.Listener {

// -- Fields : -----------------------------------------------------------------
    
    private final LsstTrendingPlugin plugin;

    private final ChannelDropHandler dropHandler;
    
    private volatile TrendingChannel selectedChannel;
    private final Action actShow, actPlot, actOverlay, actNewPlot, actNewPage, actRefresh;

    private final CompoundTrendingSource rawSource = new CompoundTrendingSource();
    private final CompoundTrendingSource filteredSource = new CompoundTrendingSource();
    private final Object sourceLock = new Object();
    
    private volatile Lookup.Result sourcesLookupResult, filtersLookupResult;
    private volatile LookupListener sourcesLookupListener, filtersLookupListener;
    
    private FTree tree;
//    private Component treeComponent;
//    private final String TREE_TITLE = "Trending";

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs {@code TrendingSourcesManager}.
     * Called on EDT.
     * @param plugin Trending plugin that uses this sources manager.
     */
    TrendingSourcesManager(LsstTrendingPlugin plugin) {
        super(100);
        
        this.plugin = plugin;
        dropHandler = new ChannelDropHandler(plugin);
        
        actShow = new AbstractAction("Show") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.plot(selectedChannel, null, TrendPlotter.Option.EXIST, TrendPlotter.Option.NEWPAGE);
            }
        };
        actShow.putValue("SHORT_DESCRIPTION", "Plot in the region previously used for this channel, if any, or on a new page.");
        
        actPlot = new AbstractAction("Plot") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.plot(selectedChannel, null);
            }
        };
        actPlot.putValue("SHORT_DESCRIPTION", "Plot in the currently selected region.");
        
        actOverlay = new AbstractAction("Overlay") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.plot(selectedChannel, null, TrendPlotter.Option.OVERLAY);
            }
        };
        actOverlay.putValue("SHORT_DESCRIPTION", "Overlay on the currently selected plot.");
        
        actNewPlot = new AbstractAction("New Plot") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.plot(selectedChannel, null, TrendPlotter.Option.NEWPLOT);
            }
        };
        actNewPlot.putValue("SHORT_DESCRIPTION", "Plot in a new region.");
        
        actNewPage = new AbstractAction("New Page") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.plot(selectedChannel, null, TrendPlotter.Option.NEWPAGE);
            }
        };
        actNewPage.putValue("SHORT_DESCRIPTION", "Plot on a new page.");
        
        actRefresh = new AbstractAction("Refresh") {
            @Override
            public void actionPerformed(ActionEvent e) {
                plugin.refresh(selectedChannel);
            }
        };
        actRefresh.putValue("SHORT_DESCRIPTION", "Refresh plots for this channel.");
        
    }
    
    /**
     * Starts {@code TrendingSourcesManager}.
     * Called on EDT.
     */
    void start() {
        
        // Create tree and register myself as node adapter
        
        if (tree == null) {
            FTreeProvider treeProvider = (FTreeProvider) plugin.getConsole().getConsoleLookup().lookup(FTreeProvider.class);
            tree = treeProvider.tree("Trending");
            JasPanelManager panMan = (JasPanelManager) plugin.getConsole().getPanelManager();
            PageContext context = panMan.findPage(PanelType.CONTROL, "Trending");
            if (context != null) {
                EnumMap<Panel,Object> prop = new EnumMap(Panel.class);
                prop.put(Panel.TYPE, PanelType.CONTROL);
                prop.put(Panel.GROUP, plugin.PANEL_GROUP);
                panMan.open(context.getPage(), prop);
            }
            treeProvider.treeNodeAdapterRegistry().registerNodeAdapter(this, TrendingChannel.class);
        } 
//        else if (treeComponent != null) {
//            ((Studio)Application.getApplication()).getControlManager().openPage(treeComponent, TREE_TITLE, null); // FIXME: once FTree plugin is updated, tree should be deleted, not hidden
//        }
        
        // Start listening to filtered source
        
        filteredSource.add(rawSource);
        filteredSource.addListener(this);
        
        // Start watching lookup for trending sources and filters
        
        if (sourcesLookupResult == null) {
            FreeHEPLookup lookup = plugin.getConsole().getConsoleLookup();
            Lookup.Template template = new Lookup.Template(TrendingSource.class);
            sourcesLookupResult = lookup.lookup(template);
        }
        sourcesLookupListener = e -> updateSources(sourcesLookupResult.allInstances());
        sourcesLookupResult.addLookupListener(sourcesLookupListener);
        updateSources(sourcesLookupResult.allInstances());
        
        // Start watching lookup for trending filters
        
        if (filtersLookupResult == null) {
            FreeHEPLookup lookup = plugin.getConsole().getConsoleLookup();
            Lookup.Template template = new Lookup.Template(TrendingFilter.class);
            filtersLookupResult = lookup.lookup(template);
        }
        filtersLookupListener = e -> applyFilters(filtersLookupResult.allInstances());
        filtersLookupResult.addLookupListener(filtersLookupListener);
        applyFilters(filtersLookupResult.allInstances());
        
    }
    
    /**
     * Stops {@code TrendingSourcesManager}.
     * Called on EDT.
     */
    void stop() {
        
        // Stop watching lookup
        
        sourcesLookupResult.removeLookupListener(sourcesLookupListener);
        sourcesLookupListener = null;
        
        filtersLookupResult.removeLookupListener(filtersLookupListener);
        filtersLookupListener = null;
        
        // Clear sources.
        
        filteredSource.clear();
        filteredSource.removeListener(this);
        rawSource.clear();
        
        selectedChannel = null;
        
    }

    
// -- Fetching data from the source : ------------------------------------------
    
    public TrendData fetchData(Trend trend, EnumSet<Trend.Meta> metadata) {
        return fetchData(Collections.singletonList(trend), metadata).get(0);
    }
    
    public List<TrendData> fetchData(List<Trend> trends, EnumSet<Trend.Meta> metadata) {
        long now = System.currentTimeMillis();
        List<TrendData> out = new ArrayList<>(trends.size());
        trends.forEach(trend -> {
            trend.setLoading(true);
            out.add(filteredSource.get(trend.getChannel(), trend.getTimeWindow().getLowerEdge(now), trend.getTimeWindow().getUpperEdge(now), metadata, trend.getData()));
        });
        return out;
    }
    
    public ArrayList<TrendingChannel> getChannels() {
        return new ArrayList<>(filteredSource.getChannels());
    }


// -- Modifying the channel tree : ---------------------------------------------
    
    /**
     * Responds to changes in the filtered channels set.
     * @param event Event fired by the filtered trending source.
     */
    @Override
    public void processEvent(TrendingSource.Event event) {
        if (SwingUtilities.isEventDispatchThread()) {
            event.getRemovedChannels().forEach(channel -> {
                FTreePath path = new FTreePath(channel.getPath());
                while (path != null) {
                    try {
                        tree.treeChanged(new FTreeNodeRemovedNotification(this, path));
                    } catch (NullPointerException x) {}
                    path = path.getParentPath();
                    if (path != null) {
                        FTreeNode node = tree.findNode(path);
                        if (node == null || node.parent() == null || !node.childNodes().isEmpty()) {
                            path = null;
                        }
                    }
                }
            });
            event.getAddedChannels().forEach(channel -> {
                tree.treeChanged(new FTreeNodeAddedNotification(this, channel.getPath(), channel));
            });
            if (plugin.getPreferences().isAutoSave() && !tree.root().childNodes().isEmpty()) {
                ComponentDescriptor desc = plugin.getServices().getDescriptor().getConfiguration();
                if (desc != null) {
                    plugin.getServices().getDescriptor().setConfiguration(null);
                    plugin.restore(desc, true);
                }                
            }
        } else {
            SwingUtilities.invokeLater(() -> processEvent(event));
        }
    }


// -- Handling filters and sources: --------------------------------------------
        
    private void updateSources(Collection<TrendingSource> sources) {
        List<TrendingSource> ss = new ArrayList<>(sources);
        synchronized (sourceLock) {
            rawSource.set(ss);
        }
    }
    
    private void applyFilters(Collection<TrendingFilter> filters) {
        synchronized (sourceLock) {
            if (filters.isEmpty()) {
                filteredSource.set(rawSource);
            } else {
                List<TrendingSource> sources = new ArrayList<>(filters.size());
                filters.forEach(filter -> sources.add(filter.filter(rawSource)));
                filteredSource.set(sources);
            }
        }
    }
    
// -- Implementing FTreeNodeAdapter : ------------------------------------------

    @Override
    public JPopupMenu modifyPopupMenu(FTreeNode[] nodes, JPopupMenu menu) {
        selectedChannel = (TrendingChannel) nodes[0].objectForClass(TrendingChannel.class);
        if (menu.getSubElements().length != 0) menu.addSeparator();
        menu.add(actShow);
        menu.add(actPlot);
        menu.add(actOverlay);
        menu.add(actNewPlot);
        menu.add(actNewPage);
        menu.add(actRefresh);
        menu.addSeparator();
        return menu;
    }

    @Override
    public boolean doubleClick(FTreeNode node) {
        selectedChannel = (TrendingChannel) node.objectForClass(TrendingChannel.class);
        switch (plugin.getPreferences().getDoubleClick()) {
            case -1:
                actShow.actionPerformed(null); break;
            case -2:
                actPlot.actionPerformed(null); break;
            case -3:
                actNewPlot.actionPerformed(null); break;
            case -4:
                actNewPage.actionPerformed(null); break;
            default:
                actShow.actionPerformed(null);
        }
        return true;
    }

    @Override
    public FTreeNodeTransferable modifyTransferable(FTreeNode[] nodes, FTreeNodeTransferable transferable) {
        transferable.addDataForClass(PlotRegionDropHandler.class, dropHandler);
        transferable.addDataForClass(Plotter.class, DummyPlotter.INSTANCE);
        return transferable;
    }


// -- Dummy plotter for hoodwinking DefaultPage.DefaultRegion into accepting drop : ---
    
    static class DummyPlotter implements Plotter {
        static final DummyPlotter INSTANCE = new DummyPlotter();
        @Override
        public void plot(Object data, int mode) {
        }
        @Override
        public void plot(Object data, int mode, Object style, String options) {
        }
        @Override
        public void remove(Object data) {
        }
        @Override
        public void clear() {
        }
        @Override
        public Component viewable() {
            return null;
        }
        @Override
        public List getData() {
            return null;
        }
    }

}
