package org.lsst.ccs.gconsole.plugins.rest;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import java.util.*;
import java.util.concurrent.locks.ReentrantLock;
import org.freehep.util.FreeHEPLookup;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;

/**
 * Plugin that creates WebResource for retrieving data from the REST server.
 * The resource can looked up by other plugins using "dataserver" key.
 * 
 * @author tonyj
 */
@Plugin(name="LSST Rest Plugin",
        id="rest",
        description="Plugin that handles connection to the REST server.")
public class LsstRestPlugin extends ConsolePlugin {
    
// -- Fields : -----------------------------------------------------------------
    
    public static final String SERVER_PROPERTY = "server";
    public static final String PORT_PROPERTY = "port";
    public static final String CONNECT_ON_STARTUP_PROPERTY = "connectOnStartup";

    private final Client client;
    private WebResource service;
    
    private final ReentrantLock lock = new ReentrantLock();  // held while (re)connecting
    
// -- Life cycle : -------------------------------------------------------------
    
    public LsstRestPlugin() {
        client = Client.create(new DefaultClientConfig());
    }

    @Override
    public void initialize() {
        
        getServices().addProperty(SERVER_PROPERTY, "localhost");
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, "URL", "Server: ${"+ SERVER_PROPERTY +"#history=5}");
        getServices().addProperty(PORT_PROPERTY, 8080);
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, "URL", "Port: ${"+ PORT_PROPERTY +"}");
        
        getServices().addProperty(CONNECT_ON_STARTUP_PROPERTY, true);
        getServices().addPreference(new String[] {"LSST","Servers","REST"}, null, "${"+ CONNECT_ON_STARTUP_PROPERTY +"} Connect on startup.");
    }

    @Override
    public void start() {
        if ((Boolean) getServices().getProperty(CONNECT_ON_STARTUP_PROPERTY)) {
            refreshConnection();
        }
    }

    @Override
    public void shutdown() {
        client.destroy();
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    @Override
    public void propertiesChanged(Object source, Map<String, Object> changes) {
        Set<String> keys = changes.keySet();
        if (keys.contains(SERVER_PROPERTY) || keys.contains(PORT_PROPERTY)) {
            refreshConnection();
        }
    }
    
    public void refreshConnection() {
        if (lock.isLocked()) return;
        Thread t = new Thread(this::connect, "REST Connector");
        t.setDaemon(true);
        t.start();
    }

    private void connect() {
        if (lock.tryLock()) {
            try {
                final FreeHEPLookup lookup = getConsole().getConsoleLookup();
                if (service != null) {
                    lookup.remove(service, "dataserver");
                    service = null;
                }
                String restFullUrl = getURL();
                getConsole().getLogger().fine("Connecting to REST server " + restFullUrl);
                WebResource serv = client.resource(restFullUrl).path("dataserver");
                try {
                    serv.head();
                    lookup.add(serv, "dataserver");
                    this.service = serv;
                    getConsole().getLogger().fine("REST server connection successful.");
                } catch (RuntimeException e) {
                    getConsole().getLogger().warn("Unable to connect to REST server at " + restFullUrl, e);
                }
            } finally {
                lock.unlock();
            }
        }
    }

    private String getURL() {
        String server = null;
        try {
            server = getServices().getProperty(SERVER_PROPERTY).toString();
            int port = (Integer) getServices().getProperty(PORT_PROPERTY);
            return String.format("http://%s:%d/rest/data", server, port);
        } catch (RuntimeException x) {
            getConsole().getLogger().warn("Invalid server "+ server +" or port "+ getServices().getProperty(PORT_PROPERTY));
            throw x;
        }
    }

}
