package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import java.util.*;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.plugins.trending.TrendingService;

/**
 * Skeleton implementation of a table that displays monitored values, and its model.
 *
 * @author onoprien
 */
abstract public class MonitorTable extends AbstractTableModel implements Updatable {

// -- Fields : -----------------------------------------------------------------
    
    /** Stateless default {@code MonitorFormat} instance. */
    static public final MonitorFormat DEFAULT_FORMAT = new MonitorFormat();

    /** Default list of fields to be displayed by monitoring tables in full mode. */
    static public final List<MonitorField> DEFAULT_FIELDS = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE, MonitorField.UNITS, MonitorField.LOW_ALARM, MonitorField.ALERT_LOW, MonitorField.HIGH_ALARM, MonitorField.ALERT_HIGH, MonitorField.DESCR));
    /** Default list of fields to be displayed by monitoring tables in compact mode. */
    static public final List<MonitorField> DEFAULT_COMPACT_FIELDS = Collections.unmodifiableList(Arrays.asList(MonitorField.VALUE));

    protected int nRows, nColumns;
    protected ArrayList<ArrayList<MonitorCell>> cells; // [row][column], covers actual table
    protected MonitorFormat format = DEFAULT_FORMAT;
    
    protected Listener listener;
    
    protected final Event CELLS_EVENT = new Event(this, Event.Reason.CELLS);
    protected final Event TABLE_EVENT = new Event(this, Event.Reason.TABLE);

// -- Life cycle : -------------------------------------------------------------
    
    /** Empty default constructor. */
    protected MonitorTable() {
    }
    
    /** Constructs the table with the specified content. */
    protected MonitorTable(MonitorCell[][] cells) {
        nRows = cells.length;
        nColumns = cells[0].length;
        this.cells = new ArrayList<>(nRows);
        for (MonitorCell[] row : cells) {
            this.cells.add(new ArrayList<>(Arrays.asList(row)));
        }
    }
    
    /** Called when the table is discarded. */
    public void destroy() {
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    /**
     * Sets the formatter to be used by this table.
     * @param format Formatter.
     */
    public void setFormat(MonitorFormat format) {
        this.format = format;
        if (cells != null) {
            boolean change = false;
            for (ArrayList<MonitorCell> row : cells) {
                for (MonitorCell cell : row) {
                    if (cell != null && cell.getField() != null && cell.getField().isUpdatable() && !cell.getField().equals(MonitorField.NULL)) {
                        change = format.format(cell);
                    }
                }
            }
            if (change) fireChangeEvent(CELLS_EVENT);
        }
    }
    
// -- Implementing TableModel : ------------------------------------------------

    @Override
    public int getRowCount() {
        return nRows;
    }

    @Override
    public int getColumnCount() {
        return nColumns;
    }

    @Override
    public FormattedValue getValueAt(int row, int column) {
        return cells.get(row).get(column).getFormattedValue();
    }

    @Override
    public boolean isCellEditable(int row, int column) {
        MonitorCell cell = cells.get(row).get(column);
        FormattedValue fv = cell.getFormattedValue();
        return (fv != null && fv.isEditable());
    }

    @Override
    public void setValueAt(Object aValue, int row, int column) {
        MonitorCell cell = cells.get(row).get(column);
        if (cell.getChannels().size() != 1) return; // no single associated channel
        MonitorField field = cell.getField();
        if (field == null) return; // no associated monitor field
        AgentChannel channel = cell.getChannels().get(0).getChannel();
        field.setValue(aValue, channel);
//        try {
//            ConfigurationParameterInfo conf = (ConfigurationParameterInfo) field.getValue(channel);
//            Console.getConsole().sendCommand(channel.getAgentName() +"/change", conf.getComponentName(), conf.getParameterName(), aValue);
//        } catch (ClassCastException | NullPointerException x) {
//        }
    }
    
    
// -- Updating table model : ---------------------------------------------------

    @Override
    public void update(ChannelHandle item, List<MonitorField> fields) {
        fields.forEach(field -> {
            List<int[]> affectedCells = getCells(item, field);
            for (int[] index : affectedCells) {
                MonitorCell cell = cells.get(index[0]).get(index[1]);
                field = cell.getField();
                FormattedValue data = format.format(field, item);
                if (!data.equals(cell.getFormattedValue())) {
                    cell.setFormattedValue(data);
                    fireTableCellUpdated(index[0], index[1]);
                }
            }
        });
        fireChangeEvent(CELLS_EVENT);
    }

    @Override
    public void update(ChannelHandle channelHandle) {
        update(channelHandle, Collections.singletonList(null));
    }

    /**
     * Returns indices of all cells whose content is affected by the specified item and field.
     * The implementation provided by this class is universal but very inefficient - subclasses
     * handling specific types of tables are expected to override it.
     * 
     * @param item ChannelHandle.
     * @param field Field. If {@code null}, all cells affected by the specified item are included.
     * @return List of cells.
     */
    protected List<int[]> getCells(ChannelHandle item, MonitorField field) {
        List<int[]> out = new ArrayList<>();
        for (int row = 0; row < nRows; row++) {
            for (int col = 0; col < nColumns; col++) {
                MonitorCell cell = cells.get(row).get(col);
                if (cell.getChannels().contains(item) && (field == null || field.equals(cell.getField()))) {
                    out.add(new int[] {row, col});
                }
            }
        }
        return out;
    }
    
    
// -- Handling change listener : -----------------------------------------------
    
    public void setListener(Listener listener) {
        this.listener = listener;
    }
    
    protected void fireChangeEvent(Event.Reason reason) {
        fireChangeEvent(reason == Event.Reason.CELLS ? CELLS_EVENT : TABLE_EVENT);
    }
    
    protected void fireChangeEvent(Event event) {
        if (listener != null) listener.stateChanged(event);
    }
    
    static public class Event extends EventObject {
        
        public enum Reason {CELLS, TABLE} 
        
        private final Reason reason;
        
        public Event(MonitorTable source, Reason reason) {
            super(source);
            this.reason = reason;
        }

        @Override
        public MonitorTable getSource() {
            return (MonitorTable) super.getSource();
        }
        
        public Reason getReason() {
            return reason;
        }
        
    }
    
    public interface Listener {

        public void stateChanged(Event e);
        
    }
    
    
// -- Customized JTable : ------------------------------------------------------
    
    public JTable getTable() {
        
        JTable table = makeTable();
        
        JTableHeader header = table.getTableHeader();
        header.setReorderingAllowed(false);
        header.setResizingAllowed(false); // no resizing to avoid mismatch between renderer and editor tables
        TableCellRenderer headerRenderer = header.getDefaultRenderer();
        if (headerRenderer instanceof DefaultTableCellRenderer) {
            ((DefaultTableCellRenderer)headerRenderer).setHorizontalAlignment(SwingConstants.CENTER);
        }

        MonitorTableCellRenderer renderer = new MonitorTableCellRenderer();
        renderer.setEnsureWidth(true);
        table.setDefaultRenderer(Object.class, renderer);
        table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
        table.setRowSelectionAllowed(false);
        table.setColumnSelectionAllowed(false);
        table.setRowHeight(table.getRowHeight() + 2);
        table.setShowGrid(true);
        table.addMouseListener(new MouseAdapter() {
            @Override
            public void mouseClicked(MouseEvent evt) {
                tableMouseClicked(evt);
            }
        });

        for (int col = 0; col < nColumns; col++) {
            TableColumn column = table.getColumnModel().getColumn(col);
            Component comp = headerRenderer.getTableCellRendererComponent(null, column.getHeaderValue(), false, false, 0, 0);
            int maxWidth = comp.getPreferredSize().width;
            for (int row=0; row<nRows; row++) {
                comp = renderer.getTableCellRendererComponent(table, getValueAt(row, col), false, false, row, col);
                maxWidth = Math.max(maxWidth, comp.getPreferredSize().width);
            }
            column.setPreferredWidth(maxWidth);
        }
        return table;
    }
    
    protected JTable makeTable() {
        return new JTable(this);
    }
    
    protected void tableMouseClicked(MouseEvent evt) {
        int nClick = evt.getClickCount();
        JTable table = (JTable) evt.getSource();
        Point point = evt.getPoint();
        int row = table.rowAtPoint(point);
        int column = table.columnAtPoint(point);
        if (nClick == 1) {
            MonitorCell cc = cells.get(row).get(column);
            FormattedValue fv = cc.getFormattedValue();
            if (fv != null && fv.getClick1() != null) {
                fv.getClick1().accept(cc);
            }
        } else if (nClick == 2) {
            MonitorCell cc = cells.get(row).get(column);
            if (cc.getField().equals(MonitorField.VALUE)) {
                List<ChannelHandle> items = cc.getChannels();
                if (items.size() == 1) {
                    ChannelHandle item = items.get(0);
                    String trendingKey = item.getChannel().get(AgentChannel.Key.TRENDING);
                    if (trendingKey == null) trendingKey = item.getChannel().getLocalPath();
                    String[] path = {item.getChannel().getAgentName(), trendingKey};
                    TrendingService trending = (TrendingService) Console.getConsole().getConsoleLookup().lookup(TrendingService.class);
                    if (trending == null) {
                        return;
                    }
                    trending.show(path);
                }
            }
        }
    }
    
    public boolean showHeader() {
        return true;
    }
    
    
// -- Static utilities : -------------------------------------------------------
    
    static public List<MonitorField> trimAbsentFields(List<MonitorField> fields, Collection<ChannelHandle> channels, Collection<MonitorField> exclude) {
        LinkedHashSet<MonitorField> requestedFields = new LinkedHashSet<>(fields);
        if (exclude != null) {
            requestedFields.removeAll(exclude);
        }
        HashSet<MonitorField> presentFields = new HashSet<>();
        if (requestedFields.remove(MonitorField.VALUE)) {
            presentFields.add(MonitorField.VALUE);
        }
        for (ChannelHandle ch : channels) {
            AgentChannel channel = ch.getChannel();
            if (channel != null) {
                Iterator<MonitorField> it = requestedFields.iterator();
                while (it.hasNext()) {
                    MonitorField f = it.next();
                    if (f.getValue(channel) != null) {
                        presentFields.add(f);
                        it.remove();
                    }
                }
                if (requestedFields.isEmpty()) break;
            }
        }
        if (fields.size() == presentFields.size()) {
            return fields;
        } else {
            if (presentFields.contains(MonitorField.ALERT_LOW)) {
                presentFields.add(MonitorField.ALERT_HIGH);
            } else if (presentFields.contains(MonitorField.ALERT_HIGH)) {
                presentFields.add(MonitorField.ALERT_LOW);
            }
            ArrayList<MonitorField> out = new ArrayList<>(presentFields.size());
            for (MonitorField f : fields) {
                if (presentFields.contains(f)) {
                    out.add(f);
                }
            }
            return out;
        }
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    /**
     * Returns JavaBean that contains data required to restore this table to its current state.
     * The default implementation provided by this class returns {@code null}, subclasses should override
     * if they need to support saving/restoring their state.
     * 
     * @return JavaBean that contains data required to restore this table to its current state.
     */
    public Serializable save() {
        return null;
    }
    
    /**
     * Restores this table to the state described by the provided object, to the extent possible.
     * The default implementation provided by this class does nothing.
     * 
     * @param storageBean JavaBean that contains data describing the state of this table.
     */
    public void restore(Serializable storageBean) {
    }

}
