 package org.lsst.ccs.gconsole.plugins.monitor;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComponent;
import org.freehep.application.Application;
import org.freehep.application.mdi.PageContext;
import org.freehep.application.mdi.PageEvent;
import org.freehep.application.mdi.PageListener;
import org.freehep.application.mdi.PageManager;
import org.freehep.application.studio.Studio;
import org.freehep.graphicsbase.util.export.ExportDialog;
import org.freehep.util.commanddispatcher.CommandProcessor;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.util.ThreadUtil;

/**
 * Graphical console plugin that supports display of monitoring data.
 * Unless otherwise specified, methods of this class can be called on any thread.
 *
 * @author onoprien
 */
@Plugin(name="LSST Monitoring Plugin",
        id="monitor",
        description="LSST CCS monitoring data display service.")
public class LsstMonitorPlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    static public final String DISPLAY_AS_TREE = "displayAsTree";
    static public final String DISPLAY_DOT_SEPARATOR = "displayDotSeparator";
    
    private final Studio app = (Studio) Application.getApplication();
    private final Action newView;
    private final ArrayList<MonitorView> currentViews = new ArrayList<>(0);
    private final ArrayDeque<MonitorView.Descriptor> recentViews = new ArrayDeque<>(0);

// -- Life cycle : -------------------------------------------------------------
    
    public LsstMonitorPlugin() {
        newView = new AbstractAction("Configure...") {
            @Override
            public void actionPerformed(ActionEvent e) {
                openNewView(null);
            }
        };
    }

    @Override
    public void initialize() {
        
        // Preference to control appearence of legacy monitoring tables:
        
        getServices().addProperty(DISPLAY_AS_TREE, false);
        getServices().addPreference(new String[] {"LSST","Monitor"}, "Data Display", "${"+ DISPLAY_AS_TREE +"}Display tables as trees.");

        // Monitoring menu:
        
        ThreadUtil.invokeLater(this::resetMenu);
        
        // Enabling dumping monitoring data into files:
        
        PageManager pm = app.getPageManager();
        pm.addPageListener(new Saver());
        
    }

    @Override
    public void shutdown() {
    }


// -- Opening monitoring pages : -----------------------------------------------
    
    /**
     * Opens a new graphical console page that displays the specified view.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to display.
     */
    public void openPage(MonitorView view) {
        PanelManager pm = getConsole().getPanelManager();
        JComponent panel = view.getPanel();
        pm.open(panel, view.getName());
        pm.addListener(e -> {
            if (e.hasKey(Panel.OPEN) && (Boolean) e.getNewValue() == false) {
                view.uninstall();
            }
        }, panel);
        view.install();
    }
    
    /**
     * Closes the console page that displays the specified view.
     * This method should be called on the AWT Event Dispatch Thread.
     * 
     * @param view View to close.
     */
    public void closePage(MonitorView view) {
        PanelManager pm = getConsole().getPanelManager();
        JComponent panel = view.getPanel();
        if (panel != null) pm.close(panel);
        view.uninstall();
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void openNewView(MonitorView.Descriptor desc) {
        MonitorView view;
        if (desc == null) {
            MonitorPageDialog d = MonitorPageDialog.show(LsstMonitorPlugin.this.getConsole().getWindow());
            view = d.getView();
        } else {
            view = MonitorPageDialog.getView(desc);
        }
        if (view != null) {
            JComponent panel = view.getPanel();
            if (panel != null) {
                TreeMap<Object, Object> par = new TreeMap<>();
                par.put(Panel.TITLE, view.getName());
                Consumer<JComponent> onClose = c -> onClose(c);
                par.put(Panel.ON_CLOSE, onClose);
                Console.getConsole().getPanelManager().open(panel, par);
                view.install();
                currentViews.add(view);
            }
        }
    }
    
    private void onClose(Component component) {
        
        if (!(component instanceof JComponent)) return;
        JComponent panel = (JComponent) component;
        
        MonitorView view = null;
        Iterator<MonitorView> it = currentViews.iterator();
        while (it.hasNext()) {
            MonitorView v = it.next();
            if (panel == v.getPanel()) {
                view = v;
                it.remove();
                break;
            }
        }
        if (view == null) return;
        view.uninstall();
        
        MonitorView.Descriptor mvd = view.save();
        if (mvd != null) {
            Serializable desc = mvd.getCreator();
            String name = mvd.getName();
            if (desc != null && name != null) {
                Iterator<MonitorView.Descriptor> itt = recentViews.iterator();
                while (itt.hasNext()) {
                    if (itt.next().getName().equals(name)) {
                        itt.remove();
                        break;
                    }
                }
                if (recentViews.size() > 9) {
                    recentViews.pollLast();
                }
                recentViews.addFirst(mvd);
                resetMenu();
            }        
        }
    }
    
    private void resetMenu() {
        Console console = getConsole();
        console.removeMenu(" CCS Tools ", "Monitoring...");
        getServices().addMenu(newView, "400: CCS Tools :-1:15", "Monitoring...:1");
        int i = 0;
        for (MonitorView.Descriptor desc : recentViews) {
            Action act = new AbstractAction(desc.getName()) {
                @Override
                public void actionPerformed(ActionEvent e) {
                    openNewView(desc);
                }
            };
            getServices().addMenu(act, "400: CCS Tools :-1:15", "Monitoring...:1:" + i++);
        }
    }
    
    
// -- Saving and restoring sessions : ------------------------------------------

    @Override
    public ComponentDescriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        
        ArrayList<MonitorView.Descriptor> current = new ArrayList<>(currentViews.size());
        for (MonitorView view : currentViews) {
            MonitorView.Descriptor d = view.save();
            if (d != null && d.getCreator() != null) {
                current.add(d);
            }
        }
        if (!current.isEmpty()) {
            desc.setCurrentViews(current.toArray(new MonitorView.Descriptor[current.size()]));
        }
        
        if (!recentViews.isEmpty()) {
            desc.setRecentViews(recentViews.toArray(new MonitorView.Descriptor[recentViews.size()]));
        }
        
        return desc;
    }

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        for (MonitorView view : new ArrayList<>(currentViews)) {
            closePage(view);
        }
        currentViews.clear();
        recentViews.clear();
        if (storageBean instanceof Descriptor) {
            Descriptor desc = (Descriptor) storageBean;
            if (desc.getCurrentViews() != null) {
                for (MonitorView.Descriptor d : desc.getCurrentViews()) {
                    openNewView(d);
                }
            }
            if (desc.getRecentViews() != null) {
                recentViews.addAll(Arrays.asList(desc.getRecentViews()));
            }
        }
        resetMenu();
        return true;
    }

    static public class Descriptor extends ComponentDescriptor {

        private MonitorView.Descriptor[] currentViews;
        private MonitorView.Descriptor[] recentViews;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public MonitorView.Descriptor[] getCurrentViews() {
            return currentViews;
        }

        public void setCurrentViews(MonitorView.Descriptor[] currentViews) {
            this.currentViews = currentViews;
        }

        public MonitorView.Descriptor getCurrentViews(int index) {
            return this.currentViews[index];
        }

        public void setCurrentViews(int index, MonitorView.Descriptor currentViews) {
            this.currentViews[index] = currentViews;
        }

        public MonitorView.Descriptor[] getRecentViews() {
            return recentViews;
        }

        public void setRecentViews(MonitorView.Descriptor[] recentViews) {
            this.recentViews = recentViews;
        }

        public MonitorView.Descriptor getRecentViews(int index) {
            return this.recentViews[index];
        }

        public void setRecentViews(int index, MonitorView.Descriptor recentViews) {
            this.recentViews[index] = recentViews;
        }
        
    }
    
    
// -- Temporary code for saving monitoring pages : -----------------------------
    
    private class Saver extends CommandProcessor implements PageListener {
        
        private boolean on;

        @Override
        public void pageChanged(PageEvent pe) {
            PageContext context = pe.getPageContext();
            if (context == null) return;
            JComponent panel = (JComponent) context.getPage();
            if (panel == null) return;
            switch (pe.getID()) {
                case PageEvent.PAGESELECTED:
                    MonitorDisplay md = findMonitorDisplay(panel, false);
                    if (md == null) {
                        if (on) {
                            app.getCommandTargetManager().remove(this);
                            on = false;
                        }
                    } else {
                        if (!on) {
                            app.getCommandTargetManager().add(this);
                            on = true;
                        }
                    }
                    break;
                case PageEvent.PAGEDESELECTED:
//                    if (on) {
//                        app.getCommandTargetManager().remove(this);
//                        on = false;
//                    }
                    break;
            }
        }
        
        private MonitorDisplay findMonitorDisplay(JComponent component, boolean visibleOnly) {
            if (visibleOnly && !component.isShowing()) {
                return null;
            }
            if (component instanceof MonitorDisplay) {
                return (MonitorDisplay) component;
            }
            Component[] cc = component.getComponents();
            for (Component c : cc) {
                if (c instanceof JComponent) {
                    MonitorDisplay md = findMonitorDisplay((JComponent)c, visibleOnly);
                    if (md != null) return md;
                }
            }
            return null;
        } 
        
        public void onSaveAs() {
            try {
                PageManager pm = app.getPageManager();
                PageContext context = pm.getSelectedPage();
                JComponent component = (JComponent) context.getPage();
                MonitorDisplay md = findMonitorDisplay(component, true);

                Properties user = app.getUserProperties();
                String creator = "creator";
                ExportDialog dlg = new ExportDialog(creator, false);
                dlg.addExportFileType(new MonitorExport());
                dlg.addExportFileType(new MonitorExport()); // working around the bug in ExportDialog line 160
                dlg.setUserProperties(user);
                dlg.showExportDialog(app, "Save As...", (Component)md, "monitor");
            } catch (NullPointerException|ClassCastException x) {
                app.error("No selected monitoring table", x);
            }
        }
        
    }
    
    static public void saveData(OutputStream out, String mimeType, List<AgentChannel> channels, List<MonitorField> fields) {
        
        PrintStream ps = new PrintStream(out);
        
        StringBuilder sb = new StringBuilder();
        for (MonitorField field : fields) {
            sb.append(field.getTitle()).append(",");
        }      
        ps.println(sb.substring(0, sb.length()-1));
        
        for (AgentChannel channel : channels) {
            sb = new StringBuilder();
            for (MonitorField field : fields) {
                sb.append(channel.<String>get(field.getKey())).append(",");
            }
            ps.println(sb.substring(0, sb.length()-1));
        }
        
    }

}
