package org.lsst.ccs.gconsole.plugins.demo.persist;

import org.lsst.ccs.gconsole.agent.filter.*;
import java.util.*;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.annotations.services.persist.Par;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;

/**
 * 
 *
 * @author onoprien
 */
public class TestFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    private final String name;
    private final List<String> agents;
    private final List<String> channels;
    private final String delimeter;
    private final boolean agentTabs;
    private final List<String> fields;
    
    private HashMap<String,String> display2origin;

// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructs a filter.
     * 
     * @param name Name of the filter.
     * @param agents Comma-separated list of subsystems accepted by this filter.
     * @param channels Comma-separated list of channels accepted by this filter (may contain templates and selectors).
     * @param delimeter Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter).
     * @param separatePagesForAgents If {@code true}, the subsystem name is used as a page name.
     * @param fields Comma-separated list of fields to display.
     */
    @Create(category = "org.lsst.ccs.gconsole.agent.AgentChannelsFilter",
            name = "Test Filter",
            path = "Built-In/Test/Generic",
            description = "Channels filter that allows the user to specify lists of accepted agents and channels.")
    public TestFilter(
            @Par(desc = "Filter name. Optional.") String name,
            @Par(desc = "Comma-separated list of subsystems accepted by this filter") String agents,
            @Par(desc = "Comma-separated list of channels accepted by this filter (may contain templates and selectors)") String channels,
            @Par(desc = "Character in the original channel path that should be replaced by a slash (and therefore used as a path delimeter). This is a very long parameter description indeed, without any html markup") String delimeter,
            @Par(desc = "If true, the subsystem name is used as a page name.") boolean separatePagesForAgents,
            @Par(desc = "Comma-separated list of fields to display in monitor views.") String fields
    ) {
        this.name = name;
        
        if (agents.trim().isEmpty()) {
            this.agents = null;
        } else if (agents.trim().equals("\"\"")) {
            this.agents = Collections.emptyList();
        } else {
            String[] ss = agents.split(",");
            this.agents = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.agents.add(s.trim());
            }
        }
        
        if (channels.trim().isEmpty()) {
            this.channels = null;
        } else if (channels.trim().equals("\"\"")) {
            this.channels = Collections.emptyList();
        } else {
            String[] ss = channels.split(",");
            this.channels = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.channels.add(s.trim());
            }
        }
        
        if (delimeter.isEmpty()) {
            this.delimeter = null;
        } else {
            String s = delimeter.trim();
            if (s.length() > 2 && s.startsWith("\"") && s.endsWith("\"")) {
                this.delimeter = s.substring(1, s.length()-1);
            } else {
                this.delimeter = Pattern.quote(delimeter);
            }
            display2origin = new HashMap<>();
        }
        
        this.agentTabs = separatePagesForAgents;
        
        if (fields == null || fields.trim().isEmpty()) {
            this.fields = null;
        } else if (fields.equals("\"\"")) {
            this.fields = Collections.singletonList("VALUE");
        } else {
            String[] ss = fields.split(",");
            this.fields = new ArrayList<>(ss.length);
            for (String s : ss) {
                this.fields.add(s.trim());
            }
        }
    }
    
    
// -- Implementing filter : ----------------------------------------------------

    @Override
    public List<String> getDisplayPath(String originPath) {
        String displayPath;
        if (delimeter == null) {
            if (agentTabs) {
                displayPath = originPath.replaceFirst("/", "//");
            } else {
                displayPath = originPath;
            }
        } else {
            int i = originPath.indexOf("/");
            String agent = originPath.substring(0, i);
            String localPath = originPath.substring(i+1);
            localPath = localPath.replaceAll(delimeter, "/");
            if (agentTabs) {
                displayPath = agent +"//"+ localPath;
            } else {
                displayPath = agent +"/"+ localPath;
            }
            display2origin.putIfAbsent(displayPath, originPath);
        }
        return Collections.singletonList(displayPath);
    }

    @Override
    public String getOriginPath(String displayPath) {
        if (delimeter == null) {
            if (agentTabs) {
                return displayPath.replaceFirst("//", "/");
            } else {
                return displayPath;
            }
        } else {
            return display2origin.get(displayPath);
        }
    }

    @Override
    public List<String> getOriginChannels() {
        return channels;
    }

    @Override
    public List<String> getAgents() {
        return agents;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public List<String> getFields(boolean compact) {
        return compact ? null : fields;
    }
    
// -- Specific filters : -------------------------------------------------------
    
    @Create(category = "org.lsst.ccs.gconsole.agent.AgentChannelsFilter",
            name = "Test Filter - States",
            path = "Built-In/Test/States",
            description = "Channels filter that display states.")
    static public TestFilter statesOnly(
            @Par(desc = "Comma-separated list of subsystems accepted by this filter. Optional")
            String agents) {
        return new TestFilter("States", agents, "/state/", null, false, null);
    }
    
}
